﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/util/util_BitUtil.h>
#include <nn/TargetConfigs/build_Base.h>

#include "hid_RingLifoHolder.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 他プロセスに許可されるメモリアクセス権
const ::nn::os::MemoryPermission OtherPermission =
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Visual Studio の std::atomic 型が load 時にロックしてしまうため
    ::nn::os::MemoryPermission_ReadWrite;
#else
    ::nn::os::MemoryPermission_ReadOnly;
#endif

} // namespace

::nn::Result InitializeSharedMemoryForRingLifo(
    ::nn::os::SharedMemoryType* sharedMemory, size_t lifoSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(sharedMemory);

    NN_RESULT_DO(
        ::nn::os::CreateSharedMemory(sharedMemory,
                                     ::nn::util::DivideUp(
                                         lifoSize,
                                         ::nn::os::MemoryPageSize
                                     ) * ::nn::os::MemoryPageSize,
                                     ::nn::os::MemoryPermission_ReadWrite,
                                     OtherPermission));

    ::nn::os::MapSharedMemory(sharedMemory,
                              ::nn::os::MemoryPermission_ReadWrite);

    NN_RESULT_SUCCESS;
}

void AttachSharedMemoryForRingLifo(::nn::os::SharedMemoryType* sharedMemory,
                                   size_t lifoSize,
                                   ::nn::os::NativeHandle handle,
                                   bool managed) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(sharedMemory);

    ::nn::os::AttachSharedMemory(sharedMemory,
                                 ::nn::util::DivideUp(
                                         lifoSize,
                                         ::nn::os::MemoryPageSize
                                     ) * ::nn::os::MemoryPageSize,
                                 handle,
                                 managed);

    ::nn::os::MapSharedMemory(sharedMemory, OtherPermission);
}

void FinalizeSharedMemoryForRingLifo(::nn::os::SharedMemoryType* sharedMemory
                                     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(sharedMemory);

    if (::nn::os::GetSharedMemoryAddress(sharedMemory) != nullptr)
    {
        ::nn::os::UnmapSharedMemory(sharedMemory);
    }

    ::nn::os::DestroySharedMemory(sharedMemory);
}

}}} // namespace nn::hid::detail
