﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include "hid_AtomicStorage.h"

namespace nn { namespace hid { namespace detail {

//!< リング LIFO を扱うためのクラスです。
template<
    typename LifoT,
    int LifoEntryCount,
    typename AtomicStorageT = AtomicStorage<LifoT>>
class RingLifo
{
    NN_STATIC_ASSERT(LifoEntryCount >= 1);
    NN_DISALLOW_COPY(RingLifo);
    NN_DISALLOW_MOVE(RingLifo);

protected:
    // Unused locally, but remains for compatibility against other classes that
    // may be still referencing it.
    int64_t m_Paddings[2];

    //!< リングバッファ上の末尾位置
    ::std::atomic<int64_t> m_Tail;

    //!< リングバッファ上の要素数
    ::std::atomic<int64_t> m_Count;

    //!< リングバッファ
    AtomicStorageT m_Buffer[LifoEntryCount + 1];

public:
    RingLifo() NN_NOEXCEPT
        : m_Paddings()
        , m_Tail(0)
        , m_Count(0)
    {
        m_Paddings[1] = NN_ARRAY_SIZE(m_Buffer);
    }

public:
    //!< リング LIFO 上の要素の型
    typedef LifoT ValueType;

public:
    //!< LIFO を空にします。
    void Clear() NN_NOEXCEPT
    {
        m_Count.store(0, ::std::memory_order_release);
    }

    //!< LIFO に要素を追加します。
    void Append(const LifoT& value) NN_NOEXCEPT
    {
        this->AppendImpl(value);
    }

    //!< LIFO が空か否か判定します。任意のタイミングでスレッドセーフです。
    bool IsEmpty() const NN_NOEXCEPT
    {
        return (m_Count.load(::std::memory_order_acquire) == 0);
    }

    //!< LIFO から要素を読み出します。任意のタイミングでスレッドセーフです。
    int Read(LifoT* buffer, int bufferCount) const NN_NOEXCEPT
    {
        return this->ReadImpl(buffer, bufferCount);
    }

protected:
    //!< 基準点からの相対位置を絶対位置に変換します。
    static int CalculateIndex(int index, int bias, int bufferCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, bufferCount);
        NN_SDK_REQUIRES_RANGE(bias, -bufferCount, bufferCount);
        return (index + bufferCount + bias) % bufferCount;
    }

    static int CalculateNextIndex(int index, int bufferCount) NN_NOEXCEPT
    {
        if(index >= (bufferCount - 1))
        {
            return 0;
        }

        return index + 1;
    }

    //!< Append() の内部実装
    template<typename T>
    void AppendImpl(const T& value) NN_NOEXCEPT
    {
        // 次の末尾位置を計算
        int nextIndex = RingLifo::CalculateNextIndex(
            static_cast<int>(m_Tail.load(::std::memory_order_relaxed)),
            NN_ARRAY_SIZE(m_Buffer));

        // 末尾位置に要素を書き込み
        m_Buffer[nextIndex].Set(value);

        // 末尾位置を更新
        m_Tail.store(nextIndex, ::std::memory_order_release);

        // リングバッファ上の要素数を更新
        if (m_Count.load(::std::memory_order_relaxed) < LifoEntryCount)
        {
            m_Count.fetch_add(1, ::std::memory_order_release);
        }
    }

    //!< Read() の内部実装
    template<typename T>
    int ReadImpl(T* buffer, int bufferCount) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        int countMax = 0;

        while (NN_STATIC_CONDITION(true))
        {
            // buffer の書き換え後に要素数が減らない様に単調増加
            countMax = ::std::min(
                bufferCount,
                ::std::max(
                    countMax,
                    static_cast<int>(
                        m_Count.load(::std::memory_order_acquire))));

            // 読み出し開始時点の末尾位置を待避
            const int tail = static_cast<int>(
                m_Tail.load(::std::memory_order_acquire));

            int bias = countMax - 1;

            while (bias >= 0)
            {
                // 古い値から順に要素の読み出し
                const int index = RingLifo::CalculateIndex(
                    tail, -bias, NN_ARRAY_SIZE(m_Buffer));

                if (!m_Buffer[index].Get(&buffer[bias]))
                {
                    // 読み出しに失敗した場合は読み出しを中断
                    break;
                }

                if (bias + 1 < countMax)
                {
                    // 2 回目以降の読み出し後に実施

                    if ((buffer[bias].samplingNumber -
                         buffer[bias + 1].samplingNumber) != 1)
                    {
                        // 前回のサンプル番号との間に連続性がなければ、
                        // 書き込みに追いつかれてしまっているので、
                        // 読み出しを中断
                        break;
                    }
                }

                --bias;
            }

            if (bias < 0)
            {
                break;
            }

            // 最後まで読み出せていなければ最初からやり直す
        }

        return countMax;
    }
};

template<typename LifoT>
class RingLifo <LifoT, 1>
{
    NN_DISALLOW_COPY(RingLifo);
    NN_DISALLOW_MOVE(RingLifo);

protected:
    // Unused - remains for compatibility.
    int64_t m_Paddings[2];

    // This could easily be a u32 or even a u8, but remains a u64
    // for compatibility.
    ::std::atomic<int64_t> m_Tail;

    // Other classes may use this directly, so I must support it as well.
    ::std::atomic<int64_t> m_Count;

    //!< リングバッファ上の要素数
    AtomicStorage<LifoT> m_Buffer[2];

public:
    RingLifo() NN_NOEXCEPT
        : m_Paddings()
        , m_Tail(0)
        , m_Count(0)
    {
        m_Paddings[1] = NN_ARRAY_SIZE(m_Buffer);
    }

public:
    //!< リング LIFO 上の要素の型
    typedef LifoT ValueType;

public:
    //!< LIFO を空にします。
    void Clear() NN_NOEXCEPT
    {
        m_Count.store(0, ::std::memory_order_release);
    }

    //!< LIFO に要素を追加します。
    void Append(const LifoT& value) NN_NOEXCEPT
    {
        this->AppendImpl(value);
    }

    //!< LIFO が空か否か判定します。任意のタイミングでスレッドセーフです。
    bool IsEmpty() const NN_NOEXCEPT
    {
        return (m_Count.load(::std::memory_order_acquire) == 0);
    }

    //!< LIFO から要素を読み出します。任意のタイミングでスレッドセーフです。
    int Read(LifoT* buffer, int bufferCount) const NN_NOEXCEPT
    {
        return this->ReadImpl(buffer, bufferCount);
    }

protected:
    //!< Append() の内部実装
    template<typename T>
    void AppendImpl(const T& value) NN_NOEXCEPT
    {
        // Read in value and place into range of 0,1
        m_Buffer[
            static_cast<int>(
                m_Tail.load(::std::memory_order_relaxed))].Set(value);

        // Swap buffer to signal change
        m_Tail.fetch_xor(1, ::std::memory_order_release);

        m_Count.store(1, ::std::memory_order_release);
    }

    //!< Read() の内部実装
    template<typename T>
    int ReadImpl(T* buffer, int bufferCount) const NN_NOEXCEPT
    {
        NN_UNUSED(bufferCount);

        do
        {
            NN_SDK_ASSERT_RANGE(
                m_Tail.load(::std::memory_order_acquire), 0, 2);

            if(this->IsEmpty())
            {
                return 0;
            }
        }
        while(!m_Buffer[m_Tail.load(::std::memory_order_acquire) ^ 1]
                  .Get(&buffer[0]));

        return 1;
    }
};

}}} // namespace nn::hid::detail
