﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "hid_ResourceManager-os.win.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "hid_ResourceManager-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif

#include "hid_VibrationDeviceHandle.h"

#ifdef VIBRATION_LOG_ENABLE

#include <nn/hid/detail/hid_Log.h>

#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_VIB_TRACE(...)            NN_DETAIL_HID_TRACE("ResourceManager: " ##__VA_ARGS__)
#else
#define NN_VIB_TRACE(format, ...)    NN_DETAIL_HID_TRACE("ResourceManager: " format, ##__VA_ARGS__)
#endif  // ifdef NN_BUILD_CONFIG_COMPILER_VC

#else

#define NN_VIB_TRACE(...)            static_cast<void>(0)

#endif

namespace {
    //一時的に振動をONにするARUIDを保存しておく InvalidIdが入っているときは機能が無効、そうでないときは機能が有効
    ::nn::applet::AppletResourceUserId g_AruidForPermitVibrationTemporarily = ::nn::applet::AppletResourceUserId::GetInvalidId();
}

namespace nn { namespace hid { namespace detail {

::nn::Result ResourceManager::AssertValidVibrationDeviceHandle(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    int playerNumber = GetVibrationDeviceHandlePlayerNumber(handle);
    int vibrationDeviceIdx = GetVibrationDeviceHandleDeviceIdx(handle);

    auto deviceType = GetVibrationDeviceHandleDeviceType(handle);
    if (deviceType == VibrationDeviceType_LinearResonantActuator ||
        deviceType == VibrationDeviceType_GcErm)
    {
        NN_RESULT_THROW_UNLESS(
            VerifyValidNpadId(playerNumber).IsSuccess(),
            ResultVibrationDeviceHandlePlayerNumberOutOfRange());
        NN_RESULT_THROW_UNLESS(
            0 <= vibrationDeviceIdx && vibrationDeviceIdx < NpadVibratorCountPerPlayerMax,
            ResultVibrationDeviceHandleDeviceIdxOutOfRange());
    }
    else
    {
        NN_RESULT_THROW(ResultVibrationDeviceHandleInvalidXpadType());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::GetVibrationDeviceInfo(VibrationDeviceInfo* pOutValue, const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->position = VibrationDevicePosition_None;
    pOutValue->deviceType = VibrationDeviceType_Unknown;

    NN_RESULT_DO(AssertValidVibrationDeviceHandle(handle));

    pOutValue->deviceType = GetVibrationDeviceHandleDeviceType(handle);
    pOutValue->position = GetVibrationDeviceHandleDevicePosition(handle);

    NN_RESULT_SUCCESS;
}

IVibratorDriver& ResourceManager::GetVibrator(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(AssertValidVibrationDeviceHandle(handle).IsSuccess());
    auto* pVibrator = this->GetNpadResourceManager().GetVibrator(handle);
    NN_SDK_REQUIRES_NOT_NULL(pVibrator);
    return *pVibrator;
}

VibratorXcdDriver* ResourceManager::GetVibratorXcd(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(AssertValidVibrationDeviceHandle(handle).IsSuccess());
    auto* pVibrator = this->GetNpadResourceManager().GetVibratorXcd(handle);
    return pVibrator;
}

VibratorGcDriver* ResourceManager::GetVibratorGc(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(AssertValidVibrationDeviceHandle(handle).IsSuccess());
    auto* pVibrator = this->GetNpadResourceManager().GetVibratorGc(handle);
    return pVibrator;
}

::nn::Result ResourceManager::SetAruidValidForVibration(
    ::nn::applet::AppletResourceUserId aruid, bool enable) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
    > locker(m_AppletResourceMutex);
    if (m_AppletResourceManager.SetAruidValidForVibration(aruid, enable) == true)
    {
        this->ResetAllVibrationDevices();
    }

    //BeginPermitVibrationSessionで指定されたARUID以外のIDが指定されたら一時的な振動ON機能を停止する
    if (g_AruidForPermitVibrationTemporarily != ::nn::applet::AppletResourceUserId::GetInvalidId()
        && g_AruidForPermitVibrationTemporarily != aruid)
    {
        EndPermitVibrationSession();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::IsAruidValidForVibration(::nn::applet::AppletResourceUserId aruid, bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_AppletResourceMutex)> locker(m_AppletResourceMutex);
    m_AppletResourceManager.IsAruidValidForVibration(aruid, pOutValue);

    NN_RESULT_SUCCESS;
}

//!< 一時的な振動強制ON期間を開始します。
::nn::Result ResourceManager::BeginPermitVibrationSession(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (aruid != ::nn::applet::AppletResourceUserId::GetInvalidId())
    {
        g_AruidForPermitVibrationTemporarily = aruid;
        NN_VIB_TRACE("START Permit Vibration\n");
        NN_RESULT_DO(::nn::xcd::SetVibrationMasterVolume(1.0f));
    }

    NN_RESULT_SUCCESS;
}

//!< 一時的な振動強制ON期間を停止します。
::nn::Result ResourceManager::EndPermitVibrationSession() NN_NOEXCEPT
{
    NN_VIB_TRACE("END Permit Vibration\n");

    //NANDの振動設定を読み込み、反映させる
    float volume;
    NN_RESULT_DO(::nn::xcd::LoadAndGetVibrationMasterVolume(&volume));
    NN_RESULT_DO(::nn::xcd::SetVibrationMasterVolume(volume));

    g_AruidForPermitVibrationTemporarily = ::nn::applet::AppletResourceUserId::GetInvalidId();

    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::ResetAllVibrationDevices() NN_NOEXCEPT
{
    //すべて振動子の振動を一旦止める
    // TODO: VibratorXcdDriver の最大数をどこかに定義したい
    IVibratorDriver* pDrivers[32];
    int outCount;
    m_NpadResourceManager.GetVibrators(&outCount, pDrivers);

    for (int i = 0; i < outCount; i++)
    {
        pDrivers[i]->ForceStopVibration();
    }
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
