﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/xcd/xcd.h>

#include "hid_BluetoothMode.h"
#include "hid_RailManager.h"
#include "hid_XcdUtility.h"

#include <nn/hid/detail/hid_Log.h>


namespace nn { namespace hid { namespace detail {


RailManager::RailManager() NN_NOEXCEPT
    : m_pSender(nullptr)
    , m_pInterruptSceneNotifier(nullptr)
{
    // 何もしない
}

RailManager::~RailManager() NN_NOEXCEPT
{
    // 何もしない
}

void RailManager::SetOvlnSenderManager(OvlnSenderManager* pSender) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSender);

    m_pSender = pSender;
}

void RailManager::SetInterruptSceneNotifier(InterruptSceneNotifier* pNotifier) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNotifier);

    m_pInterruptSceneNotifier = pNotifier;
}

void RailManager::BindRailUpdateEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    ::nn::xcd::SetRailUpdateEvent(pSystemEvent);
}

void RailManager::UnbindRailUpdateEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    NN_UNUSED(pSystemEvent);

    // 何もしない
}

void RailManager::HandleRailUpdateEvent() NN_NOEXCEPT
{
    ::nn::xcd::RailUpdateEventInfo eventInfo;

    while(NN_STATIC_CONDITION(true))
    {
        eventInfo = ::nn::xcd::GetRailUpdateEventInfo();

        if (eventInfo.eventType != ::nn::xcd::RailUpdateEventType_None)
        {
            auto color = NpadControllerColor();
            bool isColorValid = eventInfo.colorAvailable;
            if (isColorValid)
            {
                color.main = eventInfo.mainColor;
                color.sub = eventInfo.subColor;
            }
            switch (eventInfo.eventType)
            {
            case ::nn::xcd::RailUpdateEventType_LeftAttached:
                NotifyAttachment(::nn::hid::system::NpadDeviceType::JoyConLeft::Mask, isColorValid, color, true);
                break;
            case ::nn::xcd::RailUpdateEventType_LeftDetached:
                NotifyAttachment(::nn::hid::system::NpadDeviceType::JoyConLeft::Mask, isColorValid, color, false);
                break;
            case ::nn::xcd::RailUpdateEventType_RightAttached:
                NotifyAttachment(::nn::hid::system::NpadDeviceType::JoyConRight::Mask, isColorValid, color, true);
                break;
            case ::nn::xcd::RailUpdateEventType_RightDetached:
                NotifyAttachment(::nn::hid::system::NpadDeviceType::JoyConRight::Mask, isColorValid, color, false);
                break;
            default:
                // 通知するものがない
                return;
            }
        }
        else
        {
            // 通知するものがない
            return;
        }

    }
}

void RailManager::NotifyAttachment(system::NpadDeviceTypeSet device, const bool isColorValid, const NpadControllerColor& color, bool isAttached) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSender);

    // オーバーレイ通知
    nn::ovln::format::ControllerAttachToConsoleData data;
    data.eventType = (isAttached) ?
        ::nn::ovln::format::ControllerAttachEventType_ConsoleRailAttached :
        ::nn::ovln::format::ControllerAttachEventType_ConsoleRailDetached;
    data.deviceType = device;
    // コントローラーの色情報が無効なデバイスは現状存在しない
    data.color.isValid = isColorValid;
    data.color.color = color;
    m_pSender->Send(data);

    // 無線オフ中のレール抜きの割り込みシーン通知
    if (isAttached == false &&
        IsBluetoothEnabled() == false)
    {
        m_pInterruptSceneNotifier->NotifyJoyDetachOnBluetoothOffEvent();
    }
}

}}} // namespace nn::hid::detail
