﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_PalmaTypes.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/xcd/xcd_BleDevice.h>
#include "hid_NpadId.h"

namespace nn { namespace hid { namespace detail {

class PalmaManager final
{
    NN_DISALLOW_COPY(PalmaManager);
    NN_DISALLOW_MOVE(PalmaManager);

private:
    //!< Palma の操作が完了したことを通知するイベントオブジェクト
    ::nn::os::SystemEvent m_OperationCompletedEvent;

    //!< Npad Id
    ::nn::hid::NpadIdType m_NpadId;

    //!< xcd のハンドル
    ::nn::xcd::BleConnectionHandle m_Handle;

    //!< データ転送のためのトランスファーメモリ
    ::nn::os::TransferMemoryType m_TransferMemory;
    void*                        m_pMappedAddress;
    uint64_t                     m_TransferMemorySize;

    //!< トランスファーメモリが割り当たっている ARUID
    ::nn::applet::AppletResourceUserId m_TransferMemoryHolderAruid;

    //!< Palma デバイスに対しての各操作に対する実行結果の Result
    ::nn::Result m_OperationResult;

public:
    PalmaManager() NN_NOEXCEPT;

    ~PalmaManager() NN_NOEXCEPT;

    //!< Palma の固有機能を初期化します。
    ::nn::Result Initialize() NN_NOEXCEPT;

    //!< イベントをバインドします。
    ::nn::Result AcquireOperationCompleteEventHandle(::nn::os::NativeHandle* outHandle) NN_NOEXCEPT;

    //!< Npad Id を設定します。
    void SetNpadId(const ::nn::hid::NpadIdType& id) NN_NOEXCEPT
    {
        m_NpadId = id;
    };

    //!< Npad Id を取得します。
    ::nn::Result GetNpadId(nn::hid::NpadIdType* pOutNpadId) NN_NOEXCEPT;

    //!< xcd のハンドルを設定します。
    ::nn::Result SetXcdHandle(::nn::xcd::BleConnectionHandle handle) NN_NOEXCEPT;

    //!< xcd のハンドルを無効化します。
    void InvalidateXcdHandle() NN_NOEXCEPT;

    ::nn::Result GetXcdHandle(::nn::xcd::BleConnectionHandle* pOutValue) NN_NOEXCEPT;

    //!< ハンドルが有効かどうかを返します。
    bool IsValidHandle() NN_NOEXCEPT
    {
        return m_Handle != ::nn::xcd::BleInvalidConnectionHandle;
    };

    //!< TransferMemoryType を取得します。
    ::nn::Result GetTransferMemoryType(::nn::os::TransferMemoryType** pTransferMemory) NN_NOEXCEPT;

    //!< TransferMemory を設定します。
    void SetTransferMemory(void* pMappedAddress, uint64_t size, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    bool IsTransferMemoryMapped() NN_NOEXCEPT
    {
        return m_pMappedAddress != nullptr;
    };

    //!< 指定した Aruid の TransferMemory の設定を解除します。
    void ResetTransferMemory(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Palma デバイスに対しての各操作に対する実行結果を取得します。
    ::nn::Result GetPalmaOperationInfo(PalmaOperationType* pOutValue, void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT;

    //!< Palma デバイスに対しての各操作に対する実行結果の Result を返します。
    ::nn::Result GetPalmaOperationResult() NN_NOEXCEPT;

    //!< Palma デバイス演出の再生を指示します。
    ::nn::Result PlayPalmaActivity(uint16_t number) NN_NOEXCEPT;

    //!< Palma デバイスの FrModeType (Friend Buddy) を設定します。
    ::nn::Result SetFrModeType(uint64_t type) NN_NOEXCEPT;

    //!< Palma デバイスの StepCount を読み出します。
    ::nn::Result ReadStep() NN_NOEXCEPT;

    //!< Palma デバイスの StepCount 機能の有効/無効を設定します。
    ::nn::Result EnableStepCounter(bool isEnabled) NN_NOEXCEPT;

    //!< Palma デバイスの StepCount をゼロクリアします。
    ::nn::Result ResetStep() NN_NOEXCEPT;

    //!< Palma デバイスの ApplicationSection の指定の領域からデータを読み出します。
    ::nn::Result ReadApplicationSection(uint64_t address, uint64_t size) NN_NOEXCEPT;

    //!< Palma デバイスの ApplicationSection の指定の領域へデータを書き込みます。
    ::nn::Result WriteApplicationSection(uint64_t address, uint64_t size, const PalmaApplicationSectionAccessBuffer& buffer) NN_NOEXCEPT;

    //!< Palma ユニークコードを読み出します。
    ::nn::Result ReadPalmaUniqueCode() NN_NOEXCEPT;

    //!< Palma ユニークコードを無効化します。以降、無効化した Palma デバイスからは読み出しできなくなります。
    ::nn::Result SetPalmaUniqueCodeInvalid() NN_NOEXCEPT;

    //!< Activity Entry を Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaActivityEntry(uint64_t index, uint64_t ledIndex, uint64_t waveSet, uint64_t waveIndex) NN_NOEXCEPT;

    //!< RGB LED パターンを Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaRgbLedPatternEntry(uint64_t index, const char* buffer, size_t size) NN_NOEXCEPT;

    //!< 音源データを Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaWaveEntry(uint64_t waveSet, uint64_t index, size_t size) NN_NOEXCEPT;

    //!< Palma の ActivityDataBase のデータベース識別バージョンを設定します。
    ::nn::Result SetPalmaDataBaseIdentificationVersion(int32_t version) NN_NOEXCEPT;

    //!< Palma の ActivityDataBase のデータベース識別バージョンを取得します。
    ::nn::Result GetPalmaDataBaseIdentificationVersion() NN_NOEXCEPT;

    //!< Palma の Feature Set の機能のサスペンド状態を制御します。
    ::nn::Result SuspendPalmaFeature(PalmaFeatureSet suspendFeatureSet) NN_NOEXCEPT;

    //!< Palma の Play Log を読み出します。
    ::nn::Result ReadPalmaPlayLog(uint16_t index) NN_NOEXCEPT;

    //!< Palma の Play Log をゼロクリアします。
    ::nn::Result ResetPalmaPlayLog(uint16_t index) NN_NOEXCEPT;

    //!< 指定されたデバイスとペアリングする
    ::nn::Result PairPalma() NN_NOEXCEPT;

private:
};

}}} // namespace nn::hid::detail
