﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/ovln/ovln_FormatForOverlay.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_ActivationCount.h"
#include "hid_OvlnSenderManager.h"

namespace nn { namespace hid { namespace detail {

namespace {

template <typename TData>
void SendImpl(TData data, uint32_t tag, nn::ovln::SenderForOverlayType* pSender) NN_NOEXCEPT
{
    nn::ovln::Message message = {};
    message.tag = tag;
    message.dataSize = sizeof(data);
    std::memcpy(&message.data, &data, sizeof(data));
    nn::ovln::Send(pSender, message);
}

// 最大同時に発行される可能性のある Queue の数 + 1
const int OvlnQueueLength = 9;

}

OvlnSenderManager::OvlnSenderManager() NN_NOEXCEPT :
    m_ActivationCount()
{
    // 何もしない
}

::nn::Result OvlnSenderManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // オーバレイ通知ライブラリの初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());

        // SenderForOverlayType の初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderWithQueue(&m_Sender, OvlnQueueLength));
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result OvlnSenderManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        nn::ovln::FinalizeSender(&m_Sender);
        nn::ovln::FinalizeSenderLibraryForOverlay();
    }

    NN_RESULT_SUCCESS;
}

void OvlnSenderManager::Send(nn::ovln::format::ControllerAttachToConsoleData data) NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        // アクティベートされていないときは何もしない
        return;
    }

    SendImpl(data, nn::ovln::format::ControllerAttachToConsoleDataTag, &m_Sender);
}

void OvlnSenderManager::Send(nn::ovln::format::ControllerBatteryLevelLowExData data) NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        // アクティベートされていないときは何もしない
        return;
    }

    SendImpl(data, nn::ovln::format::ControllerBatteryLevelLowExDataTag, &m_Sender);
}

void OvlnSenderManager::Send(nn::ovln::format::ControllerDisconnectionData data) NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        // アクティベートされていないときは何もしない
        return;
    }

    SendImpl(data, nn::ovln::format::ControllerDisconnectionDataTag, &m_Sender);
}

void OvlnSenderManager::Send(nn::ovln::format::ControllerPairingCompletedData data) NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        // アクティベートされていないときは何もしない
        return;
    }

    SendImpl(data, nn::ovln::format::ControllerPairingCompletedDataTag, &m_Sender);
}

}}} // namespace nn::hid::detail
