﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_AppletResourceUserId.h"
#include "hid_HidSystemServer.h"
#include "hid_StaticObject.h"
#include "hid_NpadSystemImpl.h"
#include "hid_RingLifoHolder.h"

namespace nn { namespace hid { namespace detail {

::nn::Result ApplyNpadSystemCommonPolicy() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ApplyNpadSystemCommonPolicy(GetAppletResourceUserId()));

    NN_RESULT_SUCCESS;
}

::nn::Result ApplyNpadSystemCommonPolicyFull() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ApplyNpadSystemCommonPolicyFull(GetAppletResourceUserId()));

    NN_RESULT_SUCCESS;
}

::nn::Result BindNfcActivateEvent(NpadIdType& id,
    ::nn::os::SystemEventType* pEvent,
    ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    auto handle = ::nn::sf::NativeHandle();
    NN_RESULT_DO(pProxy->AcquireNfcActivateEventHandle(&handle, id));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result GetXcdHandleForNpadWithNfc(nn::xcd::DeviceHandle* pOutHandle, NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetXcdHandleForNpadWithNfc(pOutHandle, id, ::nn::applet::AppletResourceUserId::GetInvalidId()));
    NN_RESULT_SUCCESS;
}

::nn::Result ActivateNfc(NpadIdType& id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ActivateNfc(id, true, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result DeactivateNfc(NpadIdType& id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ActivateNfc(id, false, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result IsNfcActivated(bool* pOutIsActivated, NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsActivated);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->IsNfcActivated(pOutIsActivated, id));
    NN_RESULT_SUCCESS;
}

::nn::Result BindIrSensorEvent(NpadIdType& id,
    ::nn::os::SystemEventType* pEvent,
    ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    auto handle = ::nn::sf::NativeHandle();
    NN_RESULT_DO(pProxy->AcquireIrSensorEventHandle(&handle, id));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result GetXcdHandleForNpadWithIrSensor(nn::xcd::DeviceHandle* pOutHandle, NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetXcdHandleForNpadWithIrSensor(id, pOutHandle, ::nn::applet::AppletResourceUserId::GetInvalidId()));

    NN_RESULT_SUCCESS;
}

::nn::Result GetIrSensorState(system::IrSensorState* pOutState, NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    int64_t state;
    NN_RESULT_DO(pProxy->GetIrSensorState(id, &state, ::nn::applet::AppletResourceUserId::GetInvalidId()));
    switch (state)
    {
    case system::IrSensorState_NotConnected:
        *pOutState = system::IrSensorState_NotConnected;
        break;

    case system::IrSensorState_NotSupported:
        *pOutState = system::IrSensorState_NotSupported;
        break;

    case system::IrSensorState_Deactivated:
        *pOutState = system::IrSensorState_Deactivated;
        break;

    case system::IrSensorState_Activated:
        *pOutState = system::IrSensorState_Activated;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}
::nn::Result ActivateIrSensor(NpadIdType& id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ActivateIrSensor(id, true, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result DeactivateIrSensor(NpadIdType& id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ActivateIrSensor(id, false, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result EnableAssigningSingleOnSlSrPress() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->EnableAssigningSingleOnSlSrPress(GetAppletResourceUserId()));
    NN_RESULT_SUCCESS;
}

::nn::Result DisableAssigningSingleOnSlSrPress() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->DisableAssigningSingleOnSlSrPress(GetAppletResourceUserId()));
    NN_RESULT_SUCCESS;
}

::nn::Result SetNpadPlayerLedBlinkingDevice(const NpadIdType& id, const system::NpadDeviceTypeSet& deviceType) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetNpadPlayerLedBlinkingDevice(GetAppletResourceUserId(), id, deviceType));
    NN_RESULT_SUCCESS;
}

::nn::Result GetLastActiveNpadController(NpadIdType* pOutValue) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetLastActiveNpad(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadSystemExtStyle(system::NpadSystemExtMainStyle* pOutMainStyle,
                                   system::NpadSystemExtSubStyle* pOutSubStyle,
                                   const NpadIdType& id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    int64_t mainStyle;
    int64_t subStyle;

    NN_RESULT_DO(pProxy->GetNpadSystemExtStyle(&mainStyle, &subStyle, id));
    *pOutMainStyle = static_cast<system::NpadSystemExtMainStyle>(mainStyle);
    *pOutSubStyle = static_cast<system::NpadSystemExtSubStyle>(subStyle);
    NN_RESULT_SUCCESS;
}

::nn::Result GetUniquePadsFromNpad(int* pOutCount, system::UniquePadId* pOutValues, int count, const NpadIdType& npadId) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    int64_t gotCount;

    NN_RESULT_DO(
        pProxy->GetUniquePadsFromNpad(::nn::sf::Out<int64_t>(&gotCount),
                                      ::nn::sf::OutArray<system::UniquePadId>(
                                      pOutValues, static_cast<size_t>(count)), npadId));

    *pOutCount = static_cast<int>(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadFullKeyGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLeftGrip);
    NN_SDK_REQUIRES_NOT_NULL(pOutRightGrip);


    ::nn::sf::SharedPointer<IHidSystemServer> pHidServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->GetNpadFullKeyGripColor(pOutLeftGrip, pOutRightGrip, id));
    NN_RESULT_SUCCESS;
}

::nn::Result IsHandheldButtonPressedOnConsoleMode(bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::nn::sf::SharedPointer<IHidSystemServer> pHidServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->IsHandheldButtonPressedOnConsoleMode(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSupportedNpadStyleSetOfCallerApplet(NpadStyleSet* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::nn::sf::SharedPointer<IHidSystemServer> pHidServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidServer));

    ::nn::applet::AppletResourceUserId aruid;

    NN_RESULT_TRY(GetAppletResourceUserIdOfCallerApplet(&aruid));
    NN_RESULT_CATCH(nn::applet::ResultCallerAppletIsWinding)
    {
        aruid = nn::applet::AppletResourceUserId::GetInvalidId();
    };
    NN_RESULT_CATCH_ALL
    {
        NN_RESULT_RETHROW;
    }
    NN_RESULT_END_TRY

    NN_RESULT_DO(pHidServer->GetMaskedSupportedNpadStyleSet(pOutValue, aruid));
    NN_RESULT_SUCCESS;
}



}}} // namespace nn::hid::detail
