﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/system/hid_Common.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Thread.h>

#include "hid_ActivationCount.h"
#include "hid_CaptureButtonManager.h"
#include "hid_ExternalBusHandle.h"
#include "hid_HomeButtonManager.h"
#include "hid_IAbstractedPad.h"
#include "hid_NpadAppletPolicyManager.h"
#include "hid_NpadAssignmentManager.h"
#include "hid_NpadCommonResourceHolder.h"
#include "hid_NpadId.h"
#include "hid_NpadLastActiveControllerMonitor.h"
#include "hid_NpadPalmaScanController.h"
#include "hid_NpadPeripheralManager.h"
#include "hid_NpadUniquePadConverter.h"
#include "hid_PalmaResourceManager.h"
#include "hid_UniquePadResourceManager.h"
#include "hid_VibrationDeviceHandle.h"
#include "hid_OvlnSenderManager.h"
#include "hid_PlayReportManager.h"

namespace nn { namespace hid { namespace detail {

//!< Npad に関するリソースを管理するクラスです。
class NpadResourceManager final
{
private:
    //!< Npad の Activation された回数
    ActivationCount m_ActivationCount;

    //!< Npad 操作排他用のミューテックス
    ::nn::os::SdkMutex m_NpadMutex;

    //!< Npad のポリシーマネージャー
    NpadAppletPolicyManager m_AppletPolicyManager;

    //!< 共通で利用する hid のサービスのホルダー
    NpadCommonResourceHolder m_CommonResourceHolder;

    //!< Npad 毎のリソースを管理するマネージャーの配列
    NpadManagerArray m_NpadManagers;

    //!< Npad の割り当てマネージャー
    NpadAssignmentManager m_AssignmentManager;

    //!< 最後に操作したコントローラーの監視サービス
    NpadLastActiveControllerMonitor m_LastActiveControllerMonitor;

    //!< ペリフェラルマネージャー
    NpadPeripheralManager m_PeripheralManager;

    //!< Npad と UniquePad のコンバーター
    NpadUniquePadConverter m_NpadUniquePadConverter;

    //!< Palma の Scan を制御するコントローラ
    NpadPalmaScanController m_PalmaScanController;

    //!< UniquePad のリソースマネージャー
    UniquePadResourceManager* m_pUniquePadResourceManager;

public:
    NpadResourceManager() NN_NOEXCEPT;

    ~NpadResourceManager() NN_NOEXCEPT;

    //!< Npad を初期化します
    void Initialize(CaptureButtonManager* pCaptureButtonManager,
                    HomeButtonManager* pHomeButtonManager,
                    OvlnSenderManager* pOvlnSender,
                    PlayReportManager* pPlayReportManager,
                    AppletResourceManager* pAppletResourceManager,
                    nn::os::SdkRecursiveMutex* pAppletResourceManagerMutex,
                    SixAxisSensorAppletSettingManager* pSixAxisSensorSettingsManager,
                    ConsoleSixAxisSensorManager* pConsoleSixAxisSensor,
                    HandheldManager* pHandheldManager,
                    InputDetectorManager* pInputDetector,
                    nn::os::SdkMutex* pInputDetectorMutex,
                    UniquePadResourceManager* pUniquePad,
                    InterruptSceneNotifier* pInterruptSceneNotifier,
                    GcAdapterDriver* pGcAdapterDriver,
                    PalmaResourceManager* pPalma
                    ) NN_NOEXCEPT;

    //!< IAbstractedPad をセットします
    void AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT;

    //!< Npad をアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< Npad を非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Npad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureNpadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 定期処理をハンドリングします。
    void HandlePeriodcialEvent() NN_NOEXCEPT;

    //!< 携帯機コントローラーの内部状態を更新します。
    void UpdateHandheld() NN_NOEXCEPT;

    //!< 有効にする NpadStyleSet を設定します。
    ::nn::Result SetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet style) NN_NOEXCEPT;

    //!< 有効な NpadStyleSet を取得します。
    ::nn::Result GetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet* pOutValue) NN_NOEXCEPT;

    //!< アプリのビルドされた SDK バージョンに即してマスクされた有効な NpadStyleSet を取得します。
    ::nn::Result GetMaskedSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet* pOutValue) NN_NOEXCEPT;

    //!< 有効にする NpadIdType を設定します。
    ::nn::Result SetSupportedNpadIdType(::nn::applet::AppletResourceUserId aruid, const NpadIdType* pIds, int count) NN_NOEXCEPT;

    //!< ジョイコンの持ち方を設定します。
    ::nn::Result SetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid, NpadJoyHoldType holdType) NN_NOEXCEPT;

    //!< ジョイコンの持ち方を取得します。
    ::nn::Result GetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid, NpadJoyHoldType* pOutValue) NN_NOEXCEPT;

    //!< NpadStyleSet が変化したときに受ける通知をイベントオブジェクトにバインドします。
    ::nn::Result AcquireStyleUpdateEventHandle(::nn::applet::AppletResourceUserId aruid, ::nn::os::NativeHandle* pOutValue, NpadIdType id) NN_NOEXCEPT;

    //!< Npad を切断します。
    void Disconnect(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< 本体機能共通の Npad の動作ポリシーを設定します。
    ::nn::Result ApplyNpadSystemCommonPolicy(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 最大数の接続を許可する本体機能共通の Npad の動作ポリシーを設定します。
    ::nn::Result ApplyNpadSystemCommonPolicyFull(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< SlSr ボタンが押されて接続された際に、1本持ちとして接続する機能の有効状態を設定します
    ::nn::Result SetAssigningSingleOnSlSrPressMode(bool enable, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 指定した Npad に対するジョイコンの割り当て状態を取得します。
    NpadJoyAssignmentMode GetNpadJoyAssignmentMode(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< Npad の割り当てオードを デフォルト設定に基づき1本ずつに変更します。
    void SetNpadJoyAssignmentModeSingleByDefault(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< Npad の割り当てオードを 1本ずつに変更します。
    void SetNpadJoyAssignmentModeSingle(::nn::applet::AppletResourceUserId aruid, NpadIdType id, NpadJoyDeviceType type) NN_NOEXCEPT;

    //!< Npad の割り当てオードを 1本ずつに変更します。
    bool SetNpadJoyAssignmentModeSingleWithDestination(NpadIdType* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id, NpadJoyDeviceType type) NN_NOEXCEPT;

    //!< Npad の割り当てモードを 2本セットに変更します。
    void SetNpadJoyAssignmentModeDual(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< 1本ずつ接続された2つのNpadをマージして 1つの Npad に対して 2本セット
    ::nn::Result MergeSingleJoyAsDualJoy(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT;

    //!< Npad に対するコントローラーの割り当てを入れ替えます
    ::nn::Result SwapNpadAssignment(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラー割り当て決定モードを有効にします。
    ::nn::Result EnableLrAssignmentMode(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラー割り当て決定モードを無効にします。
    ::nn::Result DisableLrAssignmentMode(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Handheld 操作形態を有効化するモードを設定します。
    ::nn::Result SetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid, NpadHandheldActivationMode mode) NN_NOEXCEPT;

    //!< Handheld 操作形態を有効化するモードを取得します。
    ::nn::Result GetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid, NpadHandheldActivationMode* pOutValue) NN_NOEXCEPT;

    //!< プレイヤーランプを点滅させるデバイスを設定します
    void SetNpadPlayerLedBlinkingDevice(::nn::applet::AppletResourceUserId aruid, NpadIdType id, system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;

    //!< Nfc デバイスの状態変化通知イベントをバインドします。
    ::nn::Result AcquireNfcDeviceUpdateEventHandle(::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT;

    //!< 有効な Nfc デバイスに対する xcd のハンドルを取得します。
    int GetNpadsWithNfc(NpadIdType* pOutIds, int length) NN_NOEXCEPT;

    //!< NpadId に対して割り当てられている Nfc のデバイスハンドルを取得します
    ::nn::Result GetXcdHandleForNpadWithNfc(::nn::xcd::DeviceHandle* pOutHandle, NpadIdType id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Nfc デバイスのアクティブ通知イベントをバインドします。
    ::nn::Result AcquireNfcActivateEventHandle(::nn::os::NativeHandle* pOutHandle, NpadIdType id) NN_NOEXCEPT;

    //!< Nfc デバイスのアクティブ状態を制御します。
    ::nn::Result ActivateNfc(NpadIdType id, bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Nfc がアクティブかどうかを返します。
    bool IsNfcActivated(NpadIdType id) NN_NOEXCEPT;

    //!< IrSensor デバイスの状態変化イベントをバインドします。
    ::nn::Result AcquireIrSensorEventHandle(::nn::os::NativeHandle* pOutHandle, NpadIdType id) NN_NOEXCEPT;

    //!< IrSensor デバイスのアクティブ状態を制御します。
    ::nn::Result ActivateIrSensor(NpadIdType id, bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 最後に操作されたコントローラーを取得します。
    ::nn::Result GetLastActiveNpad(NpadIdType* pOutValue) NN_NOEXCEPT;

    //!< 本体機能の操作形態の情報を取得します
    void GetNpadSystemExtStyle(system::NpadSystemExtMainStyle* pOutMainStyle,
                               system::NpadSystemExtSubStyle* pOutSubStyle,
                               NpadIdType id) NN_NOEXCEPT;

    //!< NpadId に対して割り当てられている IrSensor のデバイスハンドルを取得します
    ::nn::Result GetXcdHandleForNpadWithIrSensor(NpadIdType id, ::nn::xcd::DeviceHandle* pOutHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadId に対する IrSensor の状態を取得します
    ::nn::Result GetIrSensorState(NpadIdType id, system::IrSensorState* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< レール接続デバイスのアクティブ状態を制御します。
    Result ActivateRailAttachment(NpadIdType id, NpadJoyConRailIndex index, bool activate) NN_NOEXCEPT;

    //!< NpadId に対して割り当てられている レール接続デバイスに割り当てられているの xcd のデバイスハンドルを取得します
    ::nn::Result GetXcdHandleForNpadWithRailAttachment(::nn::xcd::DeviceHandle* pOutHandle, NpadIdType id, NpadJoyConRailIndex index) NN_NOEXCEPT;

    //!< レール接続デバイスの状態を取得します。
    NpadJoyConRailAttachmentState GetNpadJoyConRailAttachmentState(NpadIdType id, NpadJoyConRailIndex index) NN_NOEXCEPT;

    //!< レール接続デバイス向けに AbstratedPad の id を取得します。
    ::nn::Result GetAbstractedPadIdForRailAttachment(AbstractedPadId* pOutId, NpadIdType id, NpadJoyConRailIndex index) NN_NOEXCEPT;

    //!< ホームボタンが有効かどうかを取得します。
    ::nn::Result IsUnintendedHomeButtonInputProtectionEnabled(bool* pOutEnabled,
                                                              ::nn::applet::AppletResourceUserId aruid,
                                                              NpadIdType id) NN_NOEXCEPT;

    //!< ホームボタンが有効かどうかを取得します。
    ::nn::Result EnableUnintendedHomeButtonInputProtection(::nn::applet::AppletResourceUserId aruid,
                                                           NpadIdType id,
                                                           bool enabled) NN_NOEXCEPT;

    //!< Npad に割り当てられた UniquePad を取得します。
    int GetUniquePadsFromNpad(system::UniquePadId* pOutValues, int count, NpadIdType& id) NN_NOEXCEPT;

    //!< Npad 向けのアプレットリソースユーザ ID を登録します。
    ::nn::Result RegisterNpadAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Npad 向けのアプレットリソースユーザ ID の登録を解除します。
    void UnregisterNpadAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Npad 向けのアプレットリソースのエントリを有効にします。
    ::nn::Result ActivateNpadAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Npad 向けのアプレットリソースのエントリを無効します。
    void DeactivateNpadAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Npad 向けの ARUID を設定します。
    void SetNpadAppletResourceUserId(::nn::applet::AppletResourceUserId value
                                 ) NN_NOEXCEPT;

    //!< Npad 向けの共有アプレットリソースをアクティブ化します。
    ::nn::Result ActivateNpadSharedAppletResource() NN_NOEXCEPT;

    //!< Npad 向けの共有アプレットリソースを非アクティブ化します。
    ::nn::Result DeactivateNpadSharedAppletResource() NN_NOEXCEPT;

    //!< Npad の Resume 処理を行います
    void ResumeNpad() NN_NOEXCEPT;

    //!< 有線 USB 通信中の Pro Controller かどうかを取得します。
    ::nn::Result IsUsbFullKeyControllerConnected(bool* pOutConnected, NpadIdType id) NN_NOEXCEPT;

    //!< ファームウェアバージョンを取得します
    Result GetFirmwareVersion(
        debug::FirmwareVersion *pOutValue,
        bool* pOutIsMcuIapCorrupted,
        NpadIdType id,
        system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;

    //!< 更新先のファームウェアバージョンを取得します
    Result GetDestinationFirmwareVersion(
        debug::FirmwareVersion *pOutValue,
        NpadIdType id,
        system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;

    //!< SixAxisSensorXcdDriver を返します。
    const SixAxisSensorProcessor* GetSixAxisSensor(const SixAxisSensorHandle& handle) const NN_NOEXCEPT;

    //!< 指定されたハンドルに対応する IVibratorDriver を返します。
    IVibratorDriver* GetVibrator(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< VibratorXcdDriver のリストを返します。
    void GetVibrators(int* pOutCount, IVibratorDriver** outDrivers) NN_NOEXCEPT;

    //!< 指定されたハンドルに対応する VibratorXcdDriver を返します。
    VibratorXcdDriver* GetVibratorXcd(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< 指定されたハンドルに対応する VibratorGcDriver を返します。
    VibratorGcDriver* GetVibratorGc(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< IrSensorXcdDriver を返します。
    IrSensorXcdDriver* GetIrSensorXcdDriver(NpadIdType id) NN_NOEXCEPT;

    //!< 指定されたハンドルに対応する IExternalBusDriver を返します。
    IExternalBusDriver* GetExternalBusDriver(const ExternalBusHandle& handle) NN_NOEXCEPT;

    //!< ExternalBusXcdDriver を返します。
    ExternalBusXcdDriver* GetExternalXcdDriver(NpadIdType id, const int index) NN_NOEXCEPT;

    //!< バッテリーを搭載しているかどうかを取得します。
    Result HasBattery(bool* pOutHasBattery, NpadIdType id) NN_NOEXCEPT;

    //!< バッテリーを搭載しているかどうかを取得します。
    Result HasBattery(bool* pOutLeftHasBattery, bool* pOutRightHasBattery, NpadIdType id) NN_NOEXCEPT;

    //!< 接続形式を取得します。
    Result GetInterfaceType(system::InterfaceType* interfaceType, NpadIdType id) NN_NOEXCEPT;

    //!< 接続形式を取得します。
    Result GetInterfaceType(system::InterfaceType* pOutLeftInterface, system::InterfaceType* pOutRightInterface, NpadIdType id) NN_NOEXCEPT;

    //!< Switch Pro Controller のグリップ色を取得します。
    Result GetGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip, NpadIdType id) NN_NOEXCEPT;

    //!< Npad の内部状態を強制的に更新する
    void UpdateInternalStateForTargetAruid(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Application のリビジョンをセットします。
    void SetNpadApplicationRevision(::nn::applet::AppletResourceUserId aruid, NpadApplicationRevision revision) NN_NOEXCEPT;

    //!< Application のリビジョンを取得します。
    NpadApplicationRevision GetNpadApplicationRevision(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 据置モード中の携帯機コントローラーの操作が行われたかどうか
    bool IsHandheldButtonPressedOnConsoleMode() NN_NOEXCEPT;

private:
    //!< Npad 共通のリソースを初期化します
    void InitializeNpadCommonResourceManager(OvlnSenderManager* pOvlnSender,
                                             PlayReportManager* pPlayReportManager,
                                             AppletResourceManager* pAppletResourceManager,
                                             nn::os::SdkRecursiveMutex* pAppletResourceManagerMutex,
                                             HandheldManager* pHandheldManager,
                                             InputDetectorManager* pInputDetector,
                                             nn::os::SdkMutex* pInputDetectorMutex,
                                             InterruptSceneNotifier* pInterruptSceneNotifier) NN_NOEXCEPT;

    //!< Npad マネージャを初期化します。
    void InitializeNpadManagers(CaptureButtonManager* pCaptureButtonManager,
                                HomeButtonManager* pHomeButtonManager,
                                SixAxisSensorAppletSettingManager* pSixAxisSensorSettingsManager,
                                ConsoleSixAxisSensorManager* pConsoleSixAxisSensor,
                                GcAdapterDriver* pGcAdapterDriver,
                                PalmaResourceManager* pPalma
                                ) NN_NOEXCEPT;

    //!< Npad 割り当てマネージャを初期化します。
    void InitializeNpadAssignmentManager() NN_NOEXCEPT;

    //!< Npad 最後に操作されたコントローラーの管理マネージャを初期化します。
    void InitializeNpadLastActiveControllerMonitor() NN_NOEXCEPT;

    //!< Npad ペリフェラルマネージャを初期化します。
    void InitializeNpadPeripheralManager() NN_NOEXCEPT;

    //!< NpadUniquePad コンバーターを初期化します。
    void InitializeNpadUniquePadConverter(UniquePadResourceManager* pUniquePad) NN_NOEXCEPT;

    //!< Npad Palma のスキャンマネージャーを初期化します
    void InitializeNpadPalmaScanController(NpadAppletPolicyManager* pAppletPolicyManager,
                                           PalmaResourceManager* pPalma
                                           ) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
