﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/hid_ResultPrivate.h>

#include "hid_NpadResourceManager.h"

namespace nn { namespace hid { namespace detail {

NpadResourceManager::NpadResourceManager() NN_NOEXCEPT
{
    // 何もしない
}

NpadResourceManager::~NpadResourceManager() NN_NOEXCEPT
{
    // 何もしない
}

void NpadResourceManager::Initialize(CaptureButtonManager* pCaptureButtonManager,
                                     HomeButtonManager* pHomeButtonManager,
                                     OvlnSenderManager* pOvlnSender,
                                     PlayReportManager* pPlayReportManager,
                                     AppletResourceManager* pAppletResourceManager,
                                     nn::os::SdkRecursiveMutex* pAppletResourceManagerMutex,
                                     SixAxisSensorAppletSettingManager* pSixAxisSensorSettingsManager,
                                     ConsoleSixAxisSensorManager* pConsoleSixAxisSensor,
                                     HandheldManager* pHandheldManager,
                                     InputDetectorManager* pInputDetector,
                                     nn::os::SdkMutex* pInputDetectorMutex,
                                     UniquePadResourceManager* pUniquePad,
                                     InterruptSceneNotifier* pInterruptSceneNotifier,
                                     GcAdapterDriver* pGcAdapterDriver,
                                     PalmaResourceManager* pPalma
                                     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    NN_SDK_REQUIRES_NOT_NULL(pCaptureButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(pHomeButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(pOvlnSender);
    NN_SDK_REQUIRES_NOT_NULL(pPlayReportManager);
    NN_SDK_REQUIRES_NOT_NULL(pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(pAppletResourceManagerMutex);
    NN_SDK_REQUIRES_NOT_NULL(pSixAxisSensorSettingsManager);
    NN_SDK_REQUIRES_NOT_NULL(pConsoleSixAxisSensor);
    NN_SDK_REQUIRES_NOT_NULL(pHandheldManager);
    NN_SDK_REQUIRES_NOT_NULL(pInputDetector);
    NN_SDK_REQUIRES_NOT_NULL(pInputDetectorMutex);
    NN_SDK_REQUIRES_NOT_NULL(pUniquePad);
    NN_SDK_REQUIRES_NOT_NULL(pInterruptSceneNotifier);
    NN_SDK_REQUIRES_NOT_NULL(pPalma);

    InitializeNpadCommonResourceManager(pOvlnSender,
                                        pPlayReportManager,
                                        pAppletResourceManager,
                                        pAppletResourceManagerMutex,
                                        pHandheldManager,
                                        pInputDetector,
                                        pInputDetectorMutex,
                                        pInterruptSceneNotifier);

    InitializeNpadAssignmentManager();

    InitializeNpadLastActiveControllerMonitor();

    InitializeNpadPeripheralManager();

    InitializeNpadManagers(pCaptureButtonManager,
        pHomeButtonManager,
        pSixAxisSensorSettingsManager,
        pConsoleSixAxisSensor,
        pGcAdapterDriver,
        pPalma);

    InitializeNpadUniquePadConverter(pUniquePad);

    InitializeNpadPalmaScanController(&m_AppletPolicyManager, pPalma);
}

void NpadResourceManager::AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    m_AssignmentManager.AddIAbstractedPad(pPads);
}

::nn::Result NpadResourceManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        auto needsRollback = true;

        ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);

        NN_RESULT_DO(m_AssignmentManager.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_AssignmentManager.Deactivate();
            }
        };

        NN_RESULT_DO(m_PeripheralManager.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_PeripheralManager.Deactivate();
            }
        };

        NN_RESULT_DO(m_LastActiveControllerMonitor.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_LastActiveControllerMonitor.Deactivate();
            }
        };

        int i = 0;

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                while (i > 0)
                {
                    --i;
                    m_NpadManagers[i].Deactivate();
                }
            }
        };

        while (i < NN_ARRAY_SIZE(m_NpadManagers))
        {
            m_NpadManagers[i].SetNpadIdType(SupportedNpadIdList[i]);
            NN_RESULT_DO(m_NpadManagers[i].Activate());

            ++i;
        }

        needsRollback = false;
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
        for (int i = NN_ARRAY_SIZE(m_NpadManagers) - 1; i >= 0; --i)
        {
            NN_RESULT_DO(m_NpadManagers[i].Deactivate());
        }

        for (int i = NN_ARRAY_SIZE(m_NpadManagers) - 1; i >= 0; --i)
        {
            NN_RESULT_DO(m_NpadManagers[i].Deactivate());
        }
        m_LastActiveControllerMonitor.Deactivate();
        m_PeripheralManager.Deactivate();
        m_AssignmentManager.Deactivate();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::EnsureNpadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> npadLocker(m_NpadMutex);
    ::std::lock_guard<decltype(m_CommonResourceHolder.GetAppletResourceManagerMutex())
        > appletResourceLocker(m_CommonResourceHolder.GetAppletResourceManagerMutex());
    for (auto& manager : m_NpadManagers)
    {
        NN_RESULT_DO(
            manager.EnsureAppletResource(aruid));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::SetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet style) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);

    NN_RESULT_DO(m_AppletPolicyManager.SetSupportedNpadStyleSet(aruid, style));
    m_AssignmentManager.ProceedSupportedNpadStyleSetUpdate();
    m_PalmaScanController.UpdateScanState();
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::GetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);

    auto result = m_AppletPolicyManager.GetSupportedNpadStyleSet(pOutValue, aruid);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultNpadStyleSetNotInitialized)
        {
            // NpadStyle 未初期化時は全フラグを OFF にする
            pOutValue->Reset();
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::GetMaskedSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);

    auto result = m_AppletPolicyManager.GetMaskedSupportedNpadStyleSet(pOutValue, aruid);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultNpadStyleSetNotInitialized)
        {
            // NpadStyle 未初期化時は全フラグを OFF にする
            pOutValue->Reset();
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::SetSupportedNpadIdType(::nn::applet::AppletResourceUserId aruid, const NpadIdType* pIds, int count) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    if (count > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(pIds);
    }

    if (count > NpadIdCountMax)
    {
        NN_RESULT_THROW(ResultNpadInvalidNpadIdTypeCount());
    }

    NN_RESULT_DO(m_AppletPolicyManager.SetSupportedNpadIdType(aruid, pIds, count));
    m_AssignmentManager.ProceedSupportedNpadIdTypeUpdate();
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::SetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid, NpadJoyHoldType holdType) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_AppletPolicyManager.SetNpadJoyHoldType(aruid, holdType));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::GetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid, NpadJoyHoldType* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOutValue);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_AppletPolicyManager.GetNpadJoyHoldType(pOutValue, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::AcquireStyleUpdateEventHandle(::nn::applet::AppletResourceUserId aruid, ::nn::os::NativeHandle* pOutValue, NpadIdType id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(m_AppletPolicyManager.AcquireStyleUpdateEventHandle(aruid, pOutValue, id));
    NN_RESULT_SUCCESS;
}

void NpadResourceManager::Disconnect(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    m_AssignmentManager.Disconnect(aruid, id);
}

::nn::Result NpadResourceManager::ApplyNpadSystemCommonPolicy(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(m_AppletPolicyManager.SetSystemCommonPolicy(aruid, false));
    m_AssignmentManager.ProceedApplyNpadSystemCommonPolicy();
    m_PalmaScanController.UpdateScanState();
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::ApplyNpadSystemCommonPolicyFull(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(m_AppletPolicyManager.SetSystemCommonPolicy(aruid, true));
    m_AssignmentManager.ProceedApplyNpadSystemCommonPolicy();
    m_PalmaScanController.UpdateScanState();
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::SetAssigningSingleOnSlSrPressMode(bool enable, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    bool currentMode;
    NN_RESULT_DO(m_AppletPolicyManager.IsAssigningSingleOnSlSrPressEnabled(&currentMode, aruid));

    if (currentMode != enable)
    {
        NN_RESULT_DO(m_AppletPolicyManager.SetAssigningSingleOnSlSrPressMode(aruid, enable));
    }

    NN_RESULT_SUCCESS;
}

NpadJoyAssignmentMode NpadResourceManager::GetNpadJoyAssignmentMode(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    return m_AssignmentManager.GetNpadJoyAssignmentMode(aruid, id);
}

void NpadResourceManager::SetNpadJoyAssignmentModeSingleByDefault(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    m_AssignmentManager.SetNpadJoyAssignmentModeSingleByDefault(aruid, id);
}

void NpadResourceManager::SetNpadJoyAssignmentModeSingle(::nn::applet::AppletResourceUserId aruid, NpadIdType id, NpadJoyDeviceType type) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    m_AssignmentManager.SetNpadJoyAssignmentModeSingle(aruid, id, type);
}

bool NpadResourceManager::SetNpadJoyAssignmentModeSingleWithDestination(NpadIdType* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id, NpadJoyDeviceType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    return m_AssignmentManager.SetNpadJoyAssignmentModeSingleWithDestination(pOutValue, aruid, id, type);
}

void NpadResourceManager::SetNpadJoyAssignmentModeDual(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    m_AssignmentManager.SetNpadJoyAssignmentModeDual(aruid, id);
}

::nn::Result NpadResourceManager::MergeSingleJoyAsDualJoy(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_AssignmentManager.MergeSingleJoyAsDualJoy(aruid, id1, id2));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::SwapNpadAssignment(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_AssignmentManager.SwapNpadAssignment(aruid, id1, id2));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::EnableLrAssignmentMode(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
     bool currentMode;
    NN_RESULT_DO(m_AppletPolicyManager.IsLrAssignmentModeEnabled(&currentMode, aruid));

    if (currentMode == false)
    {
        NN_RESULT_DO(m_AppletPolicyManager.SetIsLrAssignmentModeEnabled(aruid, true));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::DisableLrAssignmentMode(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    bool currentMode;
    NN_RESULT_DO(m_AppletPolicyManager.IsLrAssignmentModeEnabled(&currentMode, aruid));

    if (currentMode == true)
    {
        NN_RESULT_DO(m_AppletPolicyManager.SetIsLrAssignmentModeEnabled(aruid, false));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::SetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid, NpadHandheldActivationMode mode) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);

    NN_RESULT_DO(m_AppletPolicyManager.SetNpadHandheldActivationMode(aruid, mode));

    // Npad の内部状態を更新
    m_NpadManagers[GetIndexFromNpadIdType(nn::hid::NpadId::Handheld)].UpdateNpad();

    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::GetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid, NpadHandheldActivationMode* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);

    NN_RESULT_DO(m_AppletPolicyManager.GetNpadHandheldActivationMode(pOutValue, aruid));
    NN_RESULT_SUCCESS;
}

void NpadResourceManager::SetNpadPlayerLedBlinkingDevice(::nn::applet::AppletResourceUserId aruid, NpadIdType id, system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    GetNpadManager(m_NpadManagers, id).GetIndicator().SetPlayerLedBlinkingDevice(deviceType);
}

::nn::Result NpadResourceManager::AcquireNfcDeviceUpdateEventHandle(::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_PeripheralManager.AcquireNfcDeviceUpdateEventHandle(pOutHandle));
    NN_RESULT_SUCCESS;
}

int NpadResourceManager::GetNpadsWithNfc(NpadIdType* pOutIds, int length) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    return m_PeripheralManager.GetNpadsWithNfc(pOutIds, length);
}

::nn::Result NpadResourceManager::GetXcdHandleForNpadWithNfc(::nn::xcd::DeviceHandle* pOutHandle, NpadIdType id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        GetNpadManager(m_NpadManagers, id).GetNfc().GetXcdHandleForNpadWithNfc(pOutHandle, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::AcquireNfcActivateEventHandle(::nn::os::NativeHandle* pOutHandle, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_PeripheralManager.AcquireNfcActivateEventHandle(pOutHandle, id));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::ActivateNfc(NpadIdType id, bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_PeripheralManager.ActivateNfc(id, activate, aruid));
    NN_RESULT_SUCCESS;
}

bool NpadResourceManager::IsNfcActivated(NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    return GetNpadManager(m_NpadManagers, id).GetNfc().IsNfcActivated();
}

::nn::Result NpadResourceManager::AcquireIrSensorEventHandle(::nn::os::NativeHandle* pOutHandle, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_PeripheralManager.AcquireIrSensorEventHandle(pOutHandle, id));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::ActivateIrSensor(NpadIdType id, bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_PeripheralManager.ActivateIrSensor(id, activate, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::GetLastActiveNpad(NpadIdType* pOutValue) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_LastActiveControllerMonitor.GetLastActiveNpad(pOutValue));
    NN_RESULT_SUCCESS;
}

void NpadResourceManager::GetNpadSystemExtStyle(system::NpadSystemExtMainStyle* pOutMainStyle,
                                       system::NpadSystemExtSubStyle* pOutSubStyle,
                                       NpadIdType id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMainStyle);
    NN_SDK_REQUIRES_NOT_NULL(pOutSubStyle);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    GetNpadManager(m_NpadManagers, id).GetDeviceManager().GetSystemExtStyle(pOutMainStyle, pOutSubStyle);
}

::nn::Result NpadResourceManager::IsUsbFullKeyControllerConnected(bool* pOutConnected, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    auto intefaceType = GetNpadManager(m_NpadManagers, id).GetDeviceManager().GetInterfaceTypeOfFullKey();
    *pOutConnected = (intefaceType == system::InterfaceType_Usb);
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::GetXcdHandleForNpadWithIrSensor(NpadIdType id, ::nn::xcd::DeviceHandle* pOutHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        GetNpadManager(m_NpadManagers, id).GetIrSensor().GetXcdHandleForNpadWithIrSensor(pOutHandle, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::GetIrSensorState(NpadIdType id, system::IrSensorState* pOutValue , ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    *pOutValue = GetNpadManager(m_NpadManagers, id).GetIrSensor().GetIrSensorState(aruid);
    NN_RESULT_SUCCESS;
}

Result NpadResourceManager::ActivateRailAttachment(NpadIdType id, NpadJoyConRailIndex index, bool activate) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_PeripheralManager.ActivateRailAttachment(id, index, activate));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::GetXcdHandleForNpadWithRailAttachment(::nn::xcd::DeviceHandle* pOutHandle, NpadIdType id, NpadJoyConRailIndex index) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        GetNpadManager(m_NpadManagers, id).GetRailAttachment().GetXcdHandleForNpadWithRailAttachment(pOutHandle, index));
    NN_RESULT_SUCCESS;
}

NpadJoyConRailAttachmentState NpadResourceManager::GetNpadJoyConRailAttachmentState(NpadIdType id, NpadJoyConRailIndex index) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    return GetNpadManager(m_NpadManagers, id).GetRailAttachment().GetNpadJoyConRailAttachmentState(index);
}


::nn::Result NpadResourceManager::GetAbstractedPadIdForRailAttachment(AbstractedPadId* pOutId, NpadIdType id, NpadJoyConRailIndex index) NN_NOEXCEPT
{
    nn::hid::detail::IAbstractedPad* abstractedPad[AbstractedPadCountMaxPerNpad];
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    auto count = GetNpadManager(m_NpadManagers, id).GetAbstractedPadHolder().GetAbstractedPads(abstractedPad, AbstractedPadCountMaxPerNpad);
    for (int i = 0; i < count; i++)
    {
        auto deviceType = abstractedPad[i]->GetDeviceType();
        if ((deviceType & nn::hid::system::DeviceType::JoyConLeft::Mask).IsAnyOn())
        {
            if (index == NpadJoyConRailIndex_Left)
            {
                *pOutId = abstractedPad[i]->GetId();
                NN_RESULT_SUCCESS;
            }
        }
        else if ((deviceType & nn::hid::system::DeviceType::JoyConRight::Mask).IsAnyOn())
        {
            if (index == NpadJoyConRailIndex_Right)
            {
                *pOutId = abstractedPad[i]->GetId();
                NN_RESULT_SUCCESS;
            }
        }
    }
    return nn::hid::ResultNoExternalBusFoundOnNpad();
}


::nn::Result NpadResourceManager::IsUnintendedHomeButtonInputProtectionEnabled(bool* pOutEnabled,
                                                          ::nn::applet::AppletResourceUserId aruid,
                                                          NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(
        m_AppletPolicyManager.IsNpadHomeButtonEnabled(pOutEnabled, aruid, id));
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::EnableUnintendedHomeButtonInputProtection(::nn::applet::AppletResourceUserId aruid,
                                                                            NpadIdType id,
                                                                            bool enabled) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> npadLocker(m_NpadMutex);
    ::std::lock_guard<decltype(m_CommonResourceHolder.GetAppletResourceManagerMutex())
        > appletResourceLocker(m_CommonResourceHolder.GetAppletResourceManagerMutex());
    // Home ボタン無効化機能の有効無効 から Home ボタンの有効無効のフラグを反転
    NN_RESULT_DO(
        GetNpadManager(m_NpadManagers, id).GetInputStateUpdater().SetHomeButtonEnabledWithSharedMemoryUpdate(!enabled, aruid));
    NN_RESULT_SUCCESS;
}

int NpadResourceManager::GetUniquePadsFromNpad(system::UniquePadId* pOutValues, int count, NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    return m_NpadUniquePadConverter.GetUniquePads(pOutValues, count, id);
}

::nn::Result NpadResourceManager::RegisterNpadAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)
                      > locker(m_NpadMutex);
    NN_RESULT_DO(
        m_AppletPolicyManager.RegisterAppletResourceUserId(aruid));
    NN_RESULT_SUCCESS;
}

void NpadResourceManager::UnregisterNpadAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)
                      > locker(m_NpadMutex);
    m_AppletPolicyManager.UnregisterAppletResourceUserId(aruid);
}

::nn::Result NpadResourceManager::ActivateNpadAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)
                      > locker(m_NpadMutex);
    NN_RESULT_DO(
        m_AppletPolicyManager.ActivateNpadAppletPolicyEntry(aruid));
    NN_RESULT_SUCCESS;
}

void NpadResourceManager::DeactivateNpadAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)
                      > locker(m_NpadMutex);
    m_AppletPolicyManager.DeactivateNpadAppletPolicyEntry(aruid);
}

void NpadResourceManager::SetNpadAppletResourceUserId(
    ::nn::applet::AppletResourceUserId value) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)
                      > locker(m_NpadMutex);
    m_AppletPolicyManager.SetFocuesedAppletResourceUserId(value);
    m_AssignmentManager.UpdateOnAruidSwitch();
    m_PalmaScanController.UpdateScanState();
}

::nn::Result NpadResourceManager::ActivateNpadSharedAppletResource() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)
                      > locker(m_NpadMutex);
    NN_RESULT_DO(m_AppletPolicyManager.ActivateSharedNpadAppletPolicy());
    NN_RESULT_SUCCESS;
}

::nn::Result NpadResourceManager::DeactivateNpadSharedAppletResource() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)
                      > locker(m_NpadMutex);
    NN_RESULT_DO(m_AppletPolicyManager.DeactivateSharedNpadAppletPolicy());
    NN_RESULT_SUCCESS;
}

void NpadResourceManager::ResumeNpad() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)
                      > locker(m_NpadMutex);
    m_AssignmentManager.Resume();
}

void NpadResourceManager::HandlePeriodcialEvent() NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        // 初期化前は何もせずに返る
        return;
    }

    ::std::lock_guard<decltype(m_NpadMutex)> npadLocker(m_NpadMutex);

    m_AssignmentManager.UpdateDeviceMap();
    for (auto& manager : m_NpadManagers)
    {
        manager.ProceedPeriodicEvent();
    }

    {
        ::std::lock_guard<decltype(m_CommonResourceHolder.GetAppletResourceManagerMutex())
            > appletResourceLocker(m_CommonResourceHolder.GetAppletResourceManagerMutex());
        for (auto& manager : m_NpadManagers)
        {
            manager.UpdateSharedMemoryForPeriodicalStates();
        }
    }

    m_LastActiveControllerMonitor.UpdateLastActiveNpad();
}

void NpadResourceManager::UpdateHandheld() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);

    m_NpadManagers[GetIndexFromNpadIdType(nn::hid::NpadId::Handheld)].UpdateNpad();
}

const SixAxisSensorProcessor* NpadResourceManager::GetSixAxisSensor(const SixAxisSensorHandle& handle) const NN_NOEXCEPT
{
    if (IsSixAxisSensorHandleNpadTypeEqualTo(handle) == false)
    {
        return nullptr;
    }

    NpadIdType id = GetSixAxisSensorHandleNpadIdType(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(VerifyValidNpadId(id));
    return GetNpadManager(m_NpadManagers, id).GetSixAxisSensorProcessor(handle);
}

IVibratorDriver* NpadResourceManager::GetVibrator(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    int id = GetVibrationDeviceHandlePlayerNumber(handle);
    int vibrationDeviceIdx;
    auto deviceType = GetVibrationDeviceHandleDeviceType(handle);
    if (deviceType == VibrationDeviceType_LinearResonantActuator)
    {
        vibrationDeviceIdx = GetVibrationDeviceHandleDeviceIdx(handle);
        return GetNpadManager(m_NpadManagers, id).GetVibratorXcdDriver(vibrationDeviceIdx);
    }
    else if(deviceType == VibrationDeviceType_GcErm)
    {
        return GetNpadManager(m_NpadManagers, id).GetVibratorGcDriver();
    }

    return nullptr;
}

void NpadResourceManager::GetVibrators(int* pOutCount, IVibratorDriver** outDrivers) NN_NOEXCEPT
{
    *pOutCount = 0;
    for (auto& manager : m_NpadManagers)
    {
        int count = 0;
        VibratorXcdDriver* pXcdDrivers[NpadVibratorCountPerPlayerMax];
        manager.GetVibratorXcdDrivers(&count, pXcdDrivers);
        for (int i = 0; i < count; i++)
        {
            outDrivers[*pOutCount] = pXcdDrivers[i];
            ++*pOutCount;
        }

        auto pDriver = manager.GetVibratorGcDriver();
        if (pDriver != nullptr)
        {
            outDrivers[*pOutCount] = pDriver;
            ++(*pOutCount);
        }
    }
}

VibratorXcdDriver* NpadResourceManager::GetVibratorXcd(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    int id = GetVibrationDeviceHandlePlayerNumber(handle);
    if (GetVibrationDeviceHandleDeviceType(handle) == VibrationDeviceType_LinearResonantActuator)
    {
        int vibrationDeviceIdx = GetVibrationDeviceHandleDeviceIdx(handle);
        return GetNpadManager(m_NpadManagers, id).GetVibratorXcdDriver(vibrationDeviceIdx);
    }

    return nullptr;
}

VibratorGcDriver* NpadResourceManager::GetVibratorGc(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    int id = GetVibrationDeviceHandlePlayerNumber(handle);
    if (GetVibrationDeviceHandleDeviceType(handle) == VibrationDeviceType_GcErm)
    {
        return GetNpadManager(m_NpadManagers, id).GetVibratorGcDriver();
    }

    return nullptr;
}

IrSensorXcdDriver* NpadResourceManager::GetIrSensorXcdDriver(NpadIdType id) NN_NOEXCEPT
{
    if (VerifyValidNpadId(id).IsFailure())
    {
        return nullptr;
    }
    return GetNpadManager(m_NpadManagers, (id)).GetIrSensorXcdDriver();
}

ExternalBusXcdDriver* NpadResourceManager::GetExternalXcdDriver(NpadIdType id, const int index) NN_NOEXCEPT
{
    if (VerifyValidNpadId(id).IsFailure())
    {
        return nullptr;
    }
    return GetNpadManager(m_NpadManagers, (id)).GetExternalBusXcdDriver(index);
}

IExternalBusDriver* NpadResourceManager::GetExternalBusDriver(const ExternalBusHandle& handle) NN_NOEXCEPT
{
    int id = GetExternalBusHandlePlayerNumber(handle);
    auto busType = GetExternalBusHandleBusTypeId(handle);
    if (busType == ExternalBusType_LeftJoyRail)
    {
        return GetNpadManager(m_NpadManagers, id).GetExternalBusXcdDriver(0);
    }
    else if (busType == ExternalBusType_RightJoyRail)
    {
        return GetNpadManager(m_NpadManagers, id).GetExternalBusXcdDriver(1);
    }

    return nullptr;
}

Result NpadResourceManager::GetFirmwareVersion(debug::FirmwareVersion *pOutValue,
                                               bool* pOutIsMcuIapCorrupted,
                                               NpadIdType id,
                                               system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pOutIsMcuIapCorrupted);
    NN_SDK_REQUIRES_EQUAL(deviceType.CountPopulation(), 1);
    NN_UNUSED(pOutIsMcuIapCorrupted);

    system::UniquePadId uniquePadId;
    if (m_NpadUniquePadConverter.GetUniquePads(&uniquePadId, id, deviceType) == false)
    {
        NN_RESULT_THROW(system::ResultUniquePadDisconnected());
    }

    NN_RESULT_DO(m_pUniquePadResourceManager->GetFirmwareVersion(pOutValue, uniquePadId));
    NN_RESULT_SUCCESS;
}

Result NpadResourceManager::GetDestinationFirmwareVersion(debug::FirmwareVersion *pOutValue,
                                                          NpadIdType id,
                                                          system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_EQUAL(deviceType.CountPopulation(), 1);

    system::UniquePadId uniquePadId;
    if (m_NpadUniquePadConverter.GetUniquePads(&uniquePadId, id, deviceType) == false)
    {
        NN_RESULT_THROW(system::ResultUniquePadDisconnected());
    }

    NN_RESULT_DO(m_pUniquePadResourceManager->GetDestinationFirmwareVersion(pOutValue, uniquePadId));
    NN_RESULT_SUCCESS;

}

void NpadResourceManager::InitializeNpadCommonResourceManager(OvlnSenderManager* pOvlnSender,
                                                              PlayReportManager* pPlayReportManager,
                                                              AppletResourceManager* pAppletResourceManager,
                                                              nn::os::SdkRecursiveMutex* pAppletResourceManagerMutex,
                                                              HandheldManager* pHandheldManager,
                                                              InputDetectorManager* pInputDetector,
                                                              nn::os::SdkMutex* pInputDetectorMutex,
                                                              InterruptSceneNotifier* pInterruptSceneNotifier) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOvlnSender);
    NN_SDK_REQUIRES_NOT_NULL(pPlayReportManager);
    NN_SDK_REQUIRES_NOT_NULL(pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(pAppletResourceManagerMutex);
    NN_SDK_REQUIRES_NOT_NULL(pHandheldManager);
    NN_SDK_REQUIRES_NOT_NULL(pInputDetector);
    NN_SDK_REQUIRES_NOT_NULL(pInputDetectorMutex);
    NN_SDK_REQUIRES_NOT_NULL(pInterruptSceneNotifier);

    m_CommonResourceHolder.SetAppletResourceManager(pAppletResourceManager, pAppletResourceManagerMutex);
    m_CommonResourceHolder.SetHandheldManager(pHandheldManager);
    m_CommonResourceHolder.SetInputDetectorManager(pInputDetector, pInputDetectorMutex);
    m_CommonResourceHolder.SetInterruptSceneNotifier(pInterruptSceneNotifier);
    m_CommonResourceHolder.SetNpadAppletPolicyManager(&m_AppletPolicyManager);
    m_CommonResourceHolder.SetOvlnSenderManager(pOvlnSender);
    m_CommonResourceHolder.SetPlayReportManager(pPlayReportManager);
}

void NpadResourceManager::InitializeNpadManagers(CaptureButtonManager* pCaptureButtonManager,
                                                 HomeButtonManager* pHomeButtonManager,
                                                 SixAxisSensorAppletSettingManager* pSixAxisSensorSettingsManager,
                                                 ConsoleSixAxisSensorManager* pConsoleSixAxisSensor,
                                                 GcAdapterDriver* pGcAdapterDriver,
                                                 PalmaResourceManager* pPalma
                                                 ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCaptureButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(pHomeButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(pSixAxisSensorSettingsManager);
    NN_SDK_REQUIRES_NOT_NULL(pConsoleSixAxisSensor);
    NN_SDK_REQUIRES_NOT_NULL(pGcAdapterDriver);
    NN_SDK_REQUIRES_NOT_NULL(pPalma);

    for (auto& manager : m_NpadManagers)
    {
        manager.Initialize(&m_CommonResourceHolder,
                           pCaptureButtonManager,
                           pHomeButtonManager,
                           pSixAxisSensorSettingsManager,
                           pConsoleSixAxisSensor,
                           pGcAdapterDriver,
                           pPalma
                           );
    }
}

void NpadResourceManager::InitializeNpadAssignmentManager() NN_NOEXCEPT
{
    m_AssignmentManager.SetNpadCommonResourceHolder(&m_CommonResourceHolder);
    m_AssignmentManager.SetNpadManagers(&m_NpadManagers);
    m_AssignmentManager.SetNpadLastActiveControllerMonitor(&m_LastActiveControllerMonitor);
}

void NpadResourceManager::InitializeNpadLastActiveControllerMonitor() NN_NOEXCEPT
{
    m_LastActiveControllerMonitor.SetNpadCommonResourceHolder(&m_CommonResourceHolder);
    m_LastActiveControllerMonitor.SetNpadManagers(&m_NpadManagers);
}

void NpadResourceManager::InitializeNpadPeripheralManager() NN_NOEXCEPT
{
    m_PeripheralManager.SetNpadManagers(&m_NpadManagers);
}

void NpadResourceManager::InitializeNpadUniquePadConverter(UniquePadResourceManager* pUniquePad) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pUniquePad);

    m_pUniquePadResourceManager = pUniquePad;
    m_NpadUniquePadConverter.SetUniquePadResourceManager(pUniquePad);
}

void NpadResourceManager::InitializeNpadPalmaScanController(NpadAppletPolicyManager* pAppletPolicyManager,
                                                        PalmaResourceManager* pPalma
                                                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAppletPolicyManager);
    NN_SDK_REQUIRES_NOT_NULL(pPalma);
    m_PalmaScanController.SetNpadAppletPolicyManager(pAppletPolicyManager);
    m_PalmaScanController.SetPalmaResourceManager(pPalma);
}

Result NpadResourceManager::HasBattery(bool* pOutHasBattery, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    *pOutHasBattery = GetNpadManager(m_NpadManagers, id).GetPowerInfoUpdater().HasBattery();
    NN_RESULT_SUCCESS;
}

Result NpadResourceManager::HasBattery(bool* pOutLeftHasBattery, bool* pOutRightHasBattery, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    GetNpadManager(m_NpadManagers, id).GetPowerInfoUpdater().HasBattery(pOutLeftHasBattery, pOutRightHasBattery);
    NN_RESULT_SUCCESS;
}

Result NpadResourceManager::GetInterfaceType(system::InterfaceType* interfaceType, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    *interfaceType = GetNpadManager(m_NpadManagers, id).GetDeviceManager().GetInterfaceType();
    NN_RESULT_SUCCESS;
}

Result NpadResourceManager::GetInterfaceType(system::InterfaceType* pOutLeftInterface, system::InterfaceType* pOutRightInterface, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    GetNpadManager(m_NpadManagers, id).GetDeviceManager().GetInterfaceType(pOutLeftInterface, pOutRightInterface);
    NN_RESULT_SUCCESS;
}

Result NpadResourceManager::GetGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip, NpadIdType id) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    NN_RESULT_DO(GetNpadManager(m_NpadManagers, id).GetDeviceManager().GetGripColor(pOutLeftGrip, pOutRightGrip));
    NN_RESULT_SUCCESS;
}

void NpadResourceManager::UpdateInternalStateForTargetAruid(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    // Npad の内部状態を更新
    for (auto& manager : m_NpadManagers)
    {
        manager.UpdateSharedMemoryForTarget(aruid);
    }
}

void NpadResourceManager::SetNpadApplicationRevision(::nn::applet::AppletResourceUserId aruid, NpadApplicationRevision revision) NN_NOEXCEPT
{
    m_AppletPolicyManager.SetNpadApplicationRevision(aruid, revision);
}

NpadApplicationRevision NpadResourceManager::GetNpadApplicationRevision(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    return m_AppletPolicyManager.GetNpadApplicationRevision(aruid);
}

bool NpadResourceManager::IsHandheldButtonPressedOnConsoleMode() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_NpadMutex)> locker(m_NpadMutex);
    return GetNpadManager(m_NpadManagers, nn::hid::NpadId::Handheld).GetInputStateUpdater().IsHandheldButtonPressedOnConsoleMode();
}

}}} // namespace nn::hid::detail
