﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/system/hid_NpadCommon.h>

#include "hid_ActivationCount.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadDeviceManager.h"
#include "hid_NpadCommonResourceHolder.h"

namespace nn { namespace hid { namespace detail {

//!< 電源状態の識別子
enum PowerInfoIndex : uint8_t
{
    PowerInfoIndex_Standard = 0,  // 主となる電源状態
    PowerInfoIndex_Left,          // Joy-Con(L) の電源状態
    PowerInfoIndex_Right,         // Joy-Con(R) の電源状態
};

//!< PowerInfo の数
const int PowerInfoCountMax = 3;

//!< 電源状態の更新を担うクラスです。
class NpadPowerInfoUpdater final
{
    NN_DISALLOW_COPY(NpadPowerInfoUpdater);
    NN_DISALLOW_MOVE(NpadPowerInfoUpdater);

private:
    //!< Npad が共通で扱う hid のサービスを保持するクラス
    NpadCommonResourceHolder* m_pCommonResourceHolder;

    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager* m_pDeviceManager;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

    //!< 電源状態を保持する配列
    system::NpadPowerInfo m_PowerInfo[PowerInfoCountMax];

    //!< 共有メモリ更新が必要かどうか
    bool m_IsSharedMemoryUpdateRequired;

public:
    NpadPowerInfoUpdater() NN_NOEXCEPT;

    ~NpadPowerInfoUpdater() NN_NOEXCEPT;

    //!< NpadCommonResourceHolder を設定します。
    void SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadDeviceManager を設定します。
    void SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Npad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureNpadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 入力状態を更新します。
    void UpdatePowerInfo() NN_NOEXCEPT;

    //!< 共有メモリの状態を更新します。
    void UpdateSharedMemory() NN_NOEXCEPT;

    //!< 特定の Aruid に対して共有メモリ内部の状態を更新します
    void UpdateSharedMemoryForTarget(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 電池残量を取得します。
    system::NpadPowerInfo GetPowerInfo(int index) NN_NOEXCEPT;

    //!< バッテリーを搭載しているかどうかを取得します。
    bool HasBattery() NN_NOEXCEPT;

    //!< バッテリーを搭載しているかどうかを取得します。
    void HasBattery(bool* pOutLeftHasBattery, bool* pOutRightHasBattery) NN_NOEXCEPT;

private:
    //!< 電池状態を更新します。
    bool UpdatePowerInfoImpl() NN_NOEXCEPT;

    //!< 電池残量低下による通知の有無のチェックを行います
    void ProceedLowBatteryNotification(IAbstractedPad* pPad) NN_NOEXCEPT;

    //!< Npad の共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(NpadPowerInfoUpdater* that,
                          NpadSharedMemoryEntry* address) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 指定した Aruid のものについてのみ共有メモリを処理します。
    void ProcessSharedMemoryForTargetAruid(
        ::nn::applet::AppletResourceUserId aruid,
        void (*processor)(NpadPowerInfoUpdater* that,
                          NpadSharedMemoryEntry* address) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 共有メモリ上の電池残量を更新する
    void UpdateSharedMemoryForPowerInfo(NpadSharedMemoryEntry* address) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
