﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_CommonStateUtility.h"
#include "hid_NpadId.h"
#include "hid_NpadIndicatorPattern.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadPowerInfoUpdater.h"
#include "hid_NpadStateUtility.h"
#include "hid_Settings.h"

namespace nn { namespace hid { namespace detail {

namespace
{
void ResetPowerInfo(system::NpadPowerInfo* pPowerInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPowerInfo);
    pPowerInfo->batteryLevel = system::NpadBatteryLevel_None;
    pPowerInfo->isCharging = false;
    pPowerInfo->isPowered = false;
}

} // namespace

NpadPowerInfoUpdater::NpadPowerInfoUpdater() NN_NOEXCEPT
    : m_pCommonResourceHolder(nullptr)
    , m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_NpadActivationCount()
    , m_IsSharedMemoryUpdateRequired(false)
{
    for (auto& powerInfo : m_PowerInfo)
    {
        ResetPowerInfo(&powerInfo);
    }
}

NpadPowerInfoUpdater::~NpadPowerInfoUpdater() NN_NOEXCEPT { /* 何もしない */ }

void NpadPowerInfoUpdater::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadPowerInfoUpdater::SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(pHolder->IsInitialized());
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pCommonResourceHolder = pHolder;
}

void NpadPowerInfoUpdater::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);

    m_pDeviceManager = pHolder;
}

::nn::Result NpadPowerInfoUpdater::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pDeviceManager);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadPowerInfoUpdater::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadPowerInfoUpdater::EnsureNpadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->ProcessSharedMemoryForTargetAruid(
                              aruid,
                              [](NpadPowerInfoUpdater* that,
                                 NpadSharedMemoryEntry* address) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForPowerInfo(address);
    });

    NN_RESULT_SUCCESS;
}

void NpadPowerInfoUpdater::UpdatePowerInfo() NN_NOEXCEPT
{
    if (!m_NpadActivationCount.IsZero())
    {
        // 電池残量の状態が更新されているのであればフラグを落とす
        m_IsSharedMemoryUpdateRequired = this->UpdatePowerInfoImpl();
    }
}

void NpadPowerInfoUpdater::UpdateSharedMemory() NN_NOEXCEPT
{
    if (!m_NpadActivationCount.IsZero())
    {
        if (m_IsSharedMemoryUpdateRequired)
        {
            this->ProcessSharedMemory([](NpadPowerInfoUpdater* that,
                NpadSharedMemoryEntry* address) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(that);
                NN_SDK_REQUIRES_NOT_NULL(address);

                that->UpdateSharedMemoryForPowerInfo(address);
            });

            // フラグを落とす
            m_IsSharedMemoryUpdateRequired = false;
        }
    }
}

void NpadPowerInfoUpdater::UpdateSharedMemoryForTarget(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->ProcessSharedMemoryForTargetAruid(
                              aruid,
                              [](NpadPowerInfoUpdater* that,
                                 NpadSharedMemoryEntry* address) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForPowerInfo(
            address);
    });
}

system::NpadPowerInfo NpadPowerInfoUpdater::GetPowerInfo(int index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(index, PowerInfoCountMax);

    return m_PowerInfo[index];
}

void NpadPowerInfoUpdater::ProcessSharedMemory(
    void (*processor)(NpadPowerInfoUpdater* that,
                      NpadSharedMemoryEntry* address) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    const auto index =
        static_cast<size_t>(GetIndexFromNpadIdType(m_pDeviceManager->GetNpadIdType()));

    for (const AppletResourceEntry& entry :
             m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->npad.entries)>::value);
            NN_SDK_ASSERT_NOT_NULL(&entry.address->npad.entries[index]);
            processor(this,
                &entry.address->npad.entries[index]);
        }
    }
}

void NpadPowerInfoUpdater::ProcessSharedMemoryForTargetAruid(
    ::nn::applet::AppletResourceUserId aruid,
    void (*processor)(NpadPowerInfoUpdater* that,
                      NpadSharedMemoryEntry* address) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    const auto index =
        static_cast<size_t>(GetIndexFromNpadIdType(m_pDeviceManager->GetNpadIdType()));

    for (const AppletResourceEntry& entry :
             m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.aruid == aruid)
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->npad.entries)>::value);
            NN_SDK_ASSERT_NOT_NULL(&entry.address->npad.entries[index]);
            processor(this,
                &entry.address->npad.entries[index]);

            return;
        }
    }
}

void NpadPowerInfoUpdater::UpdateSharedMemoryForPowerInfo(NpadSharedMemoryEntry* address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);

    ::nn::util::Get(address->internalState).SetNpadPowerInfo(m_PowerInfo[PowerInfoIndex_Standard], m_PowerInfo[PowerInfoIndex_Left], m_PowerInfo[PowerInfoIndex_Right]);
}

bool NpadPowerInfoUpdater::UpdatePowerInfoImpl() NN_NOEXCEPT
{
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));
    bool isUpdated = false;
    system::NpadPowerInfo powerInfoTemp[NN_ARRAY_SIZE(m_PowerInfo)];

    for (auto& powerInfo : powerInfoTemp)
    {
        ResetPowerInfo(&powerInfo);
    }

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected())
        {
            auto power = static_cast<system::NpadPowerInfo>(pPads[i]->GetPowerInfo());
            auto deviceType = pPads[i]->GetDeviceType();
            if (deviceType.Test<system::DeviceType::FullKeyController>() ||
                deviceType.Test<system::DeviceType::Palma>())
            {
                powerInfoTemp[PowerInfoIndex_Standard] = power;
            }

            if ((deviceType & DeviceTypeMask_JoyLefts).IsAnyOn())
            {
                powerInfoTemp[PowerInfoIndex_Left] = power;
            }

            if ((deviceType & DeviceTypeMask_JoyRights).IsAnyOn())
            {
                powerInfoTemp[PowerInfoIndex_Right] = power;
            }

            // 電池残量通知の処理
            ProceedLowBatteryNotification(pPads[i]);
        }
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_PowerInfo); i++)
    {
        if (IsPowerInfoSame(m_PowerInfo[i], powerInfoTemp[i]) == false)
        {
            m_PowerInfo[i] = powerInfoTemp[i];
            isUpdated = true;
        }
    }

    if (isUpdated)
    {
        auto deviceType = m_pDeviceManager->GetNpadDeviceTypeSet();
        if (deviceType.Test<system::DeviceType::FullKeyController>() == false)
        {
            // 左コンの電池状態が優先
            if ((deviceType & DeviceTypeMask_JoyLefts).IsAnyOn())
            {
                m_PowerInfo[PowerInfoIndex_Standard] = m_PowerInfo[PowerInfoIndex_Left];
            }
            else if ((deviceType & DeviceTypeMask_JoyRights).IsAnyOn())
            {
                m_PowerInfo[PowerInfoIndex_Standard] = m_PowerInfo[PowerInfoIndex_Right];
            }
        }
    }

    return isUpdated;
}

void NpadPowerInfoUpdater::ProceedLowBatteryNotification(IAbstractedPad* pPad) NN_NOEXCEPT
{
    if (pPad->IsBatteryLowOvlnRequired())
    {
        nn::ovln::format::ControllerBatteryLevelLowExData data = {};

        m_pDeviceManager->CreateNotifyControllerInfo(&data.controllerInfo);
        data.npadId = m_pDeviceManager->GetNpadIdType();

        data.controllerInfo.fullKeyBattery.isValid = false;
        data.controllerInfo.joyBattery.left.isValid = false;
        data.controllerInfo.joyBattery.right.isValid = false;

        if ((m_pDeviceManager->GetNpadDeviceTypeSet() & DeviceTypeMask_JoyLefts).IsAnyOn())
        {
            data.controllerInfo.joyBattery.left.powerInfo = m_PowerInfo[PowerInfoIndex_Left];
            data.controllerInfo.joyBattery.left.isValid = true;
        }

        if ((m_pDeviceManager->GetNpadDeviceTypeSet() & DeviceTypeMask_JoyRights).IsAnyOn())
        {
            data.controllerInfo.joyBattery.right.powerInfo = m_PowerInfo[PowerInfoIndex_Right];
            data.controllerInfo.joyBattery.right.isValid = true;
        }

        if (m_pDeviceManager->GetNpadDeviceTypeSet().Test<system::DeviceType::FullKeyController>())
        {
            data.controllerInfo.fullKeyBattery.powerInfo = m_PowerInfo[PowerInfoIndex_Standard];
            data.controllerInfo.fullKeyBattery.isValid = true;
        }

        if (pPad->GetDeviceType().Test<system::DeviceType::JoyConLeft>())
        {
            data.controllerInfo.joyType = system::NpadDeviceType::JoyConLeft::Mask;
        }
        if (pPad->GetDeviceType().Test<system::NpadDeviceType::JoyConRight>())
        {
            data.controllerInfo.joyType = system::NpadDeviceType::JoyConRight::Mask;
        }

        m_pCommonResourceHolder->GetOvlnSenderManager().Send(data);
        pPad->ResetIsBatteryLowOvlnOnNotify(nn::os::GetSystemTick());
    }
}

bool NpadPowerInfoUpdater::HasBattery() NN_NOEXCEPT
{
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    if (count == 0)
    {
        return false;
    }

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected())
        {
            return pPads[i]->HasBattery();
        }
    }

        return false;
}

void NpadPowerInfoUpdater::HasBattery(bool* pOutLeftHasBattery, bool* pOutRightHasBattery) NN_NOEXCEPT
{
    *pOutLeftHasBattery = false;
    *pOutRightHasBattery = false;

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    if (count == 0)
    {
        return;
    }

    for (int i = 0; i < count; i++)
    {
        if ( pPads[i]->IsConnected() )
        {
            auto deviceType = pPads[i]->GetDeviceType();
            if (deviceType.Test<system::DeviceType::JoyConLeft>() ||
                deviceType.Test<system::DeviceType::HandheldJoyLeft>())
            {
                *pOutLeftHasBattery = pPads[i]->HasBattery();
            }
            else if (deviceType.Test<system::DeviceType::JoyConRight>() ||
                     deviceType.Test<system::DeviceType::HandheldJoyRight>())
            {
                *pOutRightHasBattery = pPads[i]->HasBattery();
            }
        }
    }
}

}}} // namespace nn::hid::detail
