﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>

#include "hid_ActivationCount.h"
#include "hid_NpadManager.h"

namespace nn { namespace hid { namespace detail {

const int McuDeviceActiveDeviceCountMax = 4;

//!< Nfc や IrSensor などのペリフェラルの制御を扱うマネージャーです。
class NpadPeripheralManager final
{
    NN_DISALLOW_COPY(NpadPeripheralManager) NN_NOEXCEPT;
    NN_DISALLOW_MOVE(NpadPeripheralManager) NN_NOEXCEPT;

private:
    //!< NpadManager の配列
    NpadManagerArray* m_pNpads;

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< Nfc 状態変化通知イベント
    ::nn::os::SystemEvent m_NfcSystemEvent;

    //!< Nfc のデバイス状態排他のための Mutex
    ::nn::os::Mutex m_NfcStateMutex;

    //!< IrSensor のデバイス状態排他のための Mutex
    ::nn::os::Mutex m_IrSensorStateMutex;

public:
    NpadPeripheralManager() NN_NOEXCEPT;

    ~NpadPeripheralManager() NN_NOEXCEPT;

    //!< NpadManager の配列をセットします。
    void SetNpadManagers(NpadManagerArray* pNpads) NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Nfc デバイスの状態変化通知イベントをバインドします。
    ::nn::Result AcquireNfcDeviceUpdateEventHandle(::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT;

    //!< 有効な Nfc デバイスに対する xcd のハンドルを取得します。
    int GetNpadsWithNfc(NpadIdType* pOutIds, int length) NN_NOEXCEPT;

    //!< Nfc デバイスのアクティブ通知イベントをバインドします。
    ::nn::Result AcquireNfcActivateEventHandle(::nn::os::NativeHandle* pOutHandle, NpadIdType id) NN_NOEXCEPT;

    //!< Nfc デバイスのアクティブ状態を制御します。
    ::nn::Result ActivateNfc(NpadIdType id, bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< IrSensor デバイスの状態変化イベントをバインドします。
    ::nn::Result AcquireIrSensorEventHandle(::nn::os::NativeHandle* pOutHandle, NpadIdType id) NN_NOEXCEPT;

    //!< IrSensor デバイスのアクティブ状態を制御します。
    ::nn::Result ActivateIrSensor(NpadIdType id, bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< レール接続デバイスのアクティブ状態を制御します。
    Result ActivateRailAttachment(NpadIdType id, NpadJoyConRailIndex index, bool activate) NN_NOEXCEPT;

private:
    //!< Nfc または IrSensor を使用しているデバイスの数が上限以下であるかどうかを確認します
    bool IsUnderMaxMcuDevices(NpadIdType id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
