﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_NpadPeripheralManager.h"


namespace nn { namespace hid { namespace detail {

NpadPeripheralManager::NpadPeripheralManager() NN_NOEXCEPT
    : m_pNpads(nullptr)
    , m_ActivationCount()
    , m_NfcSystemEvent(::nn::os::EventClearMode_ManualClear, true)
    , m_NfcStateMutex(false)
    , m_IrSensorStateMutex(false)
{
    // 何もしない
}

NpadPeripheralManager::~NpadPeripheralManager() NN_NOEXCEPT
{
    // 何もしない
}

void NpadPeripheralManager::SetNpadManagers(NpadManagerArray* pNpads) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNpads);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());

    m_pNpads = pNpads;

    // Nfc のイベントを登録する
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        (*m_pNpads)[i].GetNfc().SetNfcUpdateEvent(&m_NfcSystemEvent);
    }

}

::nn::Result NpadPeripheralManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pNpads);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadPeripheralManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadPeripheralManager::AcquireNfcDeviceUpdateEventHandle(
    ::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    *pOutHandle = m_NfcSystemEvent.GetReadableHandle();

    m_NfcSystemEvent.Signal();

    NN_RESULT_SUCCESS;
}

::nn::Result NpadPeripheralManager::AcquireNfcActivateEventHandle(
    ::nn::os::NativeHandle* pOutHandle, NpadIdType id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    return GetNpadManager(*m_pNpads, id).GetNfc().AcquireNfcActivateEventHandle(pOutHandle);
}

::nn::Result NpadPeripheralManager::ActivateNfc(NpadIdType id, bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // Nfc を Activate できるかどうかチェック
    if (activate == true)
    {
        // 他で Nfc がアクティブになっていないか確認
        // Handheld はほかの Npad の有効状態に影響を受けない
        if (id != NpadId::Handheld)
        {
            if(IsUnderMaxMcuDevices(id, aruid) == false)
            {
                NN_RESULT_THROW(system::ResultNfcActivatedOnOtherNpad());
            }
        }

        // Nfc と干渉する他の機能が有効になっていないか確認
        if (GetNpadManager(*m_pNpads, id).GetIrSensor().GetIrSensorState(aruid) == system::IrSensorState_Activated ||
            GetNpadManager(*m_pNpads, id).GetRailAttachment().GetNpadJoyConRailAttachmentState(NpadJoyConRailIndex_Right) == NpadJoyConRailAttachmentState_Activated)
        {
            NN_RESULT_THROW(system::ResultNfcActivateFailureNpadBusy());
        }
    }

    return GetNpadManager(*m_pNpads, id).GetNfc().ActivateNfc(activate, aruid);
}

int NpadPeripheralManager::GetNpadsWithNfc(NpadIdType* pOutIds, int length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIds);

    ::std::lock_guard<decltype(m_NfcStateMutex)
                      > locker(m_NfcStateMutex);

    int nfcOutCount = 0;

    for (int i = 0; i < NpadIdCountMax; i++)
    {
        if ((*m_pNpads)[i].GetNfc().IsNfcAvailable() == true)
        {
            pOutIds[nfcOutCount] = SupportedNpadIdList[i];
            nfcOutCount++;

            if (nfcOutCount >= length)
            {
                break;
            }
        }
    }

    return nfcOutCount;
}

::nn::Result NpadPeripheralManager::AcquireIrSensorEventHandle(
    ::nn::os::NativeHandle* pOutHandle, NpadIdType id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    return GetNpadManager(*m_pNpads, id).GetIrSensor().AcquireIrSensorEventHandle(pOutHandle);
}

::nn::Result NpadPeripheralManager::ActivateIrSensor(NpadIdType id, bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // 他で IrSensor がアクティブになっていないか確認
    if (activate == true)
    {
        // IrSensor/Nfc の有効数が上限を上回っていないか確認
        // Handheld は台数にカウントしない
        if (id != NpadId::Handheld)
        {
            if(IsUnderMaxMcuDevices(id, aruid) == false)
            {
                NN_RESULT_THROW(system::ResultIrSensorActivationLimitOver());
            }
        }

        // IrSensor と干渉する他の機能が有効になっているかどうか確認
        if (GetNpadManager(*m_pNpads, id).GetNfc().IsNfcActivated() == true ||
            GetNpadManager(*m_pNpads, id).GetRailAttachment().GetNpadJoyConRailAttachmentState(NpadJoyConRailIndex_Right) == NpadJoyConRailAttachmentState_Activated)
        {
            NN_RESULT_THROW(system::ResultIrSensorActivateFailureNpadBusy());
        }
    }

    NN_RESULT_DO(GetNpadManager(*m_pNpads, id).GetIrSensor().ActivateIrSensor(activate, aruid));

    NN_RESULT_SUCCESS;
}

::nn::Result NpadPeripheralManager::ActivateRailAttachment(NpadIdType id, NpadJoyConRailIndex index, bool activate) NN_NOEXCEPT
{
    nn::applet::AppletResourceUserId aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();

    // レールと干渉する他の機能が有効になっていないか確認
    if (activate == true && index == NpadJoyConRailIndex_Right)
    {
        if (GetNpadManager(*m_pNpads, id).GetNfc().IsNfcActivated() == true ||
            GetNpadManager(*m_pNpads, id).GetIrSensor().GetIrSensorState(aruid) == system::IrSensorState_Activated)
        {
            NN_RESULT_THROW(ResultExternalBusActivateFailureNpadBusy());
        }
    }

    NN_RESULT_DO(GetNpadManager(*m_pNpads, id).GetRailAttachment().ActivateRailAttachment(activate, index));

    NN_RESULT_SUCCESS;
}

bool NpadPeripheralManager::IsUnderMaxMcuDevices(NpadIdType id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    int connectedDeviceCount = 0;
    int mcuDeviceCount = 0;
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        if (SupportedNpadIdList[i] != NpadId::Handheld)
        {
            // コントローラーの接続台数をカウント
            connectedDeviceCount += (*m_pNpads)[i].GetAbstractedPadHolder().GetDeviceType().CountPopulation();

            // 他のコントローラーで Ir/Nfc を有効にしている数をカウント
            if (SupportedNpadIdList[i] != id)
            {
                if ((*m_pNpads)[i].GetIrSensor().GetIrSensorState(aruid) == system::IrSensorState_Activated ||
                    (*m_pNpads)[i].GetNfc().IsNfcActivated() == true)
                {
                    mcuDeviceCount++;

                    if (mcuDeviceCount == McuDeviceActiveDeviceCountMax)
                    {
                        return false;
                    }
                }
            }

            // 5台以上デバイスが接続されちえるときは、2つ以上 mcu を ON にできない
            if (connectedDeviceCount > 4 && mcuDeviceCount > 0)
            {
                return false;
            }
        }
    }

    return true;
}

}}} // namespace nn::hid::detail
