﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_CommonStateUtility.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadNfcController.h"
#include "hid_NpadStateUtility.h"
#include "hid_Settings.h"
#include "hid_VibrationDeviceHandle.h"
#include "hid_VibratorXcdDriverImpl.h"

namespace nn { namespace hid { namespace detail {

NpadNfcController::NpadNfcController() NN_NOEXCEPT
    : m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_NpadActivationCount()
    , m_NfcActivateEvent(::nn::os::EventClearMode_ManualClear, true)
    , m_NfcState(NfcState_NoDevice)
    , m_NfcXcdDeviceHandle()
{
    // 何もしない
}

NpadNfcController::~NpadNfcController() NN_NOEXCEPT { /* 何もしない */ }

void NpadNfcController::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadNfcController::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);

    m_pDeviceManager = pHolder;
}

::nn::Result NpadNfcController::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pDeviceManager);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadNfcController::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

void NpadNfcController::UpdateDeviceStates() NN_NOEXCEPT
{
    auto oldIsAvailable = this->IsNfcAvailable();
    auto state = NfcState_NoDevice;
    bool isAvailable = false;

    // 対応する NpadStyleSet の指定前は全スタイルが有効と見なす
    bool isSupportedAllStyle = !m_pDeviceManager->IsSupportedNpadStyleSetInitialized();

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected() &&
            pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::Nfc>() &&
            pPads[i]->GetType() == AbstractedPadType_Xcd)
        {
            if (isSupportedAllStyle ||
                m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
            {
                isAvailable = true;
                m_NfcXcdDeviceHandle = reinterpret_cast<AbstractedPadXcd*>(pPads[i])->GetXcdDeviceHandle();
            }
        }
    }

    if (isAvailable == false)
    {
        state = NfcState_NoDevice;
        // Nfc がアクティブ状態だったらアクティブ状態変更のイベントをシグナル
        if (m_NfcState == NfcState_Active)
        {
            m_NfcActivateEvent.Signal();
        }
    }
    else
    {
        state = IsNfcActivated() ? NfcState_Active : NfcState_Deactive;
    }
    m_NfcState = state;

    // Nfc の状態に変化があった場合は通知
    if (oldIsAvailable != IsNfcAvailable())
    {
        m_pNfcUpdateEvent->Signal();
    }
}

Result NpadNfcController::AcquireNfcActivateEventHandle(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_NfcActivateEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

void NpadNfcController::SetNfcUpdateEvent(::nn::os::SystemEvent* pEvent) NN_NOEXCEPT
{
    m_pNfcUpdateEvent = pEvent;
}

Result NpadNfcController::ActivateNfc(bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_THROW_UNLESS(IsNfcAvailable() == true, system::ResultNoNfcDeviceFoundOnNpad());

    if (m_NfcState != NfcState_NoDevice)
    {
        auto nextNfcState = (activate) ? NfcState_Active : NfcState_Deactive;
        if (m_NfcState != nextNfcState)
        {
            m_NfcState = nextNfcState;
            m_NfcActivateEvent.Signal();
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadNfcController::GetXcdHandleForNpadWithNfc(::nn::xcd::DeviceHandle* pOutHandle, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    if (this->IsNfcAvailable() == false)
    {
        NN_RESULT_THROW(system::ResultNoNfcDeviceFoundOnNpad());
    }
    *pOutHandle = m_NfcXcdDeviceHandle;
    NN_RESULT_SUCCESS;
}

bool NpadNfcController::IsNfcAvailable() NN_NOEXCEPT
{
    return m_NfcState != NfcState_NoDevice;
}

bool NpadNfcController::IsNfcActivated() NN_NOEXCEPT
{
    return m_NfcState == NfcState_Active;
}

}}} // namespace nn::hid::detail
