﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_NpadLastActiveControllerMonitor.h"

namespace nn { namespace hid { namespace detail {


NpadLastActiveControllerMonitor::NpadLastActiveControllerMonitor() NN_NOEXCEPT
    : m_pCommonResourceHolder(nullptr)
    , m_pNpads(nullptr)
    , m_ActivationCount()
    , m_LastActiveNpadIndex(0)
    , m_IsLastActiveNpadExist(false)
{
    // 何もしない
}

NpadLastActiveControllerMonitor::~NpadLastActiveControllerMonitor() NN_NOEXCEPT
{
    // 何もしない
}

void NpadLastActiveControllerMonitor::SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(pHolder->IsInitialized());
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pCommonResourceHolder = pHolder;
}

void NpadLastActiveControllerMonitor::SetNpadManagers(NpadManagerArray* pNpads) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNpads);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());

    m_pNpads = pNpads;
}

::nn::Result NpadLastActiveControllerMonitor::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pNpads);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
        m_LastActiveNpadIndex = 0;
        m_IsLastActiveNpadExist = false;
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadLastActiveControllerMonitor::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    NN_RESULT_SUCCESS;
}

void NpadLastActiveControllerMonitor::SetLastActiveNpad(const NpadIdType& id) NN_NOEXCEPT
{
    // Handheld は Handheld が有効な場合のみ更新
    if (m_pCommonResourceHolder->GetHandheldManager().IsHandheldEnabled() == false && id == NpadId::Handheld)
    {
        return;
    }

    m_LastActiveNpadIndex = GetIndexFromNpadIdType(id);
    m_IsLastActiveNpadExist = true;
}

::nn::Result NpadLastActiveControllerMonitor::GetLastActiveNpad(NpadIdType* pOutValue) NN_NOEXCEPT
{
    if (m_IsLastActiveNpadExist == false || m_LastActiveNpadIndex >= NpadIdCountMax)
    {
        NN_RESULT_THROW(system::ResultNoActiveNpad());
    }
    *pOutValue = (*m_pNpads)[m_LastActiveNpadIndex].GetNpadIdType();
    NN_RESULT_SUCCESS;
}

void NpadLastActiveControllerMonitor::UpdateLastActiveNpad() NN_NOEXCEPT
{
    if (m_IsLastActiveNpadExist == true && m_LastActiveNpadIndex < NpadIdCountMax)
    {
        // 操作中のコントローラーが接続かつ入力状態にある場合は、変更しない
        if ((*m_pNpads)[m_LastActiveNpadIndex].GetInputStateUpdater().IsNpadSystemExtInputEnable() &&
            (*m_pNpads)[m_LastActiveNpadIndex].GetAbstractedPadHolder().GetDeviceType().IsAnyOn())
        {
            return;
        }
    }

    // 入力状態にあるNpadを NpadId::No1 -> No8 -> Handheld の順で探索する
    for (int npadIndex = 0; npadIndex < NpadIdCountMax; npadIndex++)
    {
        if ((*m_pNpads)[npadIndex].GetAbstractedPadHolder().GetDeviceType().IsAnyOn() &&
            (*m_pNpads)[npadIndex].GetInputStateUpdater().IsNpadSystemExtInputEnable())
        {
            m_LastActiveNpadIndex = npadIndex;
            m_IsLastActiveNpadExist = true;
            return;
        }
    }

    // 操作状態にあるコントローラーが見つからなかった場合

    // 前回操作状態にあったコントローラーが接続状態にあったら更新しない
    if (m_IsLastActiveNpadExist && (*m_pNpads)[m_LastActiveNpadIndex].GetAbstractedPadHolder().GetDeviceType().IsAnyOn())
    {
        if ((*m_pNpads)[m_LastActiveNpadIndex].GetNpadIdType() == NpadId::Handheld &&
            m_pCommonResourceHolder->GetHandheldManager().IsHandheldEnabled() == false)
        {
            // 前回操作状態にあったコントローラーが Handheld かつ Handheld が無効になっている場合は、最後に操作したデバイスを更新する
        }
        else
        {
            // それ以外の場合は最後に操作したコントローラーが接続状態にあるので更新せず
            return;
        }
    }

    // 接続状態にあるNpadを NpadId::No1 -> No8 -> Handheld の順で探索する
    for (int npadIndex = 0; npadIndex < NpadIdCountMax; npadIndex++)
    {
        if ((*m_pNpads)[npadIndex].GetAbstractedPadHolder().GetDeviceType().IsAnyOn())
        {
            // Handheld の場合は 携帯機操作が無効になっている場合は選択しない
            if ((*m_pNpads)[npadIndex].GetNpadIdType() == NpadId::Handheld && m_pCommonResourceHolder->GetHandheldManager().IsHandheldEnabled() == false)
            {
                continue;
            }

            m_LastActiveNpadIndex = npadIndex;
            m_IsLastActiveNpadExist = true;
            return;
        }
    }

    // 接続状態にあるコントローラーがみつからなかった場合、Handheldが有効であれば Handheld にする
    if (m_pCommonResourceHolder->GetHandheldManager().IsHandheldEnabled() == true)
    {
        m_LastActiveNpadIndex = GetIndexFromNpadIdType(NpadId::Handheld);
        m_IsLastActiveNpadExist = true;
        return;
    }

    // 有効なコントローラーなし
    m_IsLastActiveNpadExist = false;
}

}}} // namespace nn::hid::detail
