﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/os/os_NativeHandle.h>

#include "hid_ActivationCount.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadDeviceManager.h"

namespace nn { namespace hid { namespace detail {

//!< IrSensor の状態の制御を担うクラスです。
class NpadIrSensorController final
{
    NN_DISALLOW_COPY(NpadIrSensorController);
    NN_DISALLOW_MOVE(NpadIrSensorController);

private:
    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager* m_pDeviceManager;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

    //!< IrSensor デバイスのアクティブ状態に変化があった際に通知する システムイベント
    ::nn::os::SystemEvent m_IrSensorEvent;

    //!< IrSensor の状態
    system::IrSensorState m_IrSensorState;

    //!< IrSensor に割り当てられた Xcd デバイスのハンドル
    ::nn::xcd::DeviceHandle m_IrSensorXcdDeviceHandle;

public:
    NpadIrSensorController() NN_NOEXCEPT;

    ~NpadIrSensorController() NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadDeviceManager を設定します。
    void SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< IrSensor の状態を更新します。
    void UpdateDeviceStates() NN_NOEXCEPT;

    //!< IrSensor のアクティブ状態が変化したときに受ける通知をイベントオブジェクトにバインドします。
    Result AcquireIrSensorEventHandle(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< IrSensor のアクティブ状態を制御します。
    Result ActivateIrSensor(bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadId に対して割り当てられている IrSensor のデバイスハンドルを取得します
    ::nn::Result GetXcdHandleForNpadWithIrSensor(::nn::xcd::DeviceHandle* pOutHandle, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< IrSensor の状態を取得します。
    system::IrSensorState GetIrSensorState(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
