﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadIrSensorController.h"
#include "hid_NpadStateUtility.h"

namespace nn { namespace hid { namespace detail {

NpadIrSensorController::NpadIrSensorController() NN_NOEXCEPT
    : m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_NpadActivationCount()
    , m_IrSensorEvent(::nn::os::EventClearMode_ManualClear, true)
    , m_IrSensorState(system::IrSensorState_NotConnected)
    , m_IrSensorXcdDeviceHandle()
{
}

NpadIrSensorController::~NpadIrSensorController() NN_NOEXCEPT { /* 何もしない */ }

void NpadIrSensorController::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadIrSensorController::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);

    m_pDeviceManager = pHolder;
}

::nn::Result NpadIrSensorController::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pDeviceManager);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadIrSensorController::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

void NpadIrSensorController::UpdateDeviceStates() NN_NOEXCEPT
{
    auto oldState = m_IrSensorState;

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    bool isAvailable = false;

    if (count == 0)
    {
        // 何も接続されていない
        m_IrSensorState = system::IrSensorState_NotConnected;
    }
    else
    {
        for (int i = 0; i < count; i++)
        {
            if (pPads[i]->IsConnected() &&
                pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::IrSensor>() &&
                pPads[i]->GetType() == AbstractedPadType_Xcd)
            {
                if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                {
                    isAvailable = true;
                    m_IrSensorXcdDeviceHandle = reinterpret_cast<AbstractedPadXcd*>(pPads[i])->GetXcdDeviceHandle();
                }
            }
        }

        if (isAvailable == true)
        {
            if (m_IrSensorState != system::IrSensorState_Activated)
            {
                m_IrSensorState = system::IrSensorState_Deactivated;
            }
            else
            {
                m_IrSensorState = system::IrSensorState_Activated;
            }
        }
        else
        {
            m_IrSensorState = system::IrSensorState_NotSupported;
            ActivateIrSensor(false, nn::applet::AppletResourceUserId::GetInvalidId());
        }
    }

    // IrSensor の状態に変化があった場合は通知
    if (oldState != m_IrSensorState)
    {
        m_IrSensorEvent.Signal();
    }
}

Result NpadIrSensorController::AcquireIrSensorEventHandle(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_IrSensorEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

::nn::Result NpadIrSensorController::GetXcdHandleForNpadWithIrSensor(::nn::xcd::DeviceHandle* pOutHandle, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    if (m_IrSensorState == system::IrSensorState_NotConnected ||
        m_IrSensorState == system::IrSensorState_NotSupported)
    {
        NN_RESULT_THROW(system::ResultNoIrSensorDeviceFoundOnNpad());
    }

    *pOutHandle = m_IrSensorXcdDeviceHandle;
    NN_RESULT_SUCCESS;
}

system::IrSensorState NpadIrSensorController::GetIrSensorState(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    return m_IrSensorState;
}

Result NpadIrSensorController::ActivateIrSensor(bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    auto state = this->m_IrSensorState;
    NN_RESULT_THROW_UNLESS(state != system::IrSensorState_NotSupported, system::ResultNoIrSensorDeviceFoundOnNpad());

    if (activate == true && state == system::IrSensorState_Deactivated)
    {
        m_IrSensorState = system::IrSensorState_Activated;
        m_IrSensorEvent.Signal();
    }
    else if (activate == false && state == system::IrSensorState_Activated)
    {
        m_IrSensorState = system::IrSensorState_Deactivated;
        m_IrSensorEvent.Signal();
    }

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
