﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

#include "hid_ActivationCount.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadCommonResourceHolder.h"
#include "hid_NpadDeviceManager.h"

namespace nn { namespace hid { namespace detail {

//!< インジケーターの制御を担うクラスです
class NpadIndicatorController final
{
    NN_DISALLOW_COPY(NpadIndicatorController);
    NN_DISALLOW_MOVE(NpadIndicatorController);

private:
    //!< Npad が共通で扱う hid のサービスを保持するクラス
    NpadCommonResourceHolder* m_pCommonResourceHolder;

    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager* m_pDeviceManager;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

    //!< LED を点滅させるデバイス
    system::NpadDeviceTypeSet m_LedBlinkingDevice;

    //!< LED の点灯パターン
    uint8_t m_LedPattern;

    //!< 反転時の LED 点灯パターン
    uint8_t m_LedRotatePattern;

public:
    NpadIndicatorController() NN_NOEXCEPT;

    ~NpadIndicatorController() NN_NOEXCEPT;

    //!< NpadCommonResourceHolder を設定します。
    void SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadDeviceManager を設定します。
    void SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< プレイヤーインジケーターの点灯パターンを更新します
    void UpdatePattern() NN_NOEXCEPT;

    //!< プレイヤーインジケーターの点灯状態を更新します
    void UpdateState() NN_NOEXCEPT;

    //!< プレイヤーランプを点滅させるコントローラーを設定します
    void SetPlayerLedBlinkingDevice(system::NpadDeviceTypeSet deviceTpe) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
