﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_NpadIndicatorController.h"
#include "hid_NpadIndicatorPattern.h"

namespace nn { namespace hid { namespace detail {

NpadIndicatorController::NpadIndicatorController() NN_NOEXCEPT
    : m_pCommonResourceHolder(nullptr)
    , m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_NpadActivationCount()
    , m_LedPattern(0)
    , m_LedRotatePattern(0)
{
    m_LedBlinkingDevice.Reset();
}

NpadIndicatorController::~NpadIndicatorController() NN_NOEXCEPT { /* 何もしない */ }

void NpadIndicatorController::SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(pHolder->IsInitialized());
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pCommonResourceHolder = pHolder;
}

void NpadIndicatorController::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadIndicatorController::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);

    m_pDeviceManager = pHolder;
}

::nn::Result NpadIndicatorController::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadIndicatorController::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

void NpadIndicatorController::UpdatePattern() NN_NOEXCEPT
{
    m_LedPattern = GetIndicatorPattern(m_pDeviceManager->GetNpadIdType(), false);
    m_LedRotatePattern = GetIndicatorPattern(m_pDeviceManager->GetNpadIdType(), true);
}

void NpadIndicatorController::UpdateState() NN_NOEXCEPT
{
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));
    if (count == 0)
    {
        return;
    }

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected())
        {
            bool blink = (pPads[i]->GetDeviceType() & m_LedBlinkingDevice).IsAnyOn();
            bool rotate = (pPads[i]->GetDeviceType().Test<system::DeviceType::JoyConRight>() &&
                m_pAbstractedPadHolder->GetJoyAssignmentMode() == NpadJoyAssignmentMode_Single &&
                m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadJoyHoldType() == NpadJoyHoldType_Horizontal);
            pPads[i]->SetIndicator((rotate) ? m_LedRotatePattern : m_LedPattern,
                blink);
        }
    }
}

void NpadIndicatorController::SetPlayerLedBlinkingDevice(system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    m_LedBlinkingDevice = deviceType;
}

}}} // namespace nn::hid::detail
