﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadControlPad.h>
#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <nn/hid/hid_NpadJoyLeft.h>
#include <nn/hid/hid_NpadJoyRight.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/hid_NpadPalma.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_NpadColor.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/hid/system/hid_NpadCommon.h>
#include <nn/hid/system/hid_NpadSystem.h>
#include <nn/hid/system/hid_NpadSystemExt.h>
#include <nn/xcd/xcd_Device.h>

namespace nn { namespace hid { namespace detail {

::nn::Result InitializeNpad() NN_NOEXCEPT;

::nn::Result FinalizeNpad() NN_NOEXCEPT;

::nn::Result BindNpadStyleSetUpdateEvent(const NpadIdType& id,
                             ::nn::os::SystemEventType* pEvent,
                             ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

::nn::Result GetNpadStyleSet(NpadStyleSet* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result DisconnectNpad(const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetPlayerLedPattern(Bit8* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadJoyAssignment(NpadJoyAssignmentMode* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result SetNpadJoyAssignmentModeSingle(const NpadIdType& id) NN_NOEXCEPT;

::nn::Result SetNpadJoyAssignmentModeSingle(const NpadIdType& id, NpadJoyDeviceType type) NN_NOEXCEPT;

::nn::Result SetNpadJoyAssignmentModeSingle(bool* pOutIsAssigned, NpadIdType* pOutId, const NpadIdType& id, NpadJoyDeviceType type) NN_NOEXCEPT;

::nn::Result SetNpadJoyAssignmentModeDual(const NpadIdType& id) NN_NOEXCEPT;

::nn::Result MergeSingleJoyAsDualJoy(const NpadIdType& id1, const NpadIdType& id2) NN_NOEXCEPT;

::nn::Result SwapNpadAssignment(const NpadIdType& id1, const NpadIdType& id2) NN_NOEXCEPT;

::nn::Result GetNpadControllerColor(NpadControllerColor* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadControllerColor(NpadControllerColor* pOutLeftColor, NpadControllerColor* pOutRightColor, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadFullKeyState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadHandheldState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadJoyDualState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadJoyLeftState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadJoyRightState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadGcState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadPalmaState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           system::NpadSystemState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadStates(int* pOutGotCount,
                           system::NpadSystemExtState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT;

template<typename Tag>
inline ::nn::Result GetNpadState(Tag* pOutState, const NpadIdType& id
                          ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    int gotCount;
    NN_RESULT_DO(GetNpadStates(&gotCount, pOutState, 1, id));
    NN_SDK_REQUIRES_EQUAL(gotCount, 1);
    NN_UNUSED(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorStates(int* pOutGotCount,
                                    SixAxisSensorState* pOutStates,
                                    int count,
                                    const NpadIdType& id,
                                    NpadStyleSet style) NN_NOEXCEPT;

::nn::Result GetSixAxisSensorRightStates(int* pOutGotCount,
                                         SixAxisSensorState* pOutStates,
                                         int count,
                                         const NpadIdType& id,
                                         NpadStyleSet style) NN_NOEXCEPT;

::nn::Result GetSixAxisSensorLeftStates(int* pOutGotCount,
                                        SixAxisSensorState* pOutStates,
                                        int count,
                                        const NpadIdType& id,
                                        NpadStyleSet style) NN_NOEXCEPT;

::nn::Result GetNpadDeviceType(system::NpadDeviceTypeSet* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadPowerInfo(system::NpadPowerInfo* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadPowerInfo(system::NpadPowerInfo* pOutLeftJoy, system::NpadPowerInfo* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result IsUnsupportedButtonPressed(bool* pOutValue, const NpadIdType& id, const NpadStyleSet& style) NN_NOEXCEPT;

::nn::Result IsAbxyButtonOriented(bool* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result IsSlSrButtonOriented(bool* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetPlusMinusButtonCapability(bool* pOutPlusButton, bool* pOutMinusButton, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result SetControlPadRestrictionType(NpadStyleSet style, ControlPadRestrictionType type) NN_NOEXCEPT;

::nn::Result GetControlPadRestrictionType(NpadStyleSet style, ControlPadRestrictionType* pOutValue) NN_NOEXCEPT;

::nn::Result EnableControlPadRestriction(const NpadStyleSet& style, bool enable) NN_NOEXCEPT;

::nn::Result SetShiftAccelerometerCalibrationValue(const SixAxisSensorHandle& handle, float shiftOrigin, float shiftSensitivity) NN_NOEXCEPT;

::nn::Result GetShiftAccelerometerCalibrationValue(float* pOutShiftOrigin, float* pOutShiftSensitivity, const SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result SetShiftGyroscopeCalibrationValue(const SixAxisSensorHandle& handle, float shiftOrigin, float shiftSensitivity) NN_NOEXCEPT;

::nn::Result GetShiftGyroscopeCalibrationValue(float* pOutShiftOrigin, float* pOutShiftSensitivity, const SixAxisSensorHandle& handle) NN_NOEXCEPT;

::nn::Result HasBattery(bool* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result HasBattery(bool* pOutLeftJoy, bool* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadInterfaceType(system::InterfaceType* pOutValue, const NpadIdType& id) NN_NOEXCEPT;

::nn::Result GetNpadInterfaceType(system::InterfaceType* pOutLeftJoy, system::InterfaceType* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT;

}}} // namespace nn::hid::detail
