﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_CommonStateUtility.h"
#include "hid_NpadIndicatorPattern.h"
#include "hid_NpadId.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadDeviceManager.h"
#include "hid_NpadStateUtility.h"
#include "hid_Settings.h"

namespace nn { namespace hid { namespace detail {

namespace
{

const NpadControllerColor NoColor            = { {{ 0, 0, 0, 0}},      {{ 0, 0, 0, 0}} };

uint8_t ConvertPlayReportControllerNumber(NpadIdType id) NN_NOEXCEPT
{
    switch (id)
    {
    case NpadId::No1:
    case NpadId::No2:
    case NpadId::No3:
    case NpadId::No4:
    case NpadId::No5:
    case NpadId::No6:
    case NpadId::No7:
    case NpadId::No8:
        return static_cast<uint8_t>(id) + 1;
    case NpadId::Handheld:
        return 0x00;
    case system::NpadSystemId::Other:
        return 0xff;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

} // namespace

NpadDeviceManager::NpadDeviceManager() NN_NOEXCEPT
    : m_pCommonResourceHolder(nullptr)
    , m_pAbstractedPadHolder(nullptr)
    , m_NpadActivationCount()
    , m_SystemExtMainStyle(system::NpadSystemExtMainStyle_None)
    , m_SystemExtSubStyle(system::NpadSystemExtSubStyle_Default)
    , m_IsAbxyOriented(false)
    , m_IsSlSrOriented(false)
    , m_IsPlusAvailable(false)
    , m_IsMinusAvailable(false)
    , m_IsPlayReportUpdateRequired(false)
{
    m_DeviceType.Reset();
    m_StyleSet.Reset();
    m_FullKeyColor.attribute = ColorAttribute_NoController;
    m_JoyColor.attribute = ColorAttribute_NoController;
}

NpadDeviceManager::~NpadDeviceManager() NN_NOEXCEPT { /* 何もしない */ }

void NpadDeviceManager::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadDeviceManager::SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(pHolder->IsInitialized());
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pCommonResourceHolder = pHolder;
}

void NpadDeviceManager::SetNpadIdType(NpadIdType id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(VerifyValidNpadId(id));

    // Npad Id をセット
    m_NpadId = id;
}

NpadIdType NpadDeviceManager::GetNpadIdType() NN_NOEXCEPT
{
    return m_NpadId;
}

::nn::Result NpadDeviceManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
        auto needsRollback = true;

        this->ProcessSharedMemory([] (NpadDeviceManager* that,
                                      NpadSharedMemoryEntry* address,
                                      ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(that);
            NN_UNUSED(aruid);
            ::nn::util::Get(address->internalState).Clear();
        });

        needsRollback = false;
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadDeviceManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadDeviceManager::EnsureNpadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // 共有メモリを更新
    this->ProcessSharedMemoryForTargetAruid(
                                            aruid,
                                            [](NpadDeviceManager* that,
                                               NpadSharedMemoryEntry* address,
                                               ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForControlStates(aruid, address);
    });

    NN_RESULT_SUCCESS;
}

void NpadDeviceManager::UpdateControlStates() NN_NOEXCEPT
{
    auto oldDeviceType = m_DeviceType;
    auto oldStyleSet = m_StyleSet;
    auto oldSystemExtMainStyle = m_SystemExtMainStyle;
    auto oldSystemExtSubStyle = m_SystemExtSubStyle;

    // デバイスタイプの更新
    UpdateDeviceType();

    // 現在のスタイルの更新処理
    UpdateStyleSet();

    // 色情報の更新
    UpdateControllerColor();

    // UI 表示情報の更新
    UpdateButtonOrient();

    // 本体機能向け操作スタイル情報の更新
    UpdateSystemExtStyle();

    // デバイスの状態や操作形態に変化があった場合に行う処理
    if (oldDeviceType != m_DeviceType ||
        oldStyleSet != m_StyleSet ||
        oldSystemExtMainStyle != m_SystemExtMainStyle ||
        oldSystemExtSubStyle != m_SystemExtSubStyle)
    {
        // プレイレポート用のコントローラーの操作状態を更新
        UpdateControllerUsageForPlayReport();
    }
}

void NpadDeviceManager::UpdateSharedMemory() NN_NOEXCEPT
{
    // 共有メモリを更新
    this->ProcessSharedMemory([](NpadDeviceManager* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForControlStates(aruid, address);
    });
}

system::NpadDeviceTypeSet NpadDeviceManager::GetNpadDeviceTypeSet() NN_NOEXCEPT
{
    return m_DeviceType;
}

NpadStyleSet NpadDeviceManager::GetNpadStyleSet() NN_NOEXCEPT
{
    return m_StyleSet;
}

system::InterfaceType NpadDeviceManager::GetInterfaceTypeOfFullKey() NN_NOEXCEPT
{
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    if (count == 0)
    {
        return system::InterfaceType_Unknown;
    }

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected() &&
            pPads[i]->GetType() == AbstractedPadType_Xcd &&
            pPads[i]->GetDeviceType().Test<system::DeviceType::FullKeyController>())
        {
            return pPads[i]->GetInterfaceType();
        }
    }

    return system::InterfaceType_Unknown;
}


NpadStyleSet NpadDeviceManager::GetAvailableNpadStyleSet(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadStyleSet style;
    if (m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetMaskedSupportedNpadStyleSet(&style, aruid).IsFailure())
    {
        return NpadStyleSet();
    }
    // Handheld の有効状態にあわせてマスクする
    if (style.Test<NpadStyleHandheld>())
    {
        NpadHandheldActivationMode handheldActivationMode;
        if (m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetNpadHandheldActivationMode(&handheldActivationMode, aruid).IsFailure())
        {
            // デフォルトは Dual
            handheldActivationMode = NpadHandheldActivationMode_Dual;
        }
        if (m_pCommonResourceHolder->GetHandheldManager().IsHandheldEnabled() == false)
        {
            style.Reset<NpadStyleHandheld>();
        }
        else if (handheldActivationMode == NpadHandheldActivationMode_Dual &&
            (m_DeviceType & DeviceTypeMask_HandheldJoyCons) != DeviceTypeMask_HandheldJoyCons
            )
        {
            style.Reset<NpadStyleHandheld>();
        }
        else if(handheldActivationMode == NpadHandheldActivationMode_Single &&
                (m_DeviceType & DeviceTypeMask_HandheldJoyCons).IsAllOff()
            )
        {
            style.Reset<NpadStyleHandheld>();
        }
    }
    style = m_StyleSet & MaskNpadStyleSetWithId(style, m_NpadId);

    if (style.Test<NpadStyleFullKey>() && style.Test<NpadStyleJoyDual>())
    {
        style.Reset<NpadStyleFullKey>();
    }

    if (style.Test<NpadStyleFullKey>() && style.Test<NpadStyleGc>())
    {
        style.Reset<NpadStyleFullKey>();
    }
    return style;
}

bool NpadDeviceManager::IsStyleActiveForDevice(system::DeviceTypeSet deviceType) NN_NOEXCEPT
{
    // 対応するスタイルが有効かどうかをチェック
    auto style = GetAvailableNpadStyleSet(m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceUserId());
    if (deviceType.Test<system::DeviceType::FullKeyController>())
    {
        if ((style & StyleSetMask_FullKeySupportedStyles).IsAnyOn())
        {
            return true;
        }
    }
    else if (deviceType.Test<system::DeviceType::JoyConLeft>())
    {
        if ((style & StyleSetMask_JoyLeftSupportedStyles).IsAnyOn())
        {
            return true;
        }
    }
    else if (deviceType.Test<system::DeviceType::JoyConRight>())
    {
        if ((style & StyleSetMask_JoyRightSupportedStyles).IsAnyOn())
        {
            return true;
        }
    }
    else if (deviceType.Test<system::DeviceType::HandheldJoyLeft>())
    {
        if ((style & StyleSetMask_HandheldLeftSupportedStyles).IsAnyOn())
        {
            return true;
        }
    }
    else if (deviceType.Test<system::DeviceType::HandheldJoyRight>())
    {
        if ((style & StyleSetMask_HandheldRightSupportedStyles).IsAnyOn())
        {
            return true;
        }
    }
    else if (deviceType.Test<system::DeviceType::Palma>())
    {
        if ((style & StyleSetMask_PalmaSupportedStyles).IsAnyOn())
        {
            return true;
        }
    }

    return false;
}

bool NpadDeviceManager::IsSupportedNpadStyleSetInitialized() NN_NOEXCEPT
{
    bool isInitialized;
    auto aruid = m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceUserId();
    if (m_pCommonResourceHolder->GetNpadAppletPolicyManager().IsSupportedNpadStyleSetInitialized(&isInitialized, aruid).IsFailure())
    {
        return false;
    }

    return isInitialized;
}

void NpadDeviceManager::GetSystemExtStyle(system::NpadSystemExtMainStyle* pOutMainStyle,
                                          system::NpadSystemExtSubStyle* pOutSubStyle) NN_NOEXCEPT
{
    *pOutMainStyle = m_SystemExtMainStyle;
    *pOutSubStyle = m_SystemExtSubStyle;
}

void NpadDeviceManager::UpdateDeviceType() NN_NOEXCEPT
{
    m_DeviceType = m_pAbstractedPadHolder->GetDeviceType();
}

void NpadDeviceManager::UpdateStyleSet() NN_NOEXCEPT
{
    NpadStyleSet style;
    style.Reset();
    auto joyAssignmentMode = m_pAbstractedPadHolder->GetJoyAssignmentMode();

    // ジョイコン左が接続されている かつ 割り当てモードが1台ずつ
    style.Set<NpadStyleJoyLeft>(
        m_DeviceType.Test<system::NpadDeviceType::JoyConLeft>() == true &&
        joyAssignmentMode == NpadJoyAssignmentMode_Single
        );

    // ジョイコン右が接続されている かつ 割り当てモードが1台ずつ
    style.Set<NpadStyleJoyRight>(
        m_DeviceType.Test<system::NpadDeviceType::JoyConRight>() == true &&
        joyAssignmentMode == NpadJoyAssignmentMode_Single
        );

    // ジョイコン左右いずれかが接続されている かつ 割り当てモードが2本セット
    style.Set<NpadStyleJoyDual>(
        (m_DeviceType & DeviceTypeMask_JoyCons).IsAnyOn() &&
        joyAssignmentMode == NpadJoyAssignmentMode_Dual
        );

    // Palma が接続されている
    style.Set<NpadStylePalma>(
        m_DeviceType.Test<system::NpadDeviceType::Palma>()
        );

    // Handheld はどんな状態でもいったん有効にする
    // プロセスごとに処理する際に、HandheldActivationMode と Handheld の有効/無効に合わせてマスクされる
    style.Set<NpadStyleHandheld>();

    // ジョイコン左右両方が接続されている または フルコンが接続されている
    style.Set<NpadStyleFullKey>(
        (m_DeviceType & DeviceTypeMask_JoyCons) == DeviceTypeMask_JoyCons ||
        m_DeviceType.Test<system::NpadDeviceType::FullKeyController>() == true
        );

    // Gc コントローラーが接続されている。暫定的に DeviceType ではなく AbstractedType で判定
    if (m_DeviceType.Test<system::NpadDeviceType::FullKeyController>() == true)
    {
        IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
        auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));
        for (int i = 0; i < count; i++)
        {
            if (pPads[i]->GetType() == AbstractedPadType_GcAdapter)
            {
                style.Set<NpadStyleGc>();
                break;
            }
        }
    }

    // ひとつでも接続されていたら有効になる操作形態
    style.Set<system::NpadStyleSystem>(m_DeviceType.IsAnyOn());
    style.Set<system::NpadStyleSystemExt>(m_DeviceType.IsAnyOn());

    m_StyleSet = style;
}

void NpadDeviceManager::UpdateSystemExtStyle() NN_NOEXCEPT
{
    // デフォルトのスタイル
    m_SystemExtMainStyle = (m_NpadId == NpadId::Handheld) ? system::NpadSystemExtMainStyle_HandheldNoJoyCon : system::NpadSystemExtMainStyle_None;
    m_SystemExtSubStyle = system::NpadSystemExtSubStyle_Default;
    auto joyAssignmentMode = m_pAbstractedPadHolder->GetJoyAssignmentMode();

    // デバイスの種類 から NpadSystemExtMainStlye を導く
    if (m_DeviceType.Test<system::NpadDeviceType::JoyConLeft>() || m_DeviceType.Test<system::NpadDeviceType::JoyConRight>())
    {
        // 未知のデバイスのエミュレーション機能が有効な場合は、ProController 互換コントローラーとして返す
        if (::nn::hid::detail::FirmwareDebugSettings::EmulateFutureDevice())
        {
            m_SystemExtMainStyle = system::NpadSystemExtMainStyle_CompatibleJoyCon;
        }
        else
        {
            if (m_DeviceType.Test<system::NpadDeviceType::JoyConLeft>() && m_DeviceType.Test<system::NpadDeviceType::JoyConRight>())
            {
                m_SystemExtMainStyle = system::NpadSystemExtMainStyle_JoyDual;
            }
            else if (m_DeviceType.Test<system::NpadDeviceType::JoyConLeft>())
            {
                if (joyAssignmentMode == NpadJoyAssignmentMode_Dual)
                {
                    m_SystemExtMainStyle = system::NpadSystemExtMainStyle_JoyDualLeftOnly;
                }
                else
                {
                    m_SystemExtMainStyle = (m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadJoyHoldType() == NpadJoyHoldType_Horizontal) ?
                        system::NpadSystemExtMainStyle_JoyLeftHorizontal :
                        system::NpadSystemExtMainStyle_JoyLeftVertical;
                }
            }
            else if (m_DeviceType.Test<system::NpadDeviceType::JoyConRight>())
            {
                if (joyAssignmentMode == NpadJoyAssignmentMode_Dual)
                {
                    m_SystemExtMainStyle = system::NpadSystemExtMainStyle_JoyDualRightOnly;
                }
                else
                {
                    m_SystemExtMainStyle = (m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadJoyHoldType() == NpadJoyHoldType_Horizontal) ?
                        system::NpadSystemExtMainStyle_JoyRightHorizontal :
                        system::NpadSystemExtMainStyle_JoyRightVertical;
                }
            }
        }
    }
    else if (m_DeviceType.Test<system::NpadDeviceType::FullKeyController>())
    {
        // 未知のデバイスのエミュレーション機能が有効な場合は、ProController 互換コントローラーとして返す
        if (::nn::hid::detail::FirmwareDebugSettings::EmulateFutureDevice())
        {
            m_SystemExtMainStyle = system::NpadSystemExtMainStyle_CompatibleProController;
        }
        else
        {
            m_SystemExtMainStyle = system::NpadSystemExtMainStyle_SwitchProController;
        }
    }
    if (m_DeviceType.Test<system::NpadDeviceType::HandheldJoyLeft>() && m_DeviceType.Test<system::NpadDeviceType::HandheldJoyRight>())
    {
        m_SystemExtMainStyle = system::NpadSystemExtMainStyle_HandheldJoyCon;
    }
    else if (m_DeviceType.Test<system::NpadDeviceType::HandheldJoyLeft>())
    {
        m_SystemExtMainStyle = system::NpadSystemExtMainStyle_HandheldLeftJoyOnly;
    }
    else if (m_DeviceType.Test<system::NpadDeviceType::HandheldJoyRight>())
    {
        m_SystemExtMainStyle = system::NpadSystemExtMainStyle_HandheldRightJoyOnly;
    }
    else if (m_DeviceType.Test<system::NpadDeviceType::Palma>())
    {
        m_SystemExtMainStyle = system::NpadSystemExtMainStyle_Unknown;
    }
}

void NpadDeviceManager::UpdateButtonOrient() NN_NOEXCEPT
{
    // 横持ちの場合のみSLSR表記は可能
    m_IsSlSrOriented = (m_StyleSet.Test<NpadStyleJoyLeft>() || m_StyleSet.Test<NpadStyleJoyRight>()) &&
                       (m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadJoyHoldType() == NpadJoyHoldType_Horizontal);


    // 左コン操作もしくは、右コン横持の時に限り クローバー表記とする
    m_IsAbxyOriented = !(
        m_StyleSet.Test<NpadStyleJoyLeft>() ||
        (m_StyleSet.Test<NpadStyleJoyRight>() && m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadJoyHoldType() == NpadJoyHoldType_Horizontal)
        );

    // + ボタンがあるかどうか
    m_IsPlusAvailable = m_DeviceType.Test<system::DeviceType::FullKeyController>() || (m_DeviceType & DeviceTypeMask_JoyRights).IsAnyOn();

    // - ボタンがあるかどうか
    m_IsMinusAvailable = m_DeviceType.Test<system::DeviceType::FullKeyController>() || (m_DeviceType & DeviceTypeMask_JoyLefts).IsAnyOn();

    // 未知のデバイスのエミュレーション機能が有効な場合かつ DeviceTypeが Joy-Con の場合は強制横持ち相当かつ、プラスマイナス両方を備えないデバイスとして見せる
    if (::nn::hid::detail::FirmwareDebugSettings::EmulateFutureDevice())
    {
        if (m_DeviceType.Test<system::NpadDeviceType::JoyConLeft>() ||
            m_DeviceType.Test<system::NpadDeviceType::JoyConRight>())
        {
            m_IsAbxyOriented = false;
            m_IsSlSrOriented = true;
            m_IsPlusAvailable = false;
            m_IsMinusAvailable = false;
        }
    }
}

void NpadDeviceManager::UpdateControllerColor() NN_NOEXCEPT
{
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    if (count == 0)
    {
        return;
    }

    m_FullKeyColor.attribute = ColorAttribute_NoController;
    m_FullKeyColor.fullKey = NoColor;
    m_JoyColor.attribute = ColorAttribute_NoController;
    m_JoyColor.left = NoColor;
    m_JoyColor.right = NoColor;

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected())
        {
            auto color = pPads[i]->GetColor();
            auto deviceType = pPads[i]->GetDeviceType();
            if (deviceType.Test<system::DeviceType::FullKeyController>())
            {
                m_FullKeyColor.attribute = ColorAttribute_Ok;
                m_FullKeyColor.fullKey = ConvertControllerColor(color);
            }

            if ((deviceType & DeviceTypeMask_JoyLefts).IsAnyOn())
            {
                m_JoyColor.attribute = ColorAttribute_Ok;
                m_JoyColor.left = ConvertControllerColor(color);
            }

            if ((deviceType & DeviceTypeMask_JoyRights).IsAnyOn())
            {
                m_JoyColor.attribute = ColorAttribute_Ok;
                m_JoyColor.right = ConvertControllerColor(color);
            }
        }
    }

    if (m_DeviceType.Test<system::DeviceType::FullKeyController>() == false)
    {
        // 左コンの色情報が優先
        if ((m_DeviceType & DeviceTypeMask_JoyLefts).IsAnyOn())
        {
            m_FullKeyColor.attribute = m_JoyColor.attribute;
            m_FullKeyColor.fullKey = m_JoyColor.left;
        }
        else if ((m_DeviceType & DeviceTypeMask_JoyRights).IsAnyOn())
        {
            m_FullKeyColor.attribute = m_JoyColor.attribute;
            m_FullKeyColor.fullKey = m_JoyColor.right;
        }
    }
}

void NpadDeviceManager::ProcessSharedMemory(
    void (*processor)(NpadDeviceManager* that,
                      NpadSharedMemoryEntry* address,
                      ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    const auto index =
        static_cast<size_t>(GetIndexFromNpadIdType(m_NpadId));

    for (const AppletResourceEntry& entry :
             m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->npad.entries)>::value);
            NN_SDK_ASSERT_NOT_NULL(&entry.address->npad.entries[index]);
            processor(this,
                &entry.address->npad.entries[index],
                entry.aruid);
        }
    }
}

void NpadDeviceManager::ProcessSharedMemoryForTargetAruid(
    ::nn::applet::AppletResourceUserId aruid,
    void (*processor)(NpadDeviceManager* that,
                      NpadSharedMemoryEntry* address,
                      ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    const auto index =
        static_cast<size_t>(GetIndexFromNpadIdType(m_NpadId));

    for (const AppletResourceEntry& entry :
             m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.aruid == aruid)
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->npad.entries)>::value);
            NN_SDK_ASSERT_NOT_NULL(&entry.address->npad.entries[index]);
            processor(this,
                &entry.address->npad.entries[index],
                entry.aruid);

            return;
        }
    }
}

void NpadDeviceManager::UpdateSharedMemoryForControlStates(::nn::applet::AppletResourceUserId aruid,
                                                           NpadSharedMemoryEntry* address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);
    NpadInternalState& lifo = ::nn::util::Get(address->internalState);

    // デバイスタイプの更新
    auto deviceTypeToWrite = m_DeviceType;
    if (::nn::hid::detail::FirmwareDebugSettings::EmulateFutureDevice())
    {
        // 未知のデバイスのエミュレーション機能が有効な場合は、FullKey または Joy-Con を Unknown に置き換える
        if (m_DeviceType.Test<system::NpadDeviceType::JoyConLeft>() ||
            m_DeviceType.Test<system::NpadDeviceType::JoyConRight>() ||
            m_DeviceType.Test<system::NpadDeviceType::FullKeyController>())
        {
            deviceTypeToWrite.Reset();
            deviceTypeToWrite.Set<system::NpadDeviceType::Unknown>();
        }
    }
    lifo.SetNpadDeviceTypeSet(deviceTypeToWrite);

    // 操作形態を更新
    auto oldStyle = lifo.GetNpadStyleSet();
    auto style = GetAvailableNpadStyleSet(aruid);
    lifo.SetNpadStyleSet(style);
    if (oldStyle != style)
    {
        m_pCommonResourceHolder->GetNpadAppletPolicyManager().SignalStyleUpdateEvent(aruid, m_NpadId);
    }

    // UI の表示関連情報を更新
    lifo.SetButtonOriented(m_IsAbxyOriented, m_IsSlSrOriented);
    lifo.SetIsPlusAvailable(m_IsPlusAvailable);
    lifo.SetIsMinusAvailable(m_IsMinusAvailable);

    lifo.SetNpadFullKeyColor(m_FullKeyColor);
    lifo.SetNpadJoyColor(m_JoyColor);
    lifo.SetNpadJoyAssignmentMode(m_pAbstractedPadHolder->GetJoyAssignmentMode());
}

void NpadDeviceManager::UpdateControllerUsageForPlayReport() NN_NOEXCEPT
{
    system::PlayReportControllerUsage usages[system::PlayReportControllerUsageCountMax];
    int playReportCount = 0;

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    auto style = this->GetAvailableNpadStyleSet(m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId());
    auto holdType = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadJoyHoldType();

    for (int i = 0; i < count; i++)
    {
        auto& playReportUsage = usages[playReportCount];
        playReportUsage = system::PlayReportControllerUsage();

        // デバイス毎の情報を取得
        if (pPads[i]->SetDeviceInfoOnPlayReportControllerUsage(&playReportUsage) == false)
        {
            // プレイレポート用の情報の呼び出しに失敗したので時間をあけてリトライする
            m_IsPlayReportUpdateRequired = true;
            return;
        }

        // コントローラー番号の設定
        playReportUsage.controllerNumber = ConvertPlayReportControllerNumber(m_NpadId);

        // 操作形態の設定
        if (style.Test<NpadStyleJoyDual>())
            {
            playReportUsage.style = system::PlayReportPlayStyle_JoyConDual;
        }
        else if (style.Test<NpadStyleJoyLeft>())
        {
            playReportUsage.style = (holdType == NpadJoyHoldType_Horizontal) ?
                system::PlayReportPlayStyle_JoyConLeftHorizontal :
                system::PlayReportPlayStyle_JoyConLeftVertical;
        }
        else if (style.Test<NpadStyleJoyRight>())
        {
            playReportUsage.style = (holdType == NpadJoyHoldType_Horizontal) ?
                system::PlayReportPlayStyle_JoyConRightHorizontal :
                system::PlayReportPlayStyle_JoyConRightVertical;
        }
        else if (style.Test<NpadStyleFullKey>())
        {
            playReportUsage.style = system::PlayReportPlayStyle_SwitchProController;
        }
        else if (style.Test<NpadStyleHandheld>())
        {
            playReportUsage.style = system::PlayReportPlayStyle_Handheld;
        }
        else
        {
            playReportUsage.style = system::PlayReportPlayStyle_Unknown;
        }

        playReportCount++;
    }

    m_pCommonResourceHolder->GetPlayReportManager().UpdateDeviceInUse(ConvertPlayReportControllerNumber(m_NpadId), usages, playReportCount);
    m_IsPlayReportUpdateRequired = false;
}

void NpadDeviceManager::CreateNotifyControllerInfo(nn::ovln::format::ControllerInfo* pOutValue) NN_NOEXCEPT
{
    pOutValue->joyHoldType = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadJoyHoldType();

    // インジケーターパターンの取得
    pOutValue->ledPattern = GetIndicatorPattern(m_NpadId, false);

    pOutValue->fullKeyColor.isValid = false;
    pOutValue->joyColor.left.isValid = false;
    pOutValue->joyColor.right.isValid = false;

    // 2本持ちが有効な場合はデバイスタイプは、Left/Right 両方有効にする
    if (m_StyleSet.Test<NpadStyleJoyDual>())
    {
        pOutValue->deviceType = system::NpadDeviceType::JoyConLeft::Mask | system::NpadDeviceType::JoyConRight::Mask;
    }
    // それ以外の場合は現状のデバイスタイプそのまま
    else
    {
        pOutValue->deviceType = m_DeviceType;
    }
    if (pOutValue->deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>() || pOutValue->deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
    {
        if (m_JoyColor.attribute == ColorAttribute_Ok)
        {
            pOutValue->joyColor.left.color = m_JoyColor.left;
            pOutValue->joyColor.left.isValid = pOutValue->deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>();
            pOutValue->joyColor.right.color = m_JoyColor.right;
            pOutValue->joyColor.right.isValid = pOutValue->deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>();
        }
    }
    else
    {
        if (m_FullKeyColor.attribute == ColorAttribute_Ok)
        {
            pOutValue->fullKeyColor.color = m_FullKeyColor.fullKey;
            pOutValue->fullKeyColor.isValid = true;
        }

    }
}

system::InterfaceType NpadDeviceManager::GetInterfaceType() NN_NOEXCEPT
{
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    if (count == 0)
    {
        return system::InterfaceType_Unknown;
    }

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected())
        {
            return pPads[i]->GetInterfaceType();;
        }
    }

    return system::InterfaceType_Unknown;
}

Result NpadDeviceManager::GetGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip) NN_NOEXCEPT
{
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected() && pPads[i]->GetDeviceType().Test<system::DeviceType::SwitchProController>())
        {
            NN_RESULT_THROW_UNLESS(pPads[i]->GetGripColor(pOutLeftGrip, pOutRightGrip), ResultNpadControllerNotConnected());
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultNpadControllerNotConnected());
}

void NpadDeviceManager::CheckForPlayReport() NN_NOEXCEPT
{
    if (m_IsPlayReportUpdateRequired == true)
    {
        UpdateControllerUsageForPlayReport();
    }
}

void NpadDeviceManager::GetInterfaceType(system::InterfaceType* pOutLeftInterface, system::InterfaceType* pOutRightInterface) NN_NOEXCEPT
{
    *pOutLeftInterface  = system::InterfaceType_Unknown;
    *pOutRightInterface = system::InterfaceType_Unknown;

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    if (count == 0)
    {
        return;
    }

    for (int i = 0; i < count; i++)
    {
        if ( pPads[i]->IsConnected() )
        {
            auto deviceType = pPads[i]->GetDeviceType();
            if (deviceType.Test<system::DeviceType::JoyConLeft>() ||
                deviceType.Test<system::DeviceType::HandheldJoyLeft>())
            {
                *pOutLeftInterface = pPads[i]->GetInterfaceType();
            }
            else if (deviceType.Test<system::DeviceType::JoyConRight>() ||
                     deviceType.Test<system::DeviceType::HandheldJoyRight>())
            {
                *pOutRightInterface = pPads[i]->GetInterfaceType();
            }
        }
    }
}

}}} // namespace nn::hid::detail
