﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadJoyCommon.h>

#include "hid_ActivationCount.h"
#include "hid_IAbstractedPad.h"
#include "hid_NpadCommonResourceHolder.h"
#include "hid_NpadLastActiveControllerMonitor.h"
#include "hid_NpadManager.h"

namespace nn { namespace hid { namespace detail {

//!< Npad デバイス割り当ての解決を担うマネージャを扱うクラスです。
class NpadAssignmentManager final
{
    NN_DISALLOW_COPY(NpadAssignmentManager) NN_NOEXCEPT;
    NN_DISALLOW_MOVE(NpadAssignmentManager) NN_NOEXCEPT;

private:
    //!< Npad が共通で扱う hid のサービスを保持するクラス
    NpadCommonResourceHolder* m_pCommonResourceHolder;

    //!< NpadManager の配列
    NpadManagerArray* m_pNpads;

    //!< 最後に操作されたコントローラーの監視を行うクラス
    NpadLastActiveControllerMonitor* m_pLastActiveControllerMonitor;

    //!< 抽象化されたコントローラー
    IAbstractedPad* m_pAbstractedPads[IAbstractedPadCountMax];

    //!< IAbstractedPad の数
    int m_AbstractedPadCount;

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

public:
    NpadAssignmentManager() NN_NOEXCEPT;

    ~NpadAssignmentManager() NN_NOEXCEPT;

    //!< NpadCommonResourceHolder を設定します。
    void SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT;

    //!< NpadManager の配列を設定します。
    void SetNpadManagers(NpadManagerArray* pNpads) NN_NOEXCEPT;

    //!< NpadLastActiveControllerMonitor を設定します。
    void SetNpadLastActiveControllerMonitor(NpadLastActiveControllerMonitor* pMonitor) NN_NOEXCEPT;

    //!< IAbstractedPad をセットします
    void AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< デバイスの解決状態を更新する
    void UpdateDeviceMap() NN_NOEXCEPT;

    //!< 最新のFocus のあたっている Aruid にあわせて、Npad の割り当て状態を更新する
    void UpdateOnAruidSwitch() NN_NOEXCEPT;

    //!< 有効にする NpadStyleSet を設定します。
    void ProceedSupportedNpadStyleSetUpdate() NN_NOEXCEPT;

    //!< 有効にする NpadIdType を設定します。
    void ProceedSupportedNpadIdTypeUpdate() NN_NOEXCEPT;

    //!< 本体機能共通の Npad の動作ポリシーを設定します。
    void ProceedApplyNpadSystemCommonPolicy() NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラー割り当て決定モードを有効にします。
    void ProceedEnableLrAssignmentMode() NN_NOEXCEPT;

    //!< Npad を切断します。
    void Disconnect(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< 指定した Npad に対するジョイコンの割り当て状態を取得します。
    NpadJoyAssignmentMode GetNpadJoyAssignmentMode(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< Npad の割り当てオードを デフォルト設定に基づき1本ずつに変更します。
    void SetNpadJoyAssignmentModeSingleByDefault(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< Npad の割り当てオードを 1本ずつに変更します。
    void SetNpadJoyAssignmentModeSingle(::nn::applet::AppletResourceUserId aruid, NpadIdType id, NpadJoyDeviceType type) NN_NOEXCEPT;

    //!< Npad の割り当てオードを 1本ずつに変更します。
    bool SetNpadJoyAssignmentModeSingleWithDestination(NpadIdType* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id, NpadJoyDeviceType type) NN_NOEXCEPT;

    //!< Npad の割り当てモードを 2本セットに変更します。
    void SetNpadJoyAssignmentModeDual(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< 1本ずつ接続された2つのNpadをマージして 1つの Npad に対して 2本セット
    ::nn::Result MergeSingleJoyAsDualJoy(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT;

    //!< Npad に対するコントローラーの割り当てを入れ替えます
    ::nn::Result SwapNpadAssignment(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT;

    //!< スリープ復帰時に必要な処理を行います
    void Resume() NN_NOEXCEPT;

private:
    //!< 指定された NpadId に対して、切断されたデバイスをチェック
    void CheckForDeviceDetach(NpadIdType id, const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< 切断されたデバイスをチェック
    void CheckForDeviceDetachOnAllNpads(const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< 接続されたデバイスをチェック
    void CheckForDeviceAttach(const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< 割り当て処理の更新
    void UpdateAssignment(const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< LR 割り当てモード中に SL/SR ボタンが押された場合に、1本持ちに変更します
    void HandleSlSrAssignment(const NpadAppletPolicy& policy,
                              const int index) NN_NOEXCEPT;

    //!< LR 割り当てモード中に L/R/ZL/ZR ボタンが押された場合に、2本持ちに変更します
    void HandleLRAssignment(const NpadAppletPolicy& policy,
                            const int index) NN_NOEXCEPT;

    //!< AbstractedPadId から NpadManager を取得する
    Result GetNpadManagerFromAbstractedPadId(NpadManager** pOutManager, const AbstractedPadId& handle) NN_NOEXCEPT;

    //!< コントローラーを指定した Npad に対して割り当てます
    void AddAbstractedPadToFixedNpad(NpadIdType id,
                                     IAbstractedPad* pPad,
                                     bool isPhysicallyConnected,
                                     const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< 非ジョイント状態のコントローラーをあいている Npad に対してデバイスを追加します。
    bool AddAbstractedPadToNewNpad(int* pOutIndex,
                                   IAbstractedPad* pPad,
                                   bool isPhysicallyConnected,
                                   const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< Npad への追加の内部実装
    void AddAbstractedPadToNpadImpl(int index,
                                IAbstractedPad* pPad,
                                bool isPhysicallyConnected) NN_NOEXCEPT;

    //!< 指定した index に対してデバイスの接続が受け入れ可能かどうかを評価する
    bool IsDeviceAcceptable(int index, system::NpadDeviceTypeSet deviceType, const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< NpadIdType に対するデバイスの接続許可を判断したうえで切断する
    void DisconnectUnsupporedNpadIds(const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< NpadStyleSet に対するデバイスの接続を判断したうえで切断する
    void DisconnectUnsupporedNpadStyle(const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< 現在のスタイルに合わせてジョイコンの割り当てモードを更新する
    void UpdateJoyAssignmentMode(const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< Npad のデバイス状態を更新する
    void UpdateNpadAll(const NpadAppletPolicy& policy) NN_NOEXCEPT;

    //!< コントローラーの接続台数チェックを行う
    bool IsNewAbstractedPadConnectionAllowed(AbstractedPadId padId) NN_NOEXCEPT;

    //!< 指定したデバイスを Disconnect する
    void DisconnectImpl(const NpadIdType& id, const system::DeviceTypeSet& deviceType) NN_NOEXCEPT;

    //!< 全てののデバイスを Disconnect する
    void DisconnectAllImpl(const NpadIdType& id) NN_NOEXCEPT;

    //!< 電池切れでのデバイス切断の場合通知を発行する
    void NotifyIfBatteryNoneDetach(NpadManager& manager, const system::DeviceTypeSet& deviceType) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
