﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_Types.h>
#include <nn/os/os_Mutex.h>

#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_HashTable.h"
#include "hid_NpadAppletPolicy.h"
#include "hid_NpadId.h"
#include "hid_NpadRevisionTypes.h"
#include "hid_NpadStyleDefinitions.h"

namespace nn { namespace hid { namespace detail {

//!< Npad アプレットポリシーのエントリを表す構造体です。
struct NpadAppletPolicyEntry final
{
    AppletResourceFlagSet flags;                   //!< 制御フラグ
    ::nn::applet::AppletResourceUserId aruid;      //!< ARUID
    NpadAppletPolicy npadPolicy;                   //!< Npad の動作ポリシー

    //!< Npad のデジタルボタンのマスク
    NpadButtonSet buttonMask[NpadIdCountMax][NpadStyleCountMax];

    //!< 最新の SamplingNumber
    int64_t samplingNumber[NpadIdCountMax][NpadStyleCountMax];

    //!< Gc のアナログトリガー用の最新の SamplingNumber
    int64_t samplingNumberGcTrigger[NpadIdCountMax];

    //!< Npad のリビジョン
    NpadApplicationRevision applicationRevision;
};

//!< アプリケーション毎の Npad の利用ポリシーの管理クラス
class NpadAppletPolicyManager final
{
    NN_DISALLOW_COPY(NpadAppletPolicyManager);
    NN_DISALLOW_MOVE(NpadAppletPolicyManager);

private:
    //!< 現在有効なポリシー
    NpadAppletPolicy m_MasterPolicy;

    //!< アプレットリソースのエントリマップ
    HashTable<
        ::nn::applet::AppletResourceUserId, NpadAppletPolicyEntry*,
        AppletResourceUserIdHasher, AppletResourceEntryCountMax> m_EntryMap;

    //!< アプレットリソースのエントリ
    NpadAppletPolicyEntry m_Entries[AppletResourceEntryCountMax];

    //!< アプレットリソースマネージャのミューテックス
    ::nn::os::Mutex* m_pAppletResourceManagerMutex;

    //!< 現在の Aruid
    ::nn::applet::AppletResourceUserId m_CurrentFocusedAruid;

    //!< アクティブ化された回数
    ActivationCount m_SharedPolicyActivationCount;

    //!< 引き継ぎのためにバックアップされた HoldType
    NpadJoyHoldType m_HoldTypeBackup;

public:
    NpadAppletPolicyManager() NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID を登録します。
    ::nn::Result RegisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID の登録を解除します。
    void UnregisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadAppletPolicy を有効にします
    ::nn::Result ActivateNpadAppletPolicyEntry(
        ::nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT;

    //!< NpadAppletPolicy を無効にします
    void DeactivateNpadAppletPolicyEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 共有 NpadAppletPolicy をアクティブ化します。
    ::nn::Result ActivateSharedNpadAppletPolicy() NN_NOEXCEPT;

    //!< 共有 NpadAppletPolicy を非アクティブ化します。
    ::nn::Result DeactivateSharedNpadAppletPolicy() NN_NOEXCEPT;

    //!< 現在有効なポリシーを取得します
    const NpadAppletPolicy& GetCurrentPolicy() NN_NOEXCEPT {return m_MasterPolicy;}

    //!< 現在フォーカスのあたっている Aruid を取得します
    ::nn::applet::AppletResourceUserId GetFocusedAppletResourceUserId() NN_NOEXCEPT;

    //!< Style の更新処理をフォーカスのあたっているアプレットに対して通知します
    void SignalStlyeUpdateEventToFocusedApplet(NpadIdType id) NN_NOEXCEPT;

    //!< ARUID を設定します。
    void SetFocuesedAppletResourceUserId(::nn::applet::AppletResourceUserId value
                                 ) NN_NOEXCEPT;

    //!< Npad のポリシーを Reset する
    Result Reset(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Npad のシステム共通のポリシーを設定する
    Result SetSystemCommonPolicy(::nn::applet::AppletResourceUserId aruid, bool isFull) NN_NOEXCEPT;

    //!< システム共通のポリシーが有効かどうかを取得します。
    Result IsAnySystemCommonPolicyEnabled(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< システム共通のポリシーが有効かどうかを取得します。
    Result IsSystemCommonPolicyInFullMode(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 有効にする NpadStyleSet を設定します。
    Result SetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet style) NN_NOEXCEPT;

    //!< 有効な NpadStyleSet を取得します。
    Result GetSupportedNpadStyleSet(NpadStyleSet* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプリがビルドされた SDK に応じてマスクされた有効な NpadStyleSet を取得します。
    Result GetMaskedSupportedNpadStyleSet(NpadStyleSet* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 対応する NpadStyleSet が初期化済みかどうかを取得します。
    Result IsSupportedNpadStyleSetInitialized(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ジョイコンの持ち方を設定します。
    Result SetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid, NpadJoyHoldType holdType) NN_NOEXCEPT;

    //!< ジョイコンの持ち方を取得します。
    Result GetNpadJoyHoldType(NpadJoyHoldType* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Handheld 操作形態を有効化するモードを設定します。
    Result SetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid, const NpadHandheldActivationMode& mode) NN_NOEXCEPT;

    //!< Handheld 操作形態を有効化するモードを取得します。
    Result GetNpadHandheldActivationMode(NpadHandheldActivationMode* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 有効にする NpadIdType を設定します。
    Result SetSupportedNpadIdType(::nn::applet::AppletResourceUserId aruid, const NpadIdType* pIds, int count) NN_NOEXCEPT;

    //!< NpadIdType が有効かどうか確認します。
    Result IsNpadIdTypeSupported(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラー割り当て決定モードの有効/無効を設定にします。
    Result SetIsLrAssignmentModeEnabled(::nn::applet::AppletResourceUserId aruid, bool enabled) NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラー割り当て決定モードが有効かどうか取得します。
    Result IsLrAssignmentModeEnabled(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< SlSr ボタンが押されて接続された際に、1本持ちとして接続する機能の有効状態を設定します
    Result SetAssigningSingleOnSlSrPressMode(::nn::applet::AppletResourceUserId aruid, bool enabled) NN_NOEXCEPT;

    //!< SlSr ボタンが押されて接続された際に、1本持ちとして接続する機能の有効状態を取得します
    Result IsAssigningSingleOnSlSrPressEnabled(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadStyleSet が変化したときに受ける通知をイベントオブジェクトにバインドします。
    Result AcquireStyleUpdateEventHandle(::nn::applet::AppletResourceUserId aruid, ::nn::os::NativeHandle* pOutValue, NpadIdType id) NN_NOEXCEPT;

    //!< イベント をシグナルします。
    Result SignalStyleUpdateEvent(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< ホームボタンが有効かどうかを取得します。
    Result IsNpadHomeButtonEnabled(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< ホームボタンの有効/無効を設定します。
    Result SetNpadHomeButtonEnabled(::nn::applet::AppletResourceUserId aruid, NpadIdType id, bool enabled) NN_NOEXCEPT;

    //!< ボタンマスクを設定します。
    Result SetButtonMask(::nn::applet::AppletResourceUserId aruid, NpadIdType id, size_t styleIndex, NpadButtonSet& button) NN_NOEXCEPT;

    //!< ボタンマスクを取得します。
    NpadButtonSet GetButtonMask(::nn::applet::AppletResourceUserId aruid, NpadIdType id, size_t styleIndex) NN_NOEXCEPT;

    NpadButtonSet UpdateButtonMask(::nn::applet::AppletResourceUserId aruid, NpadIdType id, size_t styleIndex, const NpadButtonSet& button, bool enableInput) NN_NOEXCEPT;

    //!< 新しいサンプリング番号を発行します。
    bool PublishSamplingNumber(int64_t* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id, size_t styleIndex) NN_NOEXCEPT;

    //!< 新しいサンプリング番号を発行します。
    bool PublishSamplingNumberForGcTrigger(int64_t* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT;

    //!< Application のリビジョンをセットします。
    void SetNpadApplicationRevision(::nn::applet::AppletResourceUserId aruid, NpadApplicationRevision revision) NN_NOEXCEPT;

    //!< Application のリビジョンを取得します。
    NpadApplicationRevision GetNpadApplicationRevision(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

private:
    //!< aruid で指定あれた Applet の NpadAppletPolicy を取得する
    Result GetNpadAppletPolicy(::nn::applet::AppletResourceUserId aruid, NpadAppletPolicy** pOutPolicy) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
