﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_NpadAppletPolicyManager.h"

namespace nn { namespace hid { namespace detail {

NpadAppletPolicyManager::NpadAppletPolicyManager() NN_NOEXCEPT :
    m_MasterPolicy(),
    m_CurrentFocusedAruid(::nn::applet::AppletResourceUserId::GetInvalidId()),
    m_SharedPolicyActivationCount(),
    m_HoldTypeBackup(NpadJoyHoldType_Vertical)
{
    m_MasterPolicy.Reset();
}

::nn::Result NpadAppletPolicyManager::RegisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_EntryMap.Get(aruid) == nullptr,
        ::nn::hid::ResultAppletResourceDuplicateUserId());

    for (NpadAppletPolicyEntry& entry : m_Entries)
    {
        if (!entry.flags.Test<AppletResourceFlag::IsRegistered>())
        {
            entry.flags.Set<AppletResourceFlag::IsRegistered>();
            entry.aruid = aruid;

            m_EntryMap.Set(aruid, &entry);

            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultAppletResourceTableOverflow());
}

void NpadAppletPolicyManager::UnregisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->DeactivateNpadAppletPolicyEntry(aruid);

    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        NpadAppletPolicyEntry& entry = **ppEntry;

        entry.flags.Reset();
        entry.aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();
        for (auto& numberArray : entry.samplingNumber)
        {
            for (auto& number : numberArray)
            {
                number = 0;
            }
        }
        for (auto& number : entry.samplingNumberGcTrigger)
        {
            number = 0;
        }
        m_EntryMap.Remove(aruid);
    }
}

::nn::Result NpadAppletPolicyManager::ActivateNpadAppletPolicyEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        NpadAppletPolicyEntry& entry = **ppEntry;

        NN_RESULT_THROW_UNLESS(
            !entry.flags.Test<AppletResourceFlag::IsAvailable>(),
            ::nn::hid::ResultAppletResourceDuplicateUserId());

        entry.flags.Set<AppletResourceFlag::IsAvailable>();
        entry.npadPolicy.Activate();
        entry.npadPolicy.Reset();
        entry.applicationRevision = NpadApplicationRevision_0;
        for (auto& buttonMaskPerId : entry.buttonMask)
        {
            for (auto& buttonMask : buttonMaskPerId)
            {
                buttonMask.Reset();
            }
        }

        // 先にフォーカスされている場合は、HoldType 以外の設定をマスターにコピー
        if (m_CurrentFocusedAruid == aruid)
        {
            // HoldType をコピー
            auto holdType = m_MasterPolicy.GetNpadJoyHoldType();
            m_MasterPolicy = entry.npadPolicy;
            // HoldType はまえのものを引き継ぐ
            m_MasterPolicy.SetNpadJoyHoldType(holdType);


        }
    }

    NN_RESULT_SUCCESS;
}

void NpadAppletPolicyManager::DeactivateNpadAppletPolicyEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        NpadAppletPolicyEntry& entry = **ppEntry;

        if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            entry.flags.Reset<AppletResourceFlag::IsAvailable>();
            entry.npadPolicy.Deactivate();
        }
    }
}

::nn::Result NpadAppletPolicyManager::ActivateSharedNpadAppletPolicy() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_SharedPolicyActivationCount.IsMax(),
                           ResultAppletResourceActivationUpperLimitOver());

    if (m_SharedPolicyActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        const auto aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();

        NN_RESULT_DO(this->RegisterAppletResourceUserId(aruid));

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                this->UnregisterAppletResourceUserId(aruid);
            }
        };

        NN_RESULT_DO(this->ActivateNpadAppletPolicyEntry(aruid));

        needsRollback = false;
    }

    ++m_SharedPolicyActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadAppletPolicyManager::DeactivateSharedNpadAppletPolicy() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_SharedPolicyActivationCount.IsZero(),
                           ResultAppletResourceDeactivationLowerLimitOver());

    --m_SharedPolicyActivationCount;

    if (m_SharedPolicyActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        const auto aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();

        this->UnregisterAppletResourceUserId(aruid);
    }

    NN_RESULT_SUCCESS;
}

::nn::applet::AppletResourceUserId NpadAppletPolicyManager::GetFocusedAppletResourceUserId() NN_NOEXCEPT
{
    return m_CurrentFocusedAruid;
}

void NpadAppletPolicyManager::SignalStlyeUpdateEventToFocusedApplet(NpadIdType id) NN_NOEXCEPT
{
    m_MasterPolicy.SignalStyleUpdateEvent(id);
}

void NpadAppletPolicyManager::SetFocuesedAppletResourceUserId(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (m_CurrentFocusedAruid == aruid)
    {
        // 既にフォーカスがあたっている場合は何もしない
        return;
    }

    m_CurrentFocusedAruid = aruid;
    m_HoldTypeBackup = m_MasterPolicy.GetNpadJoyHoldType();
    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);
    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        NpadAppletPolicyEntry& entry = **ppEntry;
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.flags.Test<AppletResourceFlag::IsRegistered>())
        {
            // 本体機能共通のポリシーの場合のみジョイコンの持ち方設定のみ引き継ぎを行う
            if (entry.npadPolicy.IsAnySystemCommonPolicyEnabled())
            {
                entry.npadPolicy.SetNpadJoyHoldType(m_HoldTypeBackup);
            }

            // フォーカスされたアプレットの設定をマスターにコピー
            m_MasterPolicy = entry.npadPolicy;

            // ジョイコンの持ち方設定が未設定のときは、持ち方設定を現在の設定に反映したまま
            if (entry.npadPolicy.IsNpadJoyHoldTypeInitialized() == false)
            {
                m_MasterPolicy.SetNpadJoyHoldType(m_HoldTypeBackup);
            }
        }
    }
}

Result NpadAppletPolicyManager::Reset(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->Reset();

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        m_MasterPolicy.Reset();
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetSystemCommonPolicy(::nn::applet::AppletResourceUserId aruid, bool isFull) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->SetSystemCommonPolicy(isFull);
    // 本体機能共通ポリシーが設定された場合は、バックアップされた HoldType を用いる
    pPolicy->SetNpadJoyHoldType(m_HoldTypeBackup);

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        m_MasterPolicy.SetSystemCommonPolicy(isFull);
    // 本体機能共通ポリシーが設定された場合は、バックアップされた HoldType を用いる
        m_MasterPolicy.SetNpadJoyHoldType(m_HoldTypeBackup);
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::IsAnySystemCommonPolicyEnabled(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    *pOutValue = pPolicy->IsAnySystemCommonPolicyEnabled();
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::IsSystemCommonPolicyInFullMode(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    *pOutValue = pPolicy->IsSystemCommonPolicyInFullMode();
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet style) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->SetSupportedNpadStyleSet(style);

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        m_MasterPolicy.SetSupportedNpadStyleSet(style);
        // 持ち方も Master に反映
        m_MasterPolicy.SetNpadJoyHoldType(pPolicy->GetNpadJoyHoldType());
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::GetSupportedNpadStyleSet(NpadStyleSet* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    NN_RESULT_THROW_UNLESS(
        pPolicy->IsSupportedNpadStyleSetInitialized(),
        ResultNpadStyleSetNotInitialized()
    );

    // NpadStyleSet が初期化済みの時だけ正しい値を返す
    *pOutValue = pPolicy->GetSupportedNpadStyleSet();

    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::GetMaskedSupportedNpadStyleSet(NpadStyleSet* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_DO(this->GetSupportedNpadStyleSet(pOutValue, aruid));

    // 有効な Aruid でない場合は、本体機能共通のスタイルを返す
    if (aruid == ::nn::applet::AppletResourceUserId::GetInvalidId())
    {
        *pOutValue = StyleSetMask_SystemCommonStyles;
        NN_RESULT_SUCCESS;
    }

    // バージョンごとにマスクを設定
    NpadStyleSet mask;
    mask.Reset();

    // 全ての SDK バージョン共通のマスク
    mask |= NpadStyleFullKey::Mask;
    mask |= NpadStyleHandheld::Mask;
    mask |= NpadStyleJoyDual::Mask;
    mask |= NpadStyleJoyLeft::Mask;
    mask |= NpadStyleJoyRight::Mask;
    mask |= system::NpadStyleSystem::Mask;
    mask |= system::NpadStyleSystemExt::Mask;

    switch (this->GetNpadApplicationRevision(aruid))
    {
    case NpadApplicationRevision_0:
        // 何も追加しない
        break;
    case NpadApplicationRevision_1:
    default:
        // Palma, Gc コンの対応は 5系から
        mask |= NpadStyleGc::Mask;
        mask |= NpadStylePalma::Mask;
        break;
    }

    // マスクする
    *pOutValue &= mask;

    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::IsSupportedNpadStyleSetInitialized(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    *pOutValue = pPolicy->IsSupportedNpadStyleSetInitialized();
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid, NpadJoyHoldType holdType) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->SetNpadJoyHoldType(holdType);

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        m_MasterPolicy.SetNpadJoyHoldType(holdType);
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::GetNpadJoyHoldType(NpadJoyHoldType* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    if (pPolicy->IsAnySystemCommonPolicyEnabled() == true)
    {
        // SystemCommonPolicy が有効な場合は常時マスターの持ち方を返す
        *pOutValue = m_MasterPolicy.GetNpadJoyHoldType();
    }
    else
    {
        *pOutValue = pPolicy->GetNpadJoyHoldType();
    }

    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid, const NpadHandheldActivationMode& mode) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->SetNpadHandheldActivationMode(mode);

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        m_MasterPolicy.SetNpadHandheldActivationMode(mode);
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::GetNpadHandheldActivationMode(NpadHandheldActivationMode* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    *pOutValue = pPolicy->GetNpadHandheldActivationMode();
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetSupportedNpadIdType(::nn::applet::AppletResourceUserId aruid, const NpadIdType* pIds, int count) NN_NOEXCEPT
{
    if (count > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(pIds);
    }

    NN_RESULT_THROW_UNLESS(count <= NpadIdCountMax, ResultNpadInvalidNpadIdTypeCount());

    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    NN_RESULT_DO(pPolicy->SetSupportedNpadIdType(pIds, count));

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        NN_RESULT_DO(m_MasterPolicy.SetSupportedNpadIdType(pIds, count));
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::IsNpadIdTypeSupported(bool* pOutValue,  ::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    *pOutValue = pPolicy->IsNpadIdTypeSupported(id);
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetIsLrAssignmentModeEnabled(::nn::applet::AppletResourceUserId aruid, bool enabled) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->SetIsLrAssignmentModeEnabled(enabled);

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        m_MasterPolicy.SetIsLrAssignmentModeEnabled(enabled);
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::IsLrAssignmentModeEnabled(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    *pOutValue = pPolicy->IsLrAssignmentModeEnabled();
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetAssigningSingleOnSlSrPressMode(::nn::applet::AppletResourceUserId aruid, bool enabled) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->SetAssigningSingleOnSlSrPressMode(enabled);

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        m_MasterPolicy.SetAssigningSingleOnSlSrPressMode(enabled);
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::IsAssigningSingleOnSlSrPressEnabled(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    *pOutValue = pPolicy->IsAssigningSingleOnSlSrPressEnabled();
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::AcquireStyleUpdateEventHandle(::nn::applet::AppletResourceUserId aruid, ::nn::os::NativeHandle* pOutValue, NpadIdType id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    NN_RESULT_DO(pPolicy->AcquireStyleUpdateEventHandle(pOutValue, id));
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SignalStyleUpdateEvent(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->SignalStyleUpdateEvent(id);
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::GetNpadAppletPolicy(
    ::nn::applet::AppletResourceUserId aruid,
    NpadAppletPolicy** pOutPolicy) NN_NOEXCEPT
{
    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        NpadAppletPolicyEntry& entry = **ppEntry;

        *pOutPolicy = &(entry.npadPolicy);
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(ResultNpadNotInitialized());
}

Result NpadAppletPolicyManager::IsNpadHomeButtonEnabled(bool* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    *pOutValue = pPolicy->IsHomeButtonEnabled(id);
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetNpadHomeButtonEnabled(::nn::applet::AppletResourceUserId aruid, NpadIdType id, bool enabled) NN_NOEXCEPT
{
    NpadAppletPolicy* pPolicy;
    NN_RESULT_DO(GetNpadAppletPolicy(aruid, &pPolicy));

    pPolicy->SetHomeButtonEnabled(enabled, id);

    // フォーカス中の Aruid の場合は、現在の設定を更新
    if (m_CurrentFocusedAruid == aruid)
    {
        m_MasterPolicy.SetHomeButtonEnabled(enabled, id);
    }
    NN_RESULT_SUCCESS;
}

Result NpadAppletPolicyManager::SetButtonMask(::nn::applet::AppletResourceUserId aruid, NpadIdType id, size_t styleIndex, NpadButtonSet& button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(styleIndex, NN_ARRAY_SIZE(m_Entries[0].buttonMask[0]));

    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        NpadAppletPolicyEntry& entry = **ppEntry;

        entry.buttonMask[GetIndexFromNpadIdType(id)][styleIndex] = button;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(ResultNpadNotInitialized());
}

NpadButtonSet NpadAppletPolicyManager::GetButtonMask(::nn::applet::AppletResourceUserId aruid, NpadIdType id, size_t styleIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(styleIndex, NN_ARRAY_SIZE(m_Entries[0].buttonMask[0]));

    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        NpadAppletPolicyEntry& entry = **ppEntry;

        return entry.buttonMask[GetIndexFromNpadIdType(id)][styleIndex];
    }

    NpadButtonSet returnValue;
    returnValue.Reset();

    return returnValue;
}

NpadButtonSet NpadAppletPolicyManager::UpdateButtonMask(::nn::applet::AppletResourceUserId aruid,
                                                        NpadIdType id,
                                                        size_t styleIndex,
                                                        const NpadButtonSet& button,
                                                        bool enableInput) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(styleIndex, NN_ARRAY_SIZE(m_Entries[0].buttonMask[0]));

    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        NpadAppletPolicyEntry& entry = **ppEntry;

        const int npadId = GetIndexFromNpadIdType(id);

        if(enableInput)
        {
            entry.buttonMask[npadId][styleIndex] |= ~button;
        }
        else
        {
            entry.buttonMask[npadId][styleIndex].Reset();
        }

        return entry.buttonMask[npadId][styleIndex] & NpadStyleDefinitions[styleIndex].buttonMask;
    }

    NpadButtonSet returnValue;
    returnValue.Reset();

    return returnValue;
}

bool NpadAppletPolicyManager::PublishSamplingNumber(int64_t* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id, size_t styleIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(styleIndex, NN_ARRAY_SIZE(m_Entries[0].samplingNumber[0]));

    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NpadAppletPolicyEntry& entry = **ppEntry;
        const int npadId = GetIndexFromNpadIdType(id);

        *pOutValue = ++entry.samplingNumber[npadId][styleIndex];
        return true;
    }

    return false;
}

bool NpadAppletPolicyManager::PublishSamplingNumberForGcTrigger(int64_t* pOutValue, ::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NpadAppletPolicyEntry& entry = **ppEntry;
        const int npadId = GetIndexFromNpadIdType(id);

        *pOutValue = ++entry.samplingNumberGcTrigger[npadId];
        return true;
    }

    return false;
}

void NpadAppletPolicyManager::SetNpadApplicationRevision(::nn::applet::AppletResourceUserId aruid, NpadApplicationRevision revision) NN_NOEXCEPT
{
    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NpadAppletPolicyEntry& entry = **ppEntry;
        entry.applicationRevision = revision;
    }
}

NpadApplicationRevision NpadAppletPolicyManager::GetNpadApplicationRevision(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadAppletPolicyEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NpadAppletPolicyEntry& entry = **ppEntry;
        return entry.applicationRevision;
    }

    return NpadApplicationRevision_0;
}

}}} // namespace nn::hid::detail
