﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_Types.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/xcd/xcd_Device.h>

#include "hid_ActivationCount.h"
#include "hid_NpadId.h"

namespace nn { namespace hid { namespace detail {

//!< アプリケーション毎の Npad の利用ポリシー
struct NpadAppletPolicy
{
    //!< NpadStyle がセットされたかどうか
    bool m_IsNpadStyleInitialized;

    //!< アプリが有効にしている NpadStyleSet
    NpadStyleSet m_SupportedStyles;

    //!< ジョイコンの持ち方が初期化されているかどうか
    bool m_IsNpadJoyHoldTypeInitialized;

    //!< アプリが有効にしているジョイコンの持ち方
    NpadJoyHoldType m_JoyHoldType;

    //!< アプリが有効にしている Handheld 操作形態を有効化するモード
    NpadHandheldActivationMode m_HandheldActivationMode;

    //!< アプリが有効にしている NpadId
    NpadIdType m_SupportedId[NpadIdCountMax];

    //!< 有効な NpadId の数
    int m_SupportedIdCount;

    //!< LR 割り当てモードの有効状態
    bool m_IsLrAssignmentModeEnabled;

    //!< SlSr ボタンが押されて接続された際に、1本持ちとして接続する機能の有効状態
    bool m_IsAssigningSingleOnSlSrPressEnabled;

    //!< 本体機能共通ポリシーの有効状態
    bool m_IsSystemCommonPolicyEnabled;

    //!< 最大数接続の本体機能共通ポリシーの有効状態
    bool m_IsSystemCommonPolicyFullEnabled;

    //!< ホームボタンが有効かどうか
    bool m_IsHomeButtonEnabled[NpadIdCountMax];

    //!< Style に変更があった際に通知するためのシステムイベント
    ::nn::os::SystemEventType m_StyleUpdateEvents[NpadIdCountMax];

    //!< アプレットのポリシーを有効にする
    void Activate() NN_NOEXCEPT;

    //!< アプレットのポリシーを無効にする
    void Deactivate() NN_NOEXCEPT;

    //!< Npad のポリシーを Reset する
    void Reset() NN_NOEXCEPT;

    //!< Npad のシステム共通のポリシーを設定する
    void SetSystemCommonPolicy(bool isFull) NN_NOEXCEPT;

    //!< システム共通のポリシーが有効かどうかを取得します。
    bool IsAnySystemCommonPolicyEnabled() const NN_NOEXCEPT {return m_IsSystemCommonPolicyEnabled || m_IsSystemCommonPolicyFullEnabled;}

    //!< 最大数接続のシステム共通のポリシーが有効かどうかを取得します。
    bool IsSystemCommonPolicyInFullMode() const NN_NOEXCEPT {return m_IsSystemCommonPolicyFullEnabled;}

    //!< NpadStyleSet が初期化されているかどうか
    bool IsSupportedNpadStyleSetInitialized() NN_NOEXCEPT {return m_IsNpadStyleInitialized;}

    //!< 有効にする NpadStyleSet を設定します。
    void SetSupportedNpadStyleSet(NpadStyleSet style) NN_NOEXCEPT;

    //!< 有効な NpadStyleSet を取得します。
    const NpadStyleSet& GetSupportedNpadStyleSet() const NN_NOEXCEPT {return m_SupportedStyles;}

    //!< ジョイコンの持ち方を設定します。
    void SetNpadJoyHoldType(NpadJoyHoldType holdType) NN_NOEXCEPT;

    //!< ジョイコンの持ち方を取得します。
    NpadJoyHoldType GetNpadJoyHoldType() const NN_NOEXCEPT {return m_JoyHoldType;}

    //!< ジョイコンの持ち方が初期化されているかどうか
    bool IsNpadJoyHoldTypeInitialized() NN_NOEXCEPT {return m_IsNpadJoyHoldTypeInitialized;}

    //!< Handheld 操作形態を有効化するモードを設定します。
    void SetNpadHandheldActivationMode(const NpadHandheldActivationMode& mode) NN_NOEXCEPT {m_HandheldActivationMode = mode;}

    //!< Handheld 操作形態を有効化するモードを取得します。
    NpadHandheldActivationMode GetNpadHandheldActivationMode() const NN_NOEXCEPT {return m_HandheldActivationMode;}

    //!< 有効にする NpadIdType を設定します。
    Result SetSupportedNpadIdType(const NpadIdType* pIds, int count) NN_NOEXCEPT;

    //!< NpadIdType が有効かどうか確認します。
    bool IsNpadIdTypeSupported(NpadIdType id) const NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラー割り当て決定モードの有効/無効を設定にします。
    void SetIsLrAssignmentModeEnabled(bool enabled) NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラー割り当て決定モードが有効かどうか取得します。
    bool IsLrAssignmentModeEnabled() const NN_NOEXCEPT;

    //!< SlSr ボタンが押されて接続された際に、1本持ちとして接続する機能の有効状態を設定します
    void SetAssigningSingleOnSlSrPressMode(bool enabled) NN_NOEXCEPT;

    //!< SlSr ボタンが押されて接続された際に、1本持ちとして接続する機能の有効状態を取得します
    bool IsAssigningSingleOnSlSrPressEnabled() const NN_NOEXCEPT;

    //!< ホームボタンの有効/無効を設定します。
    void SetHomeButtonEnabled(bool enabled, NpadIdType id) NN_NOEXCEPT;

    //!< ホームボタンが有効かどうか取得します。
    bool IsHomeButtonEnabled(NpadIdType id) const NN_NOEXCEPT;

    //!< NpadStyleSet が変化したときに受ける通知をイベントオブジェクトにバインドします。
    Result AcquireStyleUpdateEventHandle(::nn::os::NativeHandle* pOutValue, NpadIdType id) NN_NOEXCEPT;

    //!< イベント をシグナルします。
    void SignalStyleUpdateEvent(NpadIdType id) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
