﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_IAbstractedPad.h"

namespace nn { namespace hid { namespace detail {

const int AbstractedPadCountMaxPerNpad = 5;

//!< Npad 毎に割り当てられる AbstractedPad を保持するクラスです。
class NpadAbstractedPadHolder final
{
    NN_DISALLOW_COPY(NpadAbstractedPadHolder);
    NN_DISALLOW_MOVE(NpadAbstractedPadHolder);

private:
    //!< Npad に割り当てられている AbstractedPad
    IAbstractedPad* m_pAbstractedPads[AbstractedPadCountMaxPerNpad];

    //!< 割り当てられている AbstaractedPad の数
    int m_AbstractedPadCount;

    //!< ジョイコンの現在の持ち方
    NpadJoyAssignmentMode m_JoyAssignmentMode;

public:
    NpadAbstractedPadHolder() NN_NOEXCEPT;

    //!< Npad に AbstractedPad を追加します
    Result AttachAbstractedPad(IAbstractedPad* pPad) NN_NOEXCEPT;

    //!< Npad に対する AbstractedPad の割り当てを解除します
    void DetachAbstractedPad(const AbstractedPadId& id) NN_NOEXCEPT;

    //!< Npad に対する全ての AbstractedPad の割り当てを解除します
    void DetachAbstractedPadAll() NN_NOEXCEPT;

    //!< 指定されたデバイスタイプについて Npad に対する AbstractedPad の割り当て解除し、AbstractedPad を取得します
    IAbstractedPad* DetachAbstractedPad(const system::DeviceTypeSet& deviceType) NN_NOEXCEPT;

    //!< 切断されている AbstractedPad がないかを確認し、切断されていれば Detach する
    void DetachIfDisconnected() NN_NOEXCEPT;

    //!< アドレスで指定されたデバイスが接続済みかどうか取得します。
    bool IsConnectedDevice(const AbstractedPadId& id) NN_NOEXCEPT;

    //!< AbstractedPad を取得する
    int GetAbstractedPads(IAbstractedPad** pOutPad, int size) const NN_NOEXCEPT;

    //!< ジョイコンの割り当てモードを設定します。
    void SetJoyAssignmentMode(const NpadJoyAssignmentMode& mode) NN_NOEXCEPT;

    //!< ジョイコンの割り当てモードを取得します。
    NpadJoyAssignmentMode GetJoyAssignmentMode() NN_NOEXCEPT;

    //!< デバイスタイプを取得する
    system::DeviceTypeSet GetDeviceType() NN_NOEXCEPT;

private:
    //!< リストから AbstractedPad を取り除いて前づめする
    void PopAbstractedPad(int index) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
