﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceManager.h"
#include "hid_IKeyboardDriver.h"
#include "hid_KeyboardLifo.h"
#include "hid_KeyboardManager.h"
#include "hid_SharedMemoryFormat.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Keyboard の入力状態を計算します。
void CalculateKeyboardState(KeyboardState* pOutValue,
                            const KeyboardState& state,
                            bool enablesInput,
                            KeyboardKeySet& keyMask) NN_NOEXCEPT;

} // namespace

const ::nn::TimeSpan KeyboardManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(8);

KeyboardManager::KeyboardManager() NN_NOEXCEPT
    : m_SamplingNumber(0)
    , m_pTimer(nullptr)
    , m_pDriver(nullptr)
    , m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
    , m_State()
    , m_TempState()
    , m_TempAruid()
    , m_IsAutoPilotEnabled(false)
    , m_AutoPilotState()
{
    for (KeyboardAppletResourceEntry& extraEntry : m_AppletResourceEntries)
    {
        extraEntry = KeyboardAppletResourceEntry();
    }
}

void KeyboardManager::SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(timer);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimer = timer;
}

void KeyboardManager::SetDriver(IKeyboardDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pDriver = pDriver;
}

void KeyboardManager::SetAppletResourceManager(
    AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

void KeyboardManager::SetInputDetectorManager(
    InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pInputDetectorManager = pManager;
    m_pInputDetectorManagerMutex = pMutex;
}

::nn::Result KeyboardManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultKeyboardActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // ドライバをアクティブ化
        NN_RESULT_DO(m_pDriver->Activate());

        // LIFO を初期化
        this->ProcessSharedMemory([] (KeyboardManager* that,
                                      KeyboardSharedMemoryFormat* address,
                                      ::nn::applet::AppletResourceUserId aruid,
                                      bool enablesInput,
                                      KeyboardKeySet& keyMask) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(that);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            NN_UNUSED(keyMask);
            ::nn::util::Get(address->lifo).Clear();
        });

        // タイマーイベントをアクティブ化
        m_pTimer->Enable(SamplingInterval);

        m_State = KeyboardState();

        // 入力状態を更新
        this->Update();
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result KeyboardManager::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultKeyboardDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // 自動操作を無効化
        this->DisableAutoPilot();

        // タイマーイベントを非アクティブ化
        m_pTimer->Disable();

        // ドライバを非アクティブ化
        NN_RESULT_DO(m_pDriver->Deactivate());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result KeyboardManager::EnsureAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_TempAruid = aruid;

    this->ProcessSharedMemory([] (KeyboardManager* that,
                                  KeyboardSharedMemoryFormat* address,
                                  ::nn::applet::AppletResourceUserId haystack,
                                  bool enablesInput,
                                  KeyboardKeySet& keyMask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        KeyboardState* pOutValue = &that->m_TempState;

        const KeyboardState& state = that->m_State;

        const ::nn::applet::AppletResourceUserId& needle = that->m_TempAruid;

        KeyboardLifo& lifo = ::nn::util::Get(address->lifo);

        if (lifo.IsEmpty() && needle == haystack)
        {
            CalculateKeyboardState(pOutValue, state, enablesInput, keyMask);

            lifo.Append(*pOutValue);
        }
    });

    NN_RESULT_SUCCESS;
}

::nn::Result KeyboardManager::SetAutoPilotState(
    const ::nn::hid::debug::KeyboardAutoPilotState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultKeyboardNotInitialized());
    m_IsAutoPilotEnabled = true;
    m_AutoPilotState = value;
    NN_RESULT_SUCCESS;
}

::nn::Result KeyboardManager::UnsetAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultKeyboardNotInitialized());
    this->DisableAutoPilot();
    NN_RESULT_SUCCESS;
}

::nn::Result KeyboardManager::SendLockKeyEvent(
    ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultKeyboardNotInitialized());

    NN_RESULT_DO(m_pDriver->SendLockKeyEvent(value));

    NN_RESULT_SUCCESS;
}

void KeyboardManager::Sample() NN_NOEXCEPT
{
    if (!m_ActivationCount.IsZero())
    {
        this->Update();

        this->ProcessSharedMemory([] (
            KeyboardManager* that,
            KeyboardSharedMemoryFormat* address,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            KeyboardKeySet& keyMask) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(aruid);

            KeyboardState* pOutValue = &that->m_TempState;

            const KeyboardState& state = that->m_State;

            CalculateKeyboardState(pOutValue, state, enablesInput, keyMask);

            ::nn::util::Get(address->lifo).Append(*pOutValue);
        });
    }

}

void KeyboardManager::Update() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManagerMutex);

    // 入力変化検知用に状態を保存
    m_TempState = m_State;

    // 入力状態を取得
    m_pDriver->GetState(&m_State);

    // サンプリング番号を設定
    m_State.samplingNumber = m_SamplingNumber++;

    if (m_IsAutoPilotEnabled)
    {
        // 自動操作が有効な場合は自動操作状態をマージ

        m_State.modifiers |= m_AutoPilotState.modifiers;
        m_State.modifiers.Set<KeyboardModifier::CapsLock>(
            m_AutoPilotState.modifiers.Test<KeyboardModifier::CapsLock>());
        m_State.modifiers.Set<KeyboardModifier::ScrollLock>(
            m_AutoPilotState.modifiers.Test<KeyboardModifier::ScrollLock>());
        m_State.modifiers.Set<KeyboardModifier::NumLock>(
            m_AutoPilotState.modifiers.Test<KeyboardModifier::NumLock>());
        m_State.modifiers.Set<KeyboardModifier::Katakana>(
            m_AutoPilotState.modifiers.Test<KeyboardModifier::Katakana>());
        m_State.modifiers.Set<KeyboardModifier::Hiragana>(
            m_AutoPilotState.modifiers.Test<KeyboardModifier::Hiragana>());
        m_State.keys |= m_AutoPilotState.keys;
        m_State.attributes.Set<KeyboardAttribute::IsConnected>(
            m_State.keys.IsAnyOn() ||
            m_AutoPilotState.attributes.Test<KeyboardAttribute::IsConnected>());
    }

    ::std::lock_guard<decltype(*m_pInputDetectorManagerMutex)
                      > locker(*m_pInputDetectorManagerMutex);

    if ((m_State.keys ^ m_TempState.keys).IsAnyOn())
    {
        m_pInputDetectorManager->Notify(
            ::nn::hid::system::InputSourceId::Keyboard::Mask);
    }
}

void KeyboardManager::ProcessSharedMemory(
    void (*processor)(KeyboardManager* that,
                      KeyboardSharedMemoryFormat* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput,
                      KeyboardKeySet& keyMask) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)
                      > locker(*m_pAppletResourceManagerMutex);

    const AppletResourceEntry (&entries)[AppletResourceEntryCountMax] =
        m_pAppletResourceManager->GetAppletResourceEntries();

    for (size_t i = 0; i < AppletResourceEntryCountMax; ++i)
    {
        const AppletResourceEntry& entry = entries[i];

        KeyboardAppletResourceEntry& extraEntry = m_AppletResourceEntries[i];

        if (!entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            extraEntry = KeyboardAppletResourceEntry();
        }
        else
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);

            if (entry.aruid != extraEntry.aruid)
            {
                extraEntry = KeyboardAppletResourceEntry();

                extraEntry.aruid = entry.aruid;
            }

            processor(this,
                      &entry.address->keyboard,
                      entry.aruid,
                      entry.flags.Test<AppletResourceFlag::EnablesInput>(),
                      extraEntry.keyMask);
        }
    }
}

void KeyboardManager::DisableAutoPilot() NN_NOEXCEPT
{
    m_IsAutoPilotEnabled = false;
    m_AutoPilotState = ::nn::hid::debug::KeyboardAutoPilotState();
}

namespace {

void CalculateKeyboardState(KeyboardState* pOutValue,
                            const KeyboardState& state,
                            bool enablesInput,
                            KeyboardKeySet& keyMask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = state;

    if (enablesInput)
    {
        keyMask |= ~state.keys;

        pOutValue->keys &= keyMask;
    }
    else
    {
        keyMask.Reset();

        pOutValue->keys.Reset();
    }

    pOutValue->modifiers.Set<KeyboardModifier::Control>(
        pOutValue->keys.Test<KeyboardKey::LeftControl>() ||
        pOutValue->keys.Test<KeyboardKey::RightControl>());

    pOutValue->modifiers.Set<KeyboardModifier::Shift>(
        pOutValue->keys.Test<KeyboardKey::LeftShift>() ||
        pOutValue->keys.Test<KeyboardKey::RightShift>());

    pOutValue->modifiers.Set<KeyboardModifier::LeftAlt>(
        pOutValue->keys.Test<KeyboardKey::LeftAlt>());

    pOutValue->modifiers.Set<KeyboardModifier::RightAlt>(
        pOutValue->keys.Test<KeyboardKey::RightAlt>());

    pOutValue->modifiers.Set<KeyboardModifier::Gui>(
        pOutValue->keys.Test<KeyboardKey::LeftGui>() ||
        pOutValue->keys.Test<KeyboardKey::RightGui>());
}

} // namespace

}}} // namespace nn::hid::detail
