﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_JoyXpad.h>
#include <nn/hid/debug/hid_JoyXpad.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_JoyXpadImpl.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_Session.h"
#include "hid_StaticObject.h"
#include "hid_XpadId.h"
#include "hid_RingLifoHolder.h"
#include "hid_JoyXpadLifo.h"
#include "hid_XpadSessionManager.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 基本的な構成を持つ Xpad のセッションを扱うためのクラスです。
class JoyXpadSession final : public Session
{
    NN_DISALLOW_COPY(JoyXpadSession);
    NN_DISALLOW_MOVE(JoyXpadSession);

private:
    //!< hid サービスのセッション
    ::nn::sf::SharedPointer<IHidServer> m_Session;

    //!< 基本的な構成を持つ Xpad の ID
    JoyXpadId m_XpadId;

    //!< 基本的な構成を持つ Xpad の LIFO
    RingLifoHolder<JoyXpadLifoPair> m_LifoHolder;

public:
    explicit JoyXpadSession(const JoyXpadId& xpadId) NN_NOEXCEPT
        : m_XpadId(xpadId)
    {
        // 何もしない
    }

    virtual ~JoyXpadSession() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない */ }

    //!< Xpad ID の型
    typedef JoyXpadId XpadIdType;

    //!< 基本的な構成を持つ Xpad の LIFO を返します。
    const JoyXpadLifo* GetRightLifo() const NN_NOEXCEPT
    {
        return &m_LifoHolder.GetLifo()->GetRightLifo();
    }
    const JoyXpadLifo* GetLeftLifo() const NN_NOEXCEPT
    {
        return &m_LifoHolder.GetLifo()->GetLeftLifo();;
    }

protected:
    virtual ::nn::Result ThrowResultActivationUpperLimitOver(
        ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultJoyXpadActivationUpperLimitOver());
    }

    virtual ::nn::Result ThrowResultDeactivationLowerLimitOver(
        ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultJoyXpadDeactivationLowerLimitOver());
    }

    virtual ::nn::Result ThrowResultAllocationFailed() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultJoyXpadLifoAllocationFailed());
    }

    virtual ::nn::Result CreateProxy() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(CreateHidServerProxy(&m_Session));
    }

    virtual void DestroyProxy() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Session.Reset();
    }

    virtual ::nn::Result ActivateService() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(m_Session->ActivateJoyXpad(m_XpadId));
    }

    virtual ::nn::Result DeactivateService() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pProxy;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pProxy));
        NN_RESULT_DO(pProxy->DeactivateJoyXpad(m_XpadId));
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result GetSharedMemoryHandle(::nn::sf::NativeHandle* outValue
                                               ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        NN_RESULT_THROW(m_Session->GetJoyXpadLifoHandle(outValue, m_XpadId));
    }

    virtual bool IsSharedMemoryMapped() NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_LifoHolder.GetLifo() != nullptr);
    }

    virtual void AttachSharedMemory(::nn::os::NativeHandle handle,
                                    bool managed) NN_NOEXCEPT NN_OVERRIDE
    {
        m_LifoHolder.Attach(handle, managed);
    }

    virtual void FinalizeSharedMemory() NN_NOEXCEPT NN_OVERRIDE
    {
        m_LifoHolder.Finalize();
    }
};

XpadSessionManager<JoyXpadSession>& GetXpadSessionManager() NN_NOEXCEPT
{
    return StaticObject<XpadSessionManager<JoyXpadSession>>::Get();
}

} // namespace

::nn::Result GetXpadIds(int* outGotCount, JoyXpadId* outXpadIds, int count
                        ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(outGotCount != nullptr,
                           ResultJoyXpadNullIdCountBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultJoyXpadInvalidIdCount());
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);
    auto gotCount = int64_t();
    NN_RESULT_DO(
        pProxy->GetJoyXpadIds(::nn::sf::Out<int64_t>(&gotCount),
                              ::nn::sf::OutArray<JoyXpadId>(
                                 outXpadIds, static_cast<size_t>(count))));
    *outGotCount = static_cast<int>(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetXpadPlayerNumber(int* outValue, const JoyXpadId& xpadId
                                 ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    *outValue = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_SUCCESS;
}

::nn::Result InitializeXpad(const JoyXpadId& xpadId) NN_NOEXCEPT
{
    JoyXpadSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    NN_RESULT_DO(pSession->Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeXpad(const JoyXpadId& xpadId) NN_NOEXCEPT
{
    JoyXpadSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    NN_RESULT_DO(pSession->Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetJoyXpadRightState(JoyXpadState* outValue, const JoyXpadId& xpadId
                                  ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(outValue != nullptr,
                           ResultJoyXpadNullStateBuffer());
    JoyXpadSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    const JoyXpadLifo* pLifo = pSession->GetRightLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultJoyXpadNotInitialized());
    const int gotCount = pLifo->Read(outValue, 1);
    NN_SDK_REQUIRES_EQUAL(gotCount, 1);
    NN_UNUSED(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetJoyXpadLeftState(JoyXpadState* outValue, const JoyXpadId& xpadId
                                  ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(outValue != nullptr,
                           ResultJoyXpadNullStateBuffer());
    JoyXpadSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    const JoyXpadLifo* pLifo = pSession->GetLeftLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultJoyXpadNotInitialized());
    const int gotCount = pLifo->Read(outValue, 1);
    NN_SDK_REQUIRES_EQUAL(gotCount, 1);
    NN_UNUSED(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetJoyXpadRightStates(int* outGotCount,
                                   JoyXpadState* outStates,
                                   int count,
                                   const JoyXpadId& xpadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outGotCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultJoyXpadNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultJoyXpadInvalidStateCount());
    JoyXpadSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    const JoyXpadLifo* pLifo = pSession->GetRightLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultJoyXpadNotInitialized());
    *outGotCount = pLifo->Read(outStates, count);
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    NN_RESULT_SUCCESS;
}

::nn::Result GetJoyXpadLeftStates(int* outGotCount,
                                   JoyXpadState* outStates,
                                   int count,
                                   const JoyXpadId& xpadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outGotCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultJoyXpadNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultJoyXpadInvalidStateCount());
    JoyXpadSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    const JoyXpadLifo* pLifo = pSession->GetLeftLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultJoyXpadNotInitialized());
    *outGotCount = pLifo->Read(outStates, count);
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    NN_RESULT_SUCCESS;
}

::nn::Result AssertValidJoyXpadId(const JoyXpadId& xpadId) NN_NOEXCEPT
{
    JoyXpadSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    const JoyXpadLifo* pLifo = pSession->GetLeftLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultJoyXpadNotInitialized());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
