﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkText.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/irsensor/irsensor_Result.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_MomentProcessorTypes.h>
#include <nn/irsensor/irsensor_ClusteringProcessorTypes.h>
#include <nn/irsensor/irsensor_PointingProcessorTypesPrivate.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorTypesPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/xcd/xcd.h>
#include <nn/xcd/xcd_Irsensor.h>
#include <nn/xcd/xcd_Device.h>

#include "hid_IrSensorRegister.h"
#include "hid_IrSensorXcdDriver.h"

namespace nn { namespace hid { namespace detail {

const float IrSensorXcdDriver::AmbientNoiseThresholdLow = 0.4f;
const float IrSensorXcdDriver::AmbientNoiseThresholdHigh = 0.6f;

// FunctionLevel ごとに要求される RequiredVersion のテーブル。
// 強制アップデートのためにはここの更新が必要となる。
// プログラム内部での分岐に使うマクロは変更してはいけないので、注意が必要。
const nn::xcd::McuVersionData IrSensorXcdDriver::RequiredVersionTableForFunctionLevel[] =
{
    nn::xcd::FirmwareVersionTera_030b, //!< FunctionLevel 0 (Require 3.11)
    nn::xcd::FirmwareVersionTera_0412, //!< FunctionLevel 1 (Require 4.18)
    nn::xcd::FirmwareVersionTera_0518, //!< FunctionLevel 2 (Require 5.24)
};

namespace {

// コントローラの切断タイムアウト時間(2秒)から少しマージンをとった時間待つ
const auto CommandExecutionTimeout = ::nn::TimeSpan::FromMilliSeconds(2500);
const auto RegisterAccessTimeout = ::nn::TimeSpan::FromMilliSeconds(2500);
const auto SamplingTrialCountMax = 166;
const auto SettingTrialCountMax = 400; // 多数接続時のTSI変更時間を考慮して、暫定的に6秒設定とする
const auto ImageTransferSamplingTrialCountMax = 600; // ImageTransfer の場合のみ320x240時の最大4秒に合わせて多めに設定
const auto HardwareErrorCheckTrialCountMax = 2;
const auto MomentFrameInterval = ::nn::TimeSpan::FromMicroSeconds(14200); // 140.845fps
const auto ClusteringFrameInterval = ::nn::TimeSpan::FromMicroSeconds(14200); //140.845fps
const auto DpdFrameInterval = ::nn::TimeSpan::FromMilliSeconds(5);
const auto TeraPluginFrameInterval = ::nn::TimeSpan::FromMilliSeconds(15);

// 環境ノイズ向けの明るさの閾値
const int AmbientNoiseIntensityThreshold = 200;

class IrWriteRegisterBlockBuilder final
{
    NN_DISALLOW_COPY(IrWriteRegisterBlockBuilder);
    NN_DISALLOW_MOVE(IrWriteRegisterBlockBuilder);

public:
    IrWriteRegisterBlockBuilder(
        ::nn::xcd::IrWriteRegisterBlock* blocks, int countMax) NN_NOEXCEPT
        : m_Blocks(blocks)
        , m_CountMax(countMax)
        , m_Count(0)
    {
        // 何もしない
    }

    ~IrWriteRegisterBlockBuilder() NN_NOEXCEPT
    {
        // 何もしない
    }

    void Append(int bank, uint8_t address, uint8_t value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS(m_Count, m_CountMax);

        auto index = m_Count;
        m_Blocks[index].bankId = static_cast<uint8_t>(bank);
        m_Blocks[index].address = address;
        m_Blocks[index].data = value;
        ++m_Count;
    }

    const ::nn::xcd::IrWriteRegisterBlock* GetBlocks() const NN_NOEXCEPT
    {
        return m_Blocks;
    }

    int GetCount() const NN_NOEXCEPT
    {
        return m_Count;
    }

    int GetCountMax() const NN_NOEXCEPT
    {
        return m_CountMax;
    }

private:
    ::nn::xcd::IrWriteRegisterBlock* m_Blocks;
    int m_CountMax;
    int m_Count;
};

void MakeIrWriteRegisterSetting(
    ::nn::xcd::IrWriteRegisterSetting* pOut,
    const ::nn::xcd::IrWriteRegisterBlock* blocks,
    int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(blocks);
    NN_SDK_REQUIRES_MINMAX(count, 0, ::nn::xcd::IrWriteRegisterCountMax);

    for (auto i = 0; i < count; ++i)
    {
        pOut->registerBlock[i] = blocks[i];
    }
    pOut->registerCount = count;
}

void MakeIrWriteRegisterExSetting(
    ::nn::xcd::IrWriteRegisterSettingEx* pOut,
    const ::nn::xcd::IrWriteRegisterBlock* blocks,
    int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(blocks);
    NN_SDK_REQUIRES_MINMAX(count, 0, ::nn::xcd::IrWriteRegisterExCountMax);

    // FastMode は常に有効とする
    pOut->fastModeFlag = true;
    for (auto i = 0; i < count; ++i)
    {
        pOut->registerBlock[i] = blocks[i];
    }
    pOut->registerCount = count;
}

uint8_t GetIrCameraLightRegisterValue(
    const int8_t& lightTarget,
    bool isAlwaysOn
    ) NN_NOEXCEPT
{
    auto enableSmallAngleLed = false;
    auto enableLargeAngleLed = false;
    switch (lightTarget)
    {
    case ::nn::irsensor::IrCameraLightTarget_AllObjects:
        enableSmallAngleLed = true;
        enableLargeAngleLed = true;
        break;
    case ::nn::irsensor::IrCameraLightTarget_NearObjects:
        enableSmallAngleLed = false;
        enableLargeAngleLed = true;
        break;
    case ::nn::irsensor::IrCameraLightTarget_FarObjects:
        enableSmallAngleLed = true;
        enableLargeAngleLed = false;
        break;
    case ::nn::irsensor::IrCameraLightTarget_None:
        enableSmallAngleLed = false;
        enableLargeAngleLed = false;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    ::nn::util::BitPack8 value;

    value.Clear();
    value.Set<IrSensorRegisterIrCameraLight::DisableSmallAngleLight>(!enableSmallAngleLed);
    value.Set<IrSensorRegisterIrCameraLight::DisableLargeAngleLight>(!enableLargeAngleLed);
    value.Set<IrSensorRegisterIrCameraLight::EnableAlwaysOn>(isAlwaysOn);
    value.Set<IrSensorRegisterIrCameraLight::EnableLightSwitchMode>(false);

    return value.storage;
}

uint8_t GetFrameSubtractionRegisterValue(
    const ::nn::irsensor::PackedIrCameraConfig& config,
    bool isSubtractionEnabled
    ) NN_NOEXCEPT
{
    ::nn::util::BitPack8 value;

    FrameSubtractionMode subtractionMode;
    if (isSubtractionEnabled)
    {
        subtractionMode = FrameSubtractionMode::LedOnOffSubtraction;
    }
    else
    {
        subtractionMode = FrameSubtractionMode::NoSubtraction;
    }

    value.Clear();
    value.Set<IrSensorRegisterFrameSubtraction::FrameSubtractionMode>(
        static_cast<uint8_t>(subtractionMode));
    value.Set<IrSensorRegisterFrameSubtraction::UseNegativeImage>(
        config.isNegativeImageUsed);
    value.Set<IrSensorRegisterFrameSubtraction::FrameDifferenceMode>(
        static_cast<uint8_t>(FrameDifferenceMode::AbsoluteValue));

    return value.storage;
}

uint8_t GetFastUpdateRegisterValue() NN_NOEXCEPT
{
    ::nn::util::BitPack8 value;

    value.Clear();
    value.Set<IrSensorRegisterFastUpdate::Update>(true);

    return value.storage;
}

uint64_t GetExposureTimeRegisterValue(
    const ::nn::irsensor::PackedIrCameraConfig& config
    ) NN_NOEXCEPT
{
    const auto IrSensorClockKiloHertz = 31200;
    return config.exposureTime.GetMicroSeconds() * IrSensorClockKiloHertz / 1000;
}

void AppendRegionOfInterestRegisterBlocks(
    IrWriteRegisterBlockBuilder* pBuilder,
    const ::nn::irsensor::Rect& regionOfInterest
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBuilder);

    // 開始地点は WOI で切った領域と同じにするため、(0, 0)とする
    auto x = 0;
    auto y = 0;
    // レジスタには 1 を減じた値を設定する
    auto width = static_cast<uint16_t>(regionOfInterest.width - 1);
    auto height = static_cast<uint16_t>(regionOfInterest.height - 1);

    // ROI 機能を有効化
    pBuilder->Append(
        IrSensorRegisterEnableRegionOfInterest::Bank,
        IrSensorRegisterEnableRegionOfInterest::Address,
        static_cast<uint8_t>(1 & 0xff));

    pBuilder->Append(
        IrSensorRegisterRegionOfInterestX::Bank,
        IrSensorRegisterRegionOfInterestX::Address,
        static_cast<uint8_t>(x & 0xff));

    pBuilder->Append(
        IrSensorRegisterRegionOfInterestY::Bank,
        IrSensorRegisterRegionOfInterestY::Address,
        static_cast<uint8_t>(y & 0xff));

    pBuilder->Append(
        IrSensorRegisterRegionOfInterestWidth::Bank,
        IrSensorRegisterRegionOfInterestWidth::Address,
        static_cast<uint8_t>(width & 0xff));

    pBuilder->Append(
        IrSensorRegisterRegionOfInterestHeight::Bank,
        IrSensorRegisterRegionOfInterestHeight::Address,
        static_cast<uint8_t>(height & 0xff));

    ::nn::util::BitPack8 pack;

    pack.Clear();
    pack.Set<IrSensorRegisterRegionOfInterestBit8::X8>(
        static_cast<uint8_t>(x >> 8));
    pack.Set<IrSensorRegisterRegionOfInterestBit8::Y8>(
        static_cast<uint8_t>(y >> 8));
    pack.Set<IrSensorRegisterRegionOfInterestBit8::Width8>(
        static_cast<uint8_t>(width >> 8));
    pack.Set<IrSensorRegisterRegionOfInterestBit8::Height8>(
        static_cast<uint8_t>(height >> 8));

    pBuilder->Append(
        IrSensorRegisterRegionOfInterestBit8::Bank,
        IrSensorRegisterRegionOfInterestBit8::Address,
        pack.storage);
}

void AppendWindowOfInterestRegisterBlocks(
    IrWriteRegisterBlockBuilder* pBuilder,
    const ::nn::irsensor::Rect& windowOfInterest,
    bool isRoiSettingEnabled
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBuilder);

    auto x = static_cast<uint16_t>(windowOfInterest.x);
    auto y = static_cast<uint16_t>(windowOfInterest.y);
    // レジスタには 1 を減じた値を設定する
    auto width = static_cast<uint16_t>(windowOfInterest.width - 1);
    auto height = static_cast<uint16_t>(windowOfInterest.height - 1);

    if (isRoiSettingEnabled)
    {
        // (WorkAround) WOI が奇数設定された場合は、内部的にそれより1px 大きい偶数WOIを設定後にROIで切り取る
        if (width % 2 == 0)
        {
            width += 1;
        }
    }

    pBuilder->Append(
        IrSensorRegisterWindowOfInterestX::Bank,
        IrSensorRegisterWindowOfInterestX::Address,
        static_cast<uint8_t>(x & 0xff));

    pBuilder->Append(
        IrSensorRegisterWindowOfInterestY::Bank,
        IrSensorRegisterWindowOfInterestY::Address,
        static_cast<uint8_t>(y & 0xff));

    pBuilder->Append(
        IrSensorRegisterWindowOfInterestWidth::Bank,
        IrSensorRegisterWindowOfInterestWidth::Address,
        static_cast<uint8_t>(width & 0xff));

    pBuilder->Append(
        IrSensorRegisterWindowOfInterestHeight::Bank,
        IrSensorRegisterWindowOfInterestHeight::Address,
        static_cast<uint8_t>(height & 0xff));

    ::nn::util::BitPack8 pack;

    pack.Clear();
    pack.Set<IrSensorRegisterWindowOfInterestBit8::X8>(
        static_cast<uint8_t>(x >> 8));
    pack.Set<IrSensorRegisterWindowOfInterestBit8::Y8>(
        static_cast<uint8_t>(y >> 8));
    pack.Set<IrSensorRegisterWindowOfInterestBit8::Width8>(
        static_cast<uint8_t>(width >> 8));
    pack.Set<IrSensorRegisterWindowOfInterestBit8::Height8>(
        static_cast<uint8_t>(height >> 8));

    pBuilder->Append(
        IrSensorRegisterWindowOfInterestBit8::Bank,
        IrSensorRegisterWindowOfInterestBit8::Address,
        pack.storage);

    if (isRoiSettingEnabled)
    {
        // ROI の設定
        AppendRegionOfInterestRegisterBlocks(pBuilder, windowOfInterest);
    }
}

void AppendCameraConfigRegisterBlocks(
    IrWriteRegisterBlockBuilder* pBuilder,
    const ::nn::irsensor::PackedIrCameraConfig& config,
    bool isSubtractionEnabled
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBuilder);

    uint64_t exposureTime = GetExposureTimeRegisterValue(config);
    pBuilder->Append(
        IrSensorRegisterManualExposureTime0::Bank,
        IrSensorRegisterManualExposureTime0::Address,
        static_cast<uint8_t>(exposureTime & 0xff));
    pBuilder->Append(
        IrSensorRegisterManualExposureTime1::Bank,
        IrSensorRegisterManualExposureTime1::Address,
        static_cast<uint8_t>((exposureTime >> 8) & 0xff));
    pBuilder->Append(
        IrSensorRegisterManualExposureTime2::Bank,
        IrSensorRegisterManualExposureTime2::Address,
        static_cast<uint8_t>((exposureTime >> 16) & 0xff));

    pBuilder->Append(
        IrSensorRegisterIrCameraLight::Bank,
        IrSensorRegisterIrCameraLight::Address,
        GetIrCameraLightRegisterValue(config.lightTarget, false));

    const GainCode GainCodeMap[] =
    {
        GainCode::X0,
        GainCode::X1,
        GainCode::X2,
        GainCode::X3,
        GainCode::X4,
        GainCode::X5,
        GainCode::X6,
        GainCode::X7,
        GainCode::X8,
        GainCode::X9,
        GainCode::X10,
        GainCode::X11,
        GainCode::X12,
        GainCode::X13,
        GainCode::X14,
        GainCode::X15,
        GainCode::X16,
    };

    NN_SDK_ASSERT_RANGE(
        config.gain, 0, static_cast<int>(sizeof(GainCodeMap) / sizeof(GainCodeMap[0])));
    pBuilder->Append(
        IrSensorRegisterManualGain0::Bank,
        IrSensorRegisterManualGain0::Address,
        static_cast<uint8_t>(static_cast<uint16_t>(GainCodeMap[config.gain]) & 0xff));
    pBuilder->Append(
        IrSensorRegisterManualGain1::Bank,
        IrSensorRegisterManualGain1::Address,
        static_cast<uint8_t>((static_cast<uint16_t>(GainCodeMap[config.gain]) >> 8) & 0xff));

    pBuilder->Append(
        IrSensorRegisterFrameSubtraction::Bank,
        IrSensorRegisterFrameSubtraction::Address,
        GetFrameSubtractionRegisterValue(config, isSubtractionEnabled));

    // ノイズ判定用の閾値設定
    pBuilder->Append(
        IrSensorRegisterIntensityThresholdHeader::Bank,
        IrSensorRegisterIntensityThresholdHeader::Address,
        AmbientNoiseIntensityThreshold);
}

void AppendImageSizeFormatRegisterBlocks(
    IrWriteRegisterBlockBuilder* pBuilder,
    const ::nn::irsensor::ImageTransferProcessorFormat format
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBuilder);

    struct ImageSizeFormat final
    {
        BinningSkippingFactor horizontalSkipping;
        BinningSkippingFactor verticalSkipping;
        BinningSkippingFactor horizontalBinning;
        BinningSkippingFactor verticalBinning;
    };

    const ImageSizeFormat SizeFormat_320x240 =
    {
        BinningSkippingFactor::X1,
        BinningSkippingFactor::X1,
        BinningSkippingFactor::X1,
        BinningSkippingFactor::X1
    };

    const ImageSizeFormat SizeFormat_160x120 =
    {
        BinningSkippingFactor::X1,
        BinningSkippingFactor::X1,
        BinningSkippingFactor::X2,
        BinningSkippingFactor::X2
    };

    const ImageSizeFormat SizeFormat_80x60 =
    {
        BinningSkippingFactor::X1,
        BinningSkippingFactor::X2,
        BinningSkippingFactor::X4,
        BinningSkippingFactor::X2
    };

    const ImageSizeFormat SizeFormat_40x30 =
    {
        BinningSkippingFactor::X2,
        BinningSkippingFactor::X4,
        BinningSkippingFactor::X4,
        BinningSkippingFactor::X2
    };

    const ImageSizeFormat SizeFormat_20x15 =
    {
        BinningSkippingFactor::X4,
        BinningSkippingFactor::X8,
        BinningSkippingFactor::X4,
        BinningSkippingFactor::X2
    };

    const ImageSizeFormat imageSizeFormatMap[] =
    {
        SizeFormat_320x240,
        SizeFormat_160x120,
        SizeFormat_80x60,
        SizeFormat_40x30,
        SizeFormat_20x15,
    };

    NN_SDK_ASSERT_RANGE(format, 0, static_cast<int>(sizeof(imageSizeFormatMap) / sizeof(imageSizeFormatMap[0])));
    NN_SDK_ASSERT_MINMAX(imageSizeFormatMap[format].horizontalSkipping, BinningSkippingFactor::X1, BinningSkippingFactor::X8);
    NN_SDK_ASSERT_MINMAX(imageSizeFormatMap[format].verticalSkipping, BinningSkippingFactor::X1, BinningSkippingFactor::X8);
    NN_SDK_ASSERT_MINMAX(imageSizeFormatMap[format].horizontalBinning, BinningSkippingFactor::X1, BinningSkippingFactor::X4);
    NN_SDK_ASSERT_MINMAX(imageSizeFormatMap[format].verticalBinning, BinningSkippingFactor::X1, BinningSkippingFactor::X2);

    ::nn::util::BitPack8 pack;
    pack.Clear();
    pack.Set<IrSensorRegisterBinningSkippingBit8::HorizontalSkipping>(static_cast<uint8_t>(imageSizeFormatMap[format].horizontalSkipping));
    pack.Set<IrSensorRegisterBinningSkippingBit8::VerticalSkipping>(static_cast<uint8_t>(imageSizeFormatMap[format].verticalSkipping));
    pack.Set<IrSensorRegisterBinningSkippingBit8::HorizontalBinning>(static_cast<uint8_t>(imageSizeFormatMap[format].horizontalBinning));
    pack.Set<IrSensorRegisterBinningSkippingBit8::VerticalBinning>(static_cast<uint8_t>(imageSizeFormatMap[format].verticalBinning));

    pBuilder->Append(
        IrSensorRegisterBinningSkippingBit8::Bank,
        IrSensorRegisterBinningSkippingBit8::Address,
        pack.storage);
}

::nn::Result SetupImageProcessor(
    ::nn::xcd::IrCommonData* commonBuffer,
    ::nn::xcd::IrMomentProcessorState* mainBuffer,
    const ::nn::irsensor::PackedMomentProcessorConfig& config,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(mainBuffer);
    NN_UNUSED(config);

    NN_RESULT_THROW(::nn::xcd::SetupIrMomentProcessor(
        commonBuffer, mainBuffer, handle));
}

::nn::Result SetupImageProcessor(
    ::nn::xcd::IrCommonData* commonBuffer,
    ::nn::xcd::IrClusteringProcessorState* mainBuffer,
    const ::nn::irsensor::PackedClusteringProcessorConfig& config,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(mainBuffer);
    NN_UNUSED(config);

    NN_RESULT_THROW(::nn::xcd::SetupIrClusteringProcessor(
        commonBuffer, mainBuffer, handle));
}

::nn::Result SetupImageProcessor(
    ::nn::xcd::IrCommonData* commonBuffer,
    ::nn::xcd::IrImageTransferProcessorState* mainBuffer,
    const ::nn::irsensor::PackedImageTransferProcessorExConfig& config,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(mainBuffer);

    ::nn::xcd::IrImageTransferProcessorFormat format;
    switch (config.trimmingFormat)
    {
    case ::nn::irsensor::ImageTransferProcessorFormat_320x240:
        format = ::nn::xcd::IrImageTransferProcessorFormat::ImageSize_320x240;
        break;
    case ::nn::irsensor::ImageTransferProcessorFormat_160x120:
        format = ::nn::xcd::IrImageTransferProcessorFormat::ImageSize_160x120;
        break;
    case ::nn::irsensor::ImageTransferProcessorFormat_80x60:
        format = ::nn::xcd::IrImageTransferProcessorFormat::ImageSize_80x60;
        break;
    case ::nn::irsensor::ImageTransferProcessorFormat_40x30:
        format = ::nn::xcd::IrImageTransferProcessorFormat::ImageSize_40x30;
        break;
    case ::nn::irsensor::ImageTransferProcessorFormat_20x15:
        format = ::nn::xcd::IrImageTransferProcessorFormat::ImageSize_20x15;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_THROW(::nn::xcd::SetupIrImageTransferProcessor(
        commonBuffer, mainBuffer, format, handle));
}

::nn::Result SetupImageProcessor(
    ::nn::xcd::IrCommonData* commonBuffer,
    ::nn::xcd::IrDpdProcessorState* mainBuffer,
    const ::nn::irsensor::PackedPointingProcessorConfig& config,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(mainBuffer);
    NN_UNUSED(config);

    NN_RESULT_THROW(::nn::xcd::SetupIrDpdProcessor(
        commonBuffer, mainBuffer, handle));
}

::nn::Result SetupImageProcessor(
    ::nn::xcd::IrCommonData* commonBuffer,
    ::nn::xcd::IrTeraPluginProcessorState* mainBuffer,
    const ::nn::irsensor::PackedTeraPluginProcessorConfig& config,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(mainBuffer);
    NN_UNUSED(config);

    NN_RESULT_THROW(::nn::xcd::SetupIrTeraPluginProcessor(
        commonBuffer, mainBuffer, handle));
}


} // namespace

IrSensorXcdDriver::IrSensorXcdDriver() NN_NOEXCEPT
    : m_Handle()
    , m_SamplingSuccessEvent()
    , m_CommandRequestSuccessEvent()
    , m_CommonDataBuffer()
    , m_MainDataBuffer()
    , m_IrCameraStatus(::nn::irsensor::IrCameraStatus_Unconnected)
    , m_SamplingNumber(0)
    , m_TimeStamp(::nn::TimeSpanType())
    , m_SamplingTrialCount(0)
    , m_IsSubtractionEnabled(true)
    , m_RequiredMcuVersion()
    , m_pLatestTransferMemoryAddress(nullptr)
    , m_PluginParam()
{
    // 何もしない
}

IrSensorXcdDriver::~IrSensorXcdDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result IrSensorXcdDriver::Initialize() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::os::CreateSystemEvent(
        &m_SamplingSuccessEvent,
        ::nn::os::EventClearMode_ManualClear,
        false));

    NN_RESULT_DO(::nn::os::CreateSystemEvent(
        &m_CommandRequestSuccessEvent,
        ::nn::os::EventClearMode_ManualClear,
        false));

    NN_RESULT_SUCCESS;
}

void IrSensorXcdDriver::Finalize() NN_NOEXCEPT
{
    m_pLatestTransferMemoryAddress = nullptr;
    ::nn::os::DestroySystemEvent(&m_SamplingSuccessEvent);
    ::nn::os::DestroySystemEvent(&m_CommandRequestSuccessEvent);
}

void IrSensorXcdDriver::Activate(::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(handle);
    // Hid 側の修正と合わせて関数を削除する
    // m_Handle = handle;
    //m_IrCameraStatus = ::nn::irsensor::IrCameraStatus_Available;
}

void IrSensorXcdDriver::ConnectUnsupportedDevice() NN_NOEXCEPT
{
    // Hid 側の修正と合わせて関数を削除する
    //m_IrCameraStatus = ::nn::irsensor::IrCameraStatus_Unsupported;
}

void IrSensorXcdDriver::Deactivate() NN_NOEXCEPT
{
    // Hid 側の修正と合わせて関数を削除する
    //m_IrCameraStatus = ::nn::irsensor::IrCameraStatus_Unconnected;
}

::nn::Result IrSensorXcdDriver::SetXcdDeviceHandle(::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    m_Handle = handle;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SetLatestTransferMemoryAddress(void* pLatestTransferMemoryAddress) NN_NOEXCEPT
{
    m_pLatestTransferMemoryAddress = pLatestTransferMemoryAddress;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::WriteRegisters(
    const ::nn::xcd::IrWriteRegisterBlock* blocks, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(blocks);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    for (auto i = 0; i < count; i += ::nn::xcd::IrWriteRegisterCountMax)
    {
        auto subCount =
            ::std::min(count - i, ::nn::xcd::IrWriteRegisterCountMax);

        ::nn::xcd::IrWriteRegisterSetting setting = {};
        MakeIrWriteRegisterSetting(&setting, &blocks[i], subCount);

        ::nn::os::ClearSystemEvent(&m_CommandRequestSuccessEvent);
        NN_RESULT_DO(::nn::xcd::StartIrWriteRegister(setting, m_Handle));

        if (!::nn::os::TimedWaitSystemEvent(
            &m_CommandRequestSuccessEvent, CommandExecutionTimeout))
        {
            return ::nn::irsensor::ResultIrsensorRegisterAccessTimeout();
        }
        ::nn::os::ClearSystemEvent(&m_CommandRequestSuccessEvent);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::WriteRegistersEx(
    const ::nn::xcd::IrWriteRegisterBlock* blocks, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(blocks);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    for (auto i = 0; i < count; i += ::nn::xcd::IrWriteRegisterExCountMax)
    {
        auto subCount =
            ::std::min(count - i, ::nn::xcd::IrWriteRegisterExCountMax);

        ::nn::xcd::IrWriteRegisterSettingEx setting = {};
        MakeIrWriteRegisterExSetting(&setting, &blocks[i], subCount);

        ::nn::os::ClearSystemEvent(&m_CommandRequestSuccessEvent);
        NN_RESULT_DO(::nn::xcd::StartIrWriteRegisterEx(setting, m_Handle));

        if (!::nn::os::TimedWaitSystemEvent(
            &m_CommandRequestSuccessEvent, CommandExecutionTimeout))
        {
            return ::nn::irsensor::ResultIrsensorRegisterAccessTimeout();
        }
        ::nn::os::ClearSystemEvent(&m_CommandRequestSuccessEvent);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SetConfig(
    const ::nn::irsensor::PackedMomentProcessorConfig& config,
    const ::nn::xcd::IrCommandType& type
    ) NN_NOEXCEPT
{
    const auto MomentSettingRegisterCount = 24;
    ::nn::xcd::IrWriteRegisterBlock blocks[MomentSettingRegisterCount];
    IrWriteRegisterBlockBuilder builder(blocks, MomentSettingRegisterCount);

    bool isSubtractionEnabled = true;
    AppendCameraConfigRegisterBlocks(&builder, config.irCameraConfig, isSubtractionEnabled);

    AppendWindowOfInterestRegisterBlocks(&builder, config.windowOfInterest, true);

    auto xStep = config.windowOfInterest.width /
        ::nn::irsensor::MomentProcessorBlockColumnCount;
    auto yStep = config.windowOfInterest.height /
        ::nn::irsensor::MomentProcessorBlockRowCount;

    builder.Append(
        IrSensorRegisterMomentXStep::Bank,
        IrSensorRegisterMomentXStep::Address,
        static_cast<uint8_t>(xStep));

    builder.Append(
        IrSensorRegisterMomentYStep::Bank,
        IrSensorRegisterMomentYStep::Address,
        static_cast<uint8_t>(yStep));

    MomentPreprocessMode preprocessMode;
    switch (config.preprocess)
    {
    case ::nn::irsensor::MomentProcessorPreprocess_Binarize:
        preprocessMode = MomentPreprocessMode::Binarization;
        break;
    case ::nn::irsensor::MomentProcessorPreprocess_Cutoff:
        preprocessMode = MomentPreprocessMode::NoiseCut;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    builder.Append(
        IrSensorRegisterMomentPreprocessMode::Bank,
        IrSensorRegisterMomentPreprocessMode::Address,
        static_cast<uint8_t>(preprocessMode));

    builder.Append(
        IrSensorRegisterIntensityThreshold::Bank,
        IrSensorRegisterIntensityThreshold::Address,
        config.preprocessIntensityThreshold);

    builder.Append(
        IrSensorRegisterFastUpdate::Bank,
        IrSensorRegisterFastUpdate::Address,
        GetFastUpdateRegisterValue());

    NN_SDK_ASSERT_EQUAL(builder.GetCount(), builder.GetCountMax());

    if (type == ::nn::xcd::IrCommandType::Normal)
    {
        NN_RESULT_DO(
            WriteRegisters(builder.GetBlocks(), builder.GetCount()));
    }
    else if (type == ::nn::xcd::IrCommandType::Extension)
    {
        NN_RESULT_DO(
            WriteRegistersEx(builder.GetBlocks(), builder.GetCount()));
    }
    else
    {
        NN_SDK_ASSERT(-1);
    }

    // 書き込みが成功したら、内部状態を書き換える。
    m_IsSubtractionEnabled = isSubtractionEnabled;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SetConfig(
    const ::nn::irsensor::PackedClusteringProcessorConfig& config,
    const ::nn::xcd::IrCommandType& type
    ) NN_NOEXCEPT
{
    const auto ClusteringSettingRegisterCount = 28;
    ::nn::xcd::IrWriteRegisterBlock blocks[ClusteringSettingRegisterCount];
    IrWriteRegisterBlockBuilder builder(blocks, ClusteringSettingRegisterCount);

    bool isSubtractionEnabled = config.isExternalLightFilterEnebled;
    AppendCameraConfigRegisterBlocks(&builder, config.irCameraConfig, isSubtractionEnabled);

    // 差分機能が無効の場合はフレームレートの設定を半分にする
    auto frameInterval = static_cast<uint8_t>(IrCameraFrameInterval::Clustering);
    if (!isSubtractionEnabled)
    {
        // フレームレートを半分にするので、インターバルを2倍にする。
        frameInterval *= 2;
        NN_SDK_ASSERT_MINMAX(frameInterval, 0, 255);
    }
    builder.Append(
        IrSensorRegisterIrCameraFrameRate::Bank,
        IrSensorRegisterIrCameraFrameRate::Address,
        frameInterval);

    AppendWindowOfInterestRegisterBlocks(&builder, config.windowOfInterest, true);

    NN_SDK_ASSERT_MINMAX(
        config.objectPixelCountMin,
        0,
        ::nn::irsensor::ClusteringProcessorObjectPixelCountMax);
    builder.Append(
        IrSensorRegisterClusteringObjectPixelCountMin0::Bank,
        IrSensorRegisterClusteringObjectPixelCountMin0::Address,
        static_cast<uint8_t>(config.objectPixelCountMin & 0xff));
    builder.Append(
        IrSensorRegisterClusteringObjectPixelCountMin1::Bank,
        IrSensorRegisterClusteringObjectPixelCountMin1::Address,
        static_cast<uint8_t>((config.objectPixelCountMin >> 8) & 0xff));
    builder.Append(
        IrSensorRegisterClusteringObjectPixelCountMin2::Bank,
        IrSensorRegisterClusteringObjectPixelCountMin2::Address,
        static_cast<uint8_t>((config.objectPixelCountMin >> 16) & 0xff));

    NN_SDK_ASSERT_MINMAX(
        config.objectPixelCountMax,
        0,
        ::nn::irsensor::ClusteringProcessorObjectPixelCountMax);
    builder.Append(
        IrSensorRegisterClusteringObjectPixelCountMax0::Bank,
        IrSensorRegisterClusteringObjectPixelCountMax0::Address,
        static_cast<uint8_t>(config.objectPixelCountMax & 0xff));
    builder.Append(
        IrSensorRegisterClusteringObjectPixelCountMax1::Bank,
        IrSensorRegisterClusteringObjectPixelCountMax1::Address,
        static_cast<uint8_t>((config.objectPixelCountMax >> 8) & 0xff));
    builder.Append(
        IrSensorRegisterClusteringObjectPixelCountMax2::Bank,
        IrSensorRegisterClusteringObjectPixelCountMax2::Address,
        static_cast<uint8_t>((config.objectPixelCountMax >> 16) & 0xff));

    builder.Append(
        IrSensorRegisterIntensityThreshold::Bank,
        IrSensorRegisterIntensityThreshold::Address,
        config.objectIntensityMin);

    builder.Append(
        IrSensorRegisterFastUpdate::Bank,
        IrSensorRegisterFastUpdate::Address,
        GetFastUpdateRegisterValue());

    NN_SDK_ASSERT_EQUAL(builder.GetCount(), builder.GetCountMax());

    if (type == ::nn::xcd::IrCommandType::Normal)
    {
        NN_RESULT_DO(
            WriteRegisters(builder.GetBlocks(), builder.GetCount()));
    }
    else if (type == ::nn::xcd::IrCommandType::Extension)
    {
        NN_RESULT_DO(
            WriteRegistersEx(builder.GetBlocks(), builder.GetCount()));
    }
    else
    {
        NN_SDK_ASSERT(-1);
    }

    // 書き込みが成功したら、内部状態を書き換える。
    m_IsSubtractionEnabled = isSubtractionEnabled;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SetConfig(
    const ::nn::irsensor::PackedImageTransferProcessorExConfig& config,
    const ::nn::xcd::IrCommandType& type
    ) NN_NOEXCEPT
{
    const auto ImageTransferSettingRegisterCount = 17;
    ::nn::xcd::IrWriteRegisterBlock blocks[ImageTransferSettingRegisterCount];
    IrWriteRegisterBlockBuilder builder(blocks, ImageTransferSettingRegisterCount);

    bool isSubtractionEnabled = config.isExternalLightFilterEnebled;
    AppendCameraConfigRegisterBlocks(&builder, config.irCameraConfig, isSubtractionEnabled);

    // 差分機能が無効の場合はフレームレートの設定を半分にする
    auto frameInterval = static_cast<uint8_t>(IrCameraFrameInterval::SpiImage);
    if (!isSubtractionEnabled)
    {
        // フレームレートを半分にするので、インターバルを2倍にする。
        frameInterval *= 2;
        NN_SDK_ASSERT_MINMAX(frameInterval, 0, 255);
    }
    builder.Append(
        IrSensorRegisterIrCameraFrameRate::Bank,
        IrSensorRegisterIrCameraFrameRate::Address,
        frameInterval);

    // WOI -> Binning の順で処理が行われるため適切に処理をコンバートする必要がある。
    ::nn::irsensor::Rect windowOfInterest = CalculateImageTransferWoi(config);
    if (config.origFormat >= ::nn::irsensor::ImageTransferProcessorFormat_80x60)
    {
        // 80x60 の時に最左列が暗くなる問題の WorkAround
        windowOfInterest.x += 2;
    }
    AppendWindowOfInterestRegisterBlocks(&builder, windowOfInterest, false);

    auto origFormat = static_cast<::nn::irsensor::ImageTransferProcessorFormat>(config.origFormat);
    AppendImageSizeFormatRegisterBlocks(&builder, origFormat);

    // 40x30 以下のサイズ設定の時にINTクロック割り込みが遅くなる HW不具合の WorkAround
    uint8_t powerControlFlag = 0x00;
    if (origFormat >= ::nn::irsensor::ImageTransferProcessorFormat_40x30)
    {
        powerControlFlag = 0x80;
    }
    builder.Append(
        IrSensorRegisterPowerControl::Bank,
        IrSensorRegisterPowerControl::Address,
        powerControlFlag);

    builder.Append(
        IrSensorRegisterFastUpdate::Bank,
        IrSensorRegisterFastUpdate::Address,
        GetFastUpdateRegisterValue());

    NN_SDK_ASSERT_EQUAL(builder.GetCount(), builder.GetCountMax());

    if (type == ::nn::xcd::IrCommandType::Normal)
    {
        NN_RESULT_DO(
            WriteRegisters(builder.GetBlocks(), builder.GetCount()));
    }
    else if (type == ::nn::xcd::IrCommandType::Extension)
    {
        NN_RESULT_DO(
            WriteRegistersEx(builder.GetBlocks(), builder.GetCount()));
    }
    else
    {
        NN_SDK_ASSERT(-1);
    }

    // 書き込みが成功したら、内部状態を書き換える。
    m_IsSubtractionEnabled = isSubtractionEnabled;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SetConfig(
    const ::nn::irsensor::PackedPointingProcessorConfig& config,
    const ::nn::xcd::IrCommandType& type
    ) NN_NOEXCEPT
{
    const auto DpdSettingRegisterCount = 12;
    ::nn::xcd::IrWriteRegisterBlock blocks[DpdSettingRegisterCount];
    IrWriteRegisterBlockBuilder builder(blocks, DpdSettingRegisterCount);

    AppendWindowOfInterestRegisterBlocks(&builder, config.windowOfInterest, true);

    builder.Append(
        IrSensorRegisterFastUpdate::Bank,
        IrSensorRegisterFastUpdate::Address,
        GetFastUpdateRegisterValue());

    NN_SDK_ASSERT_EQUAL(builder.GetCount(), builder.GetCountMax());
    if (type == ::nn::xcd::IrCommandType::Normal)
    {
        NN_RESULT_DO(
            WriteRegisters(builder.GetBlocks(), builder.GetCount()));
    }
    else if (type == ::nn::xcd::IrCommandType::Extension)
    {
        NN_RESULT_DO(
            WriteRegistersEx(builder.GetBlocks(), builder.GetCount()));
    }
    else
    {
        NN_SDK_ASSERT(-1);
    }

    // デフォルト設定は差分無効
    m_IsSubtractionEnabled = false;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SetConfig(
    const ::nn::irsensor::PackedTeraPluginProcessorConfig& config,
    const ::nn::xcd::IrCommandType& type
) NN_NOEXCEPT
{
    NN_UNUSED(config);
    NN_UNUSED(type);

    // デフォルト設定は差分有効
    m_IsSubtractionEnabled = true;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SetConfig(
    const ::nn::irsensor::PackedIrLedProcessorConfig& config,
    const ::nn::xcd::IrCommandType& type
    ) NN_NOEXCEPT
{
    const auto IrLedSettingRegisterCount = 2;
    ::nn::xcd::IrWriteRegisterBlock blocks[IrLedSettingRegisterCount];
    IrWriteRegisterBlockBuilder builder(blocks, IrLedSettingRegisterCount);

    builder.Append(
        IrSensorRegisterIrCameraLight::Bank,
        IrSensorRegisterIrCameraLight::Address,
        GetIrCameraLightRegisterValue(config.lightTarget, true));

    builder.Append(
        IrSensorRegisterFastUpdate::Bank,
        IrSensorRegisterFastUpdate::Address,
        GetFastUpdateRegisterValue());

    NN_SDK_ASSERT_EQUAL(builder.GetCount(), builder.GetCountMax());
    if (type == ::nn::xcd::IrCommandType::Normal)
    {
        NN_RESULT_DO(
            WriteRegisters(builder.GetBlocks(), builder.GetCount()));
    }
    else if (type == ::nn::xcd::IrCommandType::Extension)
    {
        NN_RESULT_DO(
            WriteRegistersEx(builder.GetBlocks(), builder.GetCount()));
    }
    else
    {
        NN_SDK_ASSERT(-1);
    }

    // デフォルト設定は差分無効
    m_IsSubtractionEnabled = false;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::GetMcuVersion(::nn::xcd::McuVersionData* pOutVersion) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_IrCameraStatus == ::nn::irsensor::IrCameraStatus_Available,
        ::nn::irsensor::ResultIrsensorBusy());

    NN_RESULT_TRY(::nn::xcd::GetMcuVersion(pOutVersion, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultMcuVersionNotAvailable)
        {
            // バージョン取得ができなかった場合は、McuNotAvailable を返す
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorMcuNotAvailable());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            // 未接続状態だった場合は、Busyを返す
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            // その他のエラーの場合は、即上に通知
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    // バージョン取得に成功した場合は成功を返す
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::CheckMcuValidity() NN_NOEXCEPT
{
    // FW 破壊がないかチェック
    const auto RetrySleepTime = ::nn::TimeSpan::FromMilliSeconds(15);
    const auto RetryCountMax = 300;

    ::nn::xcd::McuVersionData mcuVersion;
    auto trialCounter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        NN_RESULT_TRY(GetMcuVersion(&mcuVersion))
            NN_RESULT_CATCH_ALL
            {
                if (trialCounter > RetryCountMax)
                {
                    NN_RESULT_RETHROW;
                }
                ::nn::os::SleepThread(RetrySleepTime);
                trialCounter++;
                continue;
            }
        NN_RESULT_END_TRY
        break;
    }

    if (mcuVersion.isIapCorrupted
        || mcuVersion.isCorrupted)
    {
        // IAP 故障,  データ故障の場合は NeedUpdate エラーを返す。
        NN_RESULT_THROW(::nn::irsensor::ResultIrsensorNeedUpdate());
    }
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::CheckVersion(
    ::nn::xcd::IrProcessorType mode,
    const::nn::irsensor::PackedFunctionLevel& functionLevel) NN_NOEXCEPT
{
    switch (mode)
    {
    case ::nn::xcd::IrProcessorType::Moment:
    case ::nn::xcd::IrProcessorType::Clustering:
    case ::nn::xcd::IrProcessorType::ImageTransfer:
    case ::nn::xcd::IrProcessorType::Dpd:
    case ::nn::xcd::IrProcessorType::TeraPlugin:
        {
            // バージョンチェック
            ::nn::irsensor::PackedMcuVersion requiredVersion;
            requiredVersion.major = m_RequiredMcuVersion.major;
            requiredVersion.minor = m_RequiredMcuVersion.minor;
            NN_RESULT_DO(CheckVersion(requiredVersion, functionLevel));
        }
        break;
    default:
        // 何もしない
        break;
    }
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::CheckVersion(
    const ::nn::irsensor::PackedMcuVersion& requiredVersion,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel) NN_NOEXCEPT
{
    NN_UNUSED(requiredVersion);

    // FW 破壊がないかチェック
    ::nn::xcd::McuVersionData mcuVersion;
    NN_RESULT_DO(GetMcuVersion(&mcuVersion));

    if (mcuVersion.isIapCorrupted
        || mcuVersion.isCorrupted)
    {
        // IAP 故障,  データ故障の場合は NeedUpdate エラーを返す。
        NN_RESULT_THROW(::nn::irsensor::ResultIrsensorNeedUpdate());
    }

    // バージョンチェック  (FunctionLevel による制限を行う)
    NN_SDK_ASSERT_GREATER(NN_ARRAY_SIZE(RequiredVersionTableForFunctionLevel), functionLevel.level);
    uint32_t mcuVersionNumber = ((mcuVersion.major << 16) | mcuVersion.minor);
    uint32_t requiredVersionNumber = RequiredVersionTableForFunctionLevel[functionLevel.level].ToUint32();
    NN_RESULT_THROW_UNLESS(mcuVersionNumber >= requiredVersionNumber,
        ::nn::irsensor::ResultIrsensorNeedUpdate());

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::RequestVersion() NN_NOEXCEPT
{
    NN_RESULT_TRY(::nn::xcd::RequestMcuVersion(m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorUnconnected());
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::CheckBatteryStatus() NN_NOEXCEPT
{
    NN_RESULT_TRY(::nn::xcd::CheckIrDevicePower(m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultLowBattery)
        {
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorNeedCharge());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultNotSupported)
        {
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorInvalidHandle());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::StopImageProcessor() NN_NOEXCEPT
{
    // 既に停止済みの場合はスキップ
    if (m_RequiredMcuVersion.major != 0 && m_RequiredMcuVersion.minor != 0)
    {
        // 要求バージョンをリセット
        m_RequiredMcuVersion.major = 0;
        m_RequiredMcuVersion.minor = 0;
        // TeraPlugin のパラメータをリセット
        m_PluginParam.isParameterEnabled = false;
        m_PluginParam.parameterSize = 0;
        memset(&m_PluginParam.parameter, 0, ::nn::xcd::IrTeraPluginParameterSizeMax);

        // ポーリングモードを無効化する (BTMCU 3.82以降のみ)
        ::nn::xcd::DisableIrPollingMode(m_Handle);
        ::nn::xcd::StopIrSampling(m_Handle);
        ::nn::irsensor::PackedFunctionLevel functionLevel = {};
        SetMode(::nn::xcd::IrProcessorType::Ready, 0, ::nn::xcd::IrCommandType::Normal, functionLevel);
        ::nn::xcd::TeardownIrProcessor(m_Handle);
        UnlockMcu();
        ::nn::xcd::ResetDataFormat(m_Handle);
    }
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SuspendImageProcessor() NN_NOEXCEPT
{
    // 既に停止済みの場合はスキップ
    if (m_RequiredMcuVersion.major != 0 && m_RequiredMcuVersion.minor != 0)
    {
        // ポーリングモードを無効化する (BTMCU 3.82以降のみ)
        ::nn::xcd::DisableIrPollingMode(m_Handle);
        ::nn::xcd::StopIrSampling(m_Handle);
        ::nn::irsensor::PackedFunctionLevel functionLevel = {};
        SetMode(::nn::xcd::IrProcessorType::Ready, 0, ::nn::xcd::IrCommandType::Normal, functionLevel);
        ::nn::xcd::TeardownIrProcessor(m_Handle);
    }
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::StopSampling() NN_NOEXCEPT
{
    // 既に停止済みの場合はスキップ
    if (m_RequiredMcuVersion.major != 0 && m_RequiredMcuVersion.minor != 0)
    {
        // ポーリングモードを無効化する (BTMCU 3.82以降のみ)
        ::nn::xcd::DisableIrPollingMode(m_Handle);
        ::nn::xcd::StopIrSampling(m_Handle);
    }
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::StartSampling(
    ::nn::xcd::IrProcessorType mode,
    void* pLatestTranfserMemoryAddress,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    // ImageTransfer モードの場合は、最新のトランスファーメモリが適切に
    // 設定されていることを確認してからサンプリングを開始する
    if (mode == ::nn::xcd::IrProcessorType::ImageTransfer
        && m_MainDataBuffer.imageTransfer.pImage != pLatestTranfserMemoryAddress)
    {
        // 満たしていない場合はサンプリングを開始しない
        NN_RESULT_THROW(nn::irsensor::ResultIrsensorBusy());
    }
    NN_RESULT_DO(::nn::xcd::StartIrSampling(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::RunImageProcessor(
    const ::nn::irsensor::PackedMomentProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    // 要求バージョンを登録
    m_RequiredMcuVersion.major = config.requiredVersion.major;
    m_RequiredMcuVersion.minor = config.requiredVersion.minor;

    int modeOffset = 0;

    NN_RESULT_THROW(RunImageProcessor<MomentProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.moment, modeOffset));
}

::nn::Result IrSensorXcdDriver::RunImageProcessor(
    const ::nn::irsensor::PackedClusteringProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    // 要求バージョンを登録
    m_RequiredMcuVersion.major = config.requiredVersion.major;
    m_RequiredMcuVersion.minor = config.requiredVersion.minor;

    int modeOffset = 0;

    NN_RESULT_THROW(RunImageProcessor<ClusteringProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.clustering, modeOffset));
}

::nn::Result IrSensorXcdDriver::RunImageProcessor(
    const ::nn::irsensor::PackedImageTransferProcessorExConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel,
    void* buffer) NN_NOEXCEPT
{
    // 要求バージョンを登録
    m_RequiredMcuVersion.major = config.requiredVersion.major;
    m_RequiredMcuVersion.minor = config.requiredVersion.minor;

    int modeOffset = 0;
    // TODO: xcd の ImageTransfer へのバッファの渡し方が変更された後に修正する
    m_MainDataBuffer.imageTransfer.pImage = buffer;

    NN_RESULT_THROW(RunImageProcessor<ImageTransferProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.imageTransfer, modeOffset));
}

::nn::Result IrSensorXcdDriver::RunImageProcessor(
    const ::nn::irsensor::PackedPointingProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    // 要求バージョンを登録
    m_RequiredMcuVersion.major = config.requiredVersion.major;
    m_RequiredMcuVersion.minor = config.requiredVersion.minor;

    int modeOffset = 0;

    NN_RESULT_THROW(RunImageProcessor<PointingProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.dpd[0], modeOffset));
}

::nn::Result IrSensorXcdDriver::RunImageProcessor(
    const ::nn::irsensor::PackedTeraPluginProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    // 要求バージョンを登録
    m_RequiredMcuVersion.major = config.requiredVersion.major;
    m_RequiredMcuVersion.minor = config.requiredVersion.minor;

    int modeOffset = config.mode;
    NN_SDK_ASSERT_GREATER_EQUAL(modeOffset, 0);

    m_PluginParam.isParameterEnabled = (config.param.setting >> 7) ? true : false;
    m_PluginParam.parameterSize = static_cast<size_t>(config.param.setting & 0x1F);
    ::std::memcpy(&m_PluginParam.parameter, &config.param.data, m_PluginParam.parameterSize);

    NN_RESULT_THROW(RunImageProcessor<TeraPluginProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.teraPlugin, modeOffset));
}

::nn::Result IrSensorXcdDriver::RunImageProcessor(
    const ::nn::irsensor::PackedIrLedProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    // 要求バージョンを登録
    m_RequiredMcuVersion.major = config.requiredVersion.major;
    m_RequiredMcuVersion.minor = config.requiredVersion.minor;

    NN_RESULT_THROW_UNLESS(
        m_IrCameraStatus == ::nn::irsensor::IrCameraStatus_Available,
        ::nn::irsensor::ResultIrsensorBusy());

    // 電池残量のチェックを行う
    NN_RESULT_DO(CheckBatteryStatus());

    auto needsRollback = true;

    // TODO: 将来的には ActivateIrsensor の内部で設定してもらう
    NN_RESULT_DO(::nn::xcd::SetIrDataFormat(m_Handle));

    NN_RESULT_DO(LockMcu());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            UnlockMcu();
            ::nn::xcd::ResetDataFormat(m_Handle);
        }
    };

    // Required バージョンのチェックを行う
    NN_RESULT_DO(CheckVersion(IrLedProcessorInfo::XcdType, functionLevel));

    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::xcd::SetIrControlEvent(
        &m_SamplingSuccessEvent,
        &m_CommandRequestSuccessEvent,
        m_Handle));

    // 内部的には Moment モードとして動作させる
    ::nn::irsensor::PackedMomentProcessorConfig configMoment = {};
    NN_RESULT_DO(SetupImageProcessor(
        &m_CommonDataBuffer,
        &m_MainDataBuffer.moment,
        configMoment,
        m_Handle));

    // m_RequiredMcuVerion は実際のFWバージョンではないが、ここに到達している時点で、
    // 実際のFWバージョンは、m_RequiredMcuVersion より新しいことが保証される。
    // ここでは、FunctionLevel1 は強制アップデートであるため、
    // FunctionLevel0 のshim かつ FWを利用している人のみが旧シーケンスを通る。
    int modeOffset = 0;
    if (m_RequiredMcuVersion <= ::nn::xcd::FirmwareVersionTera_030b)
    {
        // 旧 FW の場合は旧シーケンスで呼び出す
        NN_RESULT_DO(SetMode(IrLedProcessorInfo::XcdType, modeOffset, ::nn::xcd::IrCommandType::Normal, functionLevel));
        NN_RESULT_DO(SetConfig(config, ::nn::xcd::IrCommandType::Normal));
    }
    else
    {
        NN_RESULT_DO(SetConfig(config, ::nn::xcd::IrCommandType::Normal));
        NN_RESULT_DO(SetMode(IrLedProcessorInfo::XcdType, modeOffset, ::nn::xcd::IrCommandType::Normal, functionLevel));
    }

    // 次回取得するサンプリングを開始
    ::nn::os::ClearSystemEvent(&m_SamplingSuccessEvent);
    NN_RESULT_DO(StartSampling(IrLedProcessorInfo::XcdType, m_pLatestTransferMemoryAddress, m_Handle));
    m_SamplingTrialCount = 0;

    // ポーリングモードを有効化する (BTMCU 3.82以降のみ)
    NN_RESULT_DO(::nn::xcd::EnableIrPollingMode(m_Handle));

    // バッテリーの充電状態を必要に応じてリセット
    NN_RESULT_DO(::nn::xcd::ResetBatteryChargeTimer(m_Handle));

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

template <typename Info>
::nn::Result IrSensorXcdDriver::RunImageProcessor(
    const typename Info::IrSensorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel,
    void* buffer, int modeOffset) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_IrCameraStatus == ::nn::irsensor::IrCameraStatus_Available,
        ::nn::irsensor::ResultIrsensorBusy());

    // 電池残量のチェックを行う
    NN_RESULT_DO(CheckBatteryStatus());

    auto needsRollback = true;

    // TODO: 将来的には ActivateIrsensor の内部で設定してもらう
    NN_RESULT_DO(::nn::xcd::SetIrDataFormat(m_Handle));

    NN_RESULT_DO(LockMcu());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            UnlockMcu();
            ::nn::xcd::ResetDataFormat(m_Handle);
        }
    };

    // Required バージョンのチェックを行う
    NN_RESULT_DO(CheckVersion(Info::XcdType, functionLevel));

    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::xcd::SetIrControlEvent(
        &m_SamplingSuccessEvent,
        &m_CommandRequestSuccessEvent,
        m_Handle));

    NN_RESULT_DO(SetupImageProcessor(
        &m_CommonDataBuffer,
        reinterpret_cast<typename Info::XcdBuffer>(buffer),
        config,
        m_Handle));

    // m_RequiredMcuVerion は実際のFWバージョンではないが、ここに到達している時点で、
    // 実際のFWバージョンは、m_RequiredMcuVersion より新しいことが保証される。
    // ここでは、FunctionLevel1 は強制アップデートであるため、
    // FunctionLevel0 のshim かつ FWを利用している人のみが旧シーケンスを通る。
    if (m_RequiredMcuVersion <= ::nn::xcd::FirmwareVersionTera_030b)
    {
        // 旧 FW の場合は旧シーケンスで呼び出す
        NN_RESULT_DO(SetMode(Info::XcdType, modeOffset, ::nn::xcd::IrCommandType::Normal, functionLevel));
        NN_RESULT_DO(SetConfig(config, ::nn::xcd::IrCommandType::Normal));
    }
    else
    {
        NN_RESULT_DO(SetConfig(config, ::nn::xcd::IrCommandType::Normal));
        NN_RESULT_DO(SetMode(Info::XcdType, modeOffset, ::nn::xcd::IrCommandType::Normal, functionLevel));
    }

    // 次回取得するサンプリングを開始
    ::nn::os::ClearSystemEvent(&m_SamplingSuccessEvent);
    NN_RESULT_DO(StartSampling(Info::XcdType, m_pLatestTransferMemoryAddress, m_Handle));
    m_SamplingTrialCount = 0;

    // ポーリングモードを有効化する (BTMCU 3.82以降のみ)
    NN_RESULT_DO(::nn::xcd::EnableIrPollingMode(m_Handle));

    // バッテリーの充電状態を必要に応じてリセット
    NN_RESULT_DO(::nn::xcd::ResetBatteryChargeTimer(m_Handle));

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::ResumeImageProcessor(
    const ::nn::irsensor::PackedMomentProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    int modeOffset = 0;

    NN_RESULT_THROW(ResumeImageProcessor<MomentProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.moment, modeOffset));
}

::nn::Result IrSensorXcdDriver::ResumeImageProcessor(
    const ::nn::irsensor::PackedClusteringProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    int modeOffset = 0;

    NN_RESULT_THROW(ResumeImageProcessor<ClusteringProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.clustering, modeOffset));
}

::nn::Result IrSensorXcdDriver::ResumeImageProcessor(
    const ::nn::irsensor::PackedImageTransferProcessorExConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel,
    void* buffer) NN_NOEXCEPT
{
    int modeOffset = 0;
    // TODO: xcd の ImageTransfer へのバッファの渡し方が変更された後に修正する
    m_MainDataBuffer.imageTransfer.pImage = buffer;

    NN_RESULT_THROW(ResumeImageProcessor<ImageTransferProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.imageTransfer, modeOffset));
}

::nn::Result IrSensorXcdDriver::ResumeImageProcessor(
    const ::nn::irsensor::PackedPointingProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    int modeOffset = 0;

    NN_RESULT_THROW(ResumeImageProcessor<PointingProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.dpd[0], modeOffset));
}

::nn::Result IrSensorXcdDriver::ResumeImageProcessor(
    const ::nn::irsensor::PackedTeraPluginProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    int modeOffset = config.mode;
    NN_SDK_ASSERT_GREATER_EQUAL(modeOffset, 0);
    m_PluginParam.isParameterEnabled = (config.param.setting >> 7) ? true : false;
    m_PluginParam.parameterSize = (config.param.setting & 0x1F);
    ::std::memcpy(&m_PluginParam.parameter, &config.param.data, m_PluginParam.parameterSize);

    NN_RESULT_THROW(ResumeImageProcessor<TeraPluginProcessorInfo>(
        config, functionLevel, &m_MainDataBuffer.teraPlugin, modeOffset));
}

::nn::Result IrSensorXcdDriver::ResumeImageProcessor(
    const ::nn::irsensor::PackedIrLedProcessorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_IrCameraStatus == ::nn::irsensor::IrCameraStatus_Available,
        ::nn::irsensor::ResultIrsensorBusy());

    // 電池残量のチェックを行う
    NN_RESULT_DO(CheckBatteryStatus());

    // Required バージョンのチェックを行う
    NN_RESULT_DO(CheckVersion(IrLedProcessorInfo::XcdType, functionLevel));

    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::xcd::SetIrControlEvent(
        &m_SamplingSuccessEvent,
        &m_CommandRequestSuccessEvent,
        m_Handle));

    // 内部的には Moment モードとして動作させる
    ::nn::irsensor::PackedMomentProcessorConfig configMoment = {};
    NN_RESULT_DO(SetupImageProcessor(
        &m_CommonDataBuffer,
        &m_MainDataBuffer.moment,
        configMoment,
        m_Handle));

    // Resume では Ready 状態になっていることが保証されているため、先にレジスタ設定を行う
    int modeOffset = 0;
    NN_RESULT_DO(SetConfig(config, ::nn::xcd::IrCommandType::Extension));
    NN_RESULT_DO(SetMode(IrLedProcessorInfo::XcdType, modeOffset, ::nn::xcd::IrCommandType::Normal, functionLevel));

    // ポーリングモードを有効化する (BTMCU 3.82以降のみ)
    NN_RESULT_DO(::nn::xcd::EnableIrPollingMode(m_Handle));

    // 次回取得するサンプリングを開始
    ::nn::os::ClearSystemEvent(&m_SamplingSuccessEvent);
    NN_RESULT_DO(StartSampling(IrLedProcessorInfo::XcdType, m_pLatestTransferMemoryAddress, m_Handle));
    m_SamplingTrialCount = 0;

    NN_RESULT_SUCCESS;
}

template <typename Info>
::nn::Result IrSensorXcdDriver::ResumeImageProcessor(
    const typename Info::IrSensorConfig& config,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel,
    void* buffer, int modeOffset) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_IrCameraStatus == ::nn::irsensor::IrCameraStatus_Available,
        ::nn::irsensor::ResultIrsensorBusy());

    // 電池残量のチェックを行う
    NN_RESULT_DO(CheckBatteryStatus());

    // Required バージョンのチェックを行う
    NN_RESULT_DO(CheckVersion(Info::XcdType, functionLevel));

    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::xcd::SetIrControlEvent(
        &m_SamplingSuccessEvent,
        &m_CommandRequestSuccessEvent,
        m_Handle));

    NN_RESULT_DO(SetupImageProcessor(
        &m_CommonDataBuffer,
        reinterpret_cast<typename Info::XcdBuffer>(buffer),
        config,
        m_Handle));

    // Resume では Ready 状態になっていることが保証されているため、先にレジスタ設定を行う
    NN_RESULT_DO(SetConfig(config, ::nn::xcd::IrCommandType::Extension));

    NN_RESULT_DO(SetMode(Info::XcdType, modeOffset, ::nn::xcd::IrCommandType::Normal, functionLevel));

    // ポーリングモードを有効化する (BTMCU 3.82以降のみ)
    NN_RESULT_DO(::nn::xcd::EnableIrPollingMode(m_Handle));

    // 次回取得するサンプリングを開始
    ::nn::os::ClearSystemEvent(&m_SamplingSuccessEvent);
    NN_RESULT_DO(StartSampling(Info::XcdType, m_pLatestTransferMemoryAddress, m_Handle));
    m_SamplingTrialCount = 0;

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::GetImageProcessorStates(
    ::nn::irsensor::MomentProcessorState* pOutStates,
    int* pOutCount, int countMax) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetImageProcessorStates<MomentProcessorInfo>(
        pOutStates, pOutCount, countMax));
}

::nn::Result IrSensorXcdDriver::GetImageProcessorStates(
    ::nn::irsensor::ClusteringProcessorState* pOutStates,
    int* pOutCount, int countMax) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetImageProcessorStates<ClusteringProcessorInfo>(
        pOutStates, pOutCount, countMax));
}

::nn::Result IrSensorXcdDriver::GetImageProcessorState(
    ::nn::irsensor::ImageTransferProcessorState* pOutState,
    void* pOutImage) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutState);
    NN_SDK_ASSERT_NOT_NULL(pOutImage);

    NN_RESULT_THROW_UNLESS(
        m_IrCameraStatus == ::nn::irsensor::IrCameraStatus_Available,
        ::nn::irsensor::ResultIrsensorBusy());

    if (::nn::os::TryWaitSystemEvent(&m_CommandRequestSuccessEvent))
    {
        // 次回取得するサンプリングを開始
        NN_RESULT_DO(::nn::xcd::StartIrSampling(m_Handle));
        ::nn::os::ClearSystemEvent(&m_CommandRequestSuccessEvent);
    }

    // データが更新されていた場合は、最新のものを取得
    if (!::nn::os::TryWaitSystemEvent(&m_SamplingSuccessEvent))
    {
        ++m_SamplingTrialCount;
        NN_RESULT_THROW_UNLESS(
            m_SamplingTrialCount < ImageTransferSamplingTrialCountMax,
            ::nn::irsensor::ResultIrsensorSamplingTimeout());

        NN_RESULT_THROW(::nn::irsensor::ResultIrsensorSamplingIncompleted());
    }

    m_SamplingTrialCount = 0;

    NN_RESULT_DO(GetXcdImageProcessorState(pOutState, pOutImage, m_Handle));
    ::nn::os::ClearSystemEvent(&m_SamplingSuccessEvent);

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::GetImageProcessorStates(
    ::nn::irsensor::PointingProcessorMarkerState* pOutStates,
    int* pOutCount, int countMax) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetImageProcessorStates<PointingProcessorInfo>(
        pOutStates, pOutCount, countMax));
}

::nn::Result IrSensorXcdDriver::GetImageProcessorStates(
    ::nn::irsensor::TeraPluginProcessorState* pOutStates,
    int* pOutCount, int countMax) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetImageProcessorStates<TeraPluginProcessorInfo>(
        pOutStates, pOutCount, countMax));
}

template <typename Info>
::nn::Result IrSensorXcdDriver::GetImageProcessorStates(
    typename Info::IrSensorState* pOutStates,
    int* pOutCount, int countMax) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutStates);
    NN_SDK_ASSERT_NOT_NULL(pOutCount);
    NN_SDK_ASSERT_GREATER_EQUAL(countMax, 1);

    NN_RESULT_THROW_UNLESS(
        m_IrCameraStatus == ::nn::irsensor::IrCameraStatus_Available,
        ::nn::irsensor::ResultIrsensorBusy());

    // 電池残量のチェックを行う
    NN_RESULT_DO(CheckBatteryStatus());

    if (::nn::os::TryWaitSystemEvent(&m_CommandRequestSuccessEvent))
    {
        // 次回取得するサンプリングを開始
        NN_RESULT_DO(::nn::xcd::StartIrSampling(m_Handle));
        ::nn::os::ClearSystemEvent(&m_CommandRequestSuccessEvent);
    }

    if (!::nn::os::TryWaitSystemEvent(&m_SamplingSuccessEvent))
    {
        ++m_SamplingTrialCount;
        NN_RESULT_THROW_UNLESS(
            m_SamplingTrialCount < SamplingTrialCountMax,
            ::nn::irsensor::ResultIrsensorSamplingTimeout());

        NN_RESULT_THROW(::nn::irsensor::ResultIrsensorSamplingIncompleted());
    }

    m_SamplingTrialCount = 0;

    // データが更新されていた場合は、最新のものを取得
    NN_RESULT_DO(GetXcdImageProcessorStates(
        pOutStates, pOutCount, countMax, m_Handle));
    ::nn::os::ClearSystemEvent(&m_SamplingSuccessEvent);

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::GetXcdImageProcessorStates(
    ::nn::irsensor::MomentProcessorState* pOutStates,
    int* pOutCount,
    int countMax,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    auto countMaxMin = ::std::min(::nn::xcd::IrMomentProcessorStateCountMax, countMax);

    // スタックサイズを小さくするためにひとつずつ取得する
    for (auto i = 0; i < countMaxMin; ++i)
    {
        ::nn::xcd::IrCommonData commonData;
        ::nn::xcd::IrMomentProcessorState state;
        auto subcount = 0;
        NN_RESULT_TRY(::nn::xcd::GetIrMomentStates(
            &commonData, &state, &subcount, 1, handle))
            NN_RESULT_CATCH(::nn::xcd::ResultIrWriteRegisterCountTimeout)
            {
                NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        NN_SDK_ASSERT_MINMAX(subcount, 0, 1);

        if (subcount == 0)
        {
            *pOutCount = i;
            NN_RESULT_SUCCESS;
        }

        ConvertImageProcessorState(&pOutStates[i], commonData, state);
    }

    *pOutCount = countMaxMin;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::GetXcdImageProcessorStates(
    ::nn::irsensor::ClusteringProcessorState* pOutStates,
    int* pOutCount,
    int countMax,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    auto countMaxMin = ::std::min(::nn::xcd::IrClusteringProcessorStateCountMax, countMax);

    // スタックサイズを小さくするためにひとつずつ取得する
    for (auto i = 0; i < countMaxMin; ++i)
    {
        ::nn::xcd::IrCommonData commonData;
        ::nn::xcd::IrClusteringProcessorState state;
        auto subcount = 0;
        NN_RESULT_TRY(::nn::xcd::GetIrClusteringStates(
            &commonData, &state, &subcount, 1, handle))
            NN_RESULT_CATCH(::nn::xcd::ResultIrWriteRegisterCountTimeout)
            {
                NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        NN_SDK_ASSERT_MINMAX(subcount, 0, 1);

        if (subcount == 0)
        {
            *pOutCount = i;
            NN_RESULT_SUCCESS;
        }

        ConvertImageProcessorState(&pOutStates[i], commonData, state);
    }

    *pOutCount = countMaxMin;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::GetXcdImageProcessorState(
    ::nn::irsensor::ImageTransferProcessorState* pOutState,
    void* pOutImage,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    ::nn::xcd::IrCommonData commonData;
    ::nn::xcd::IrImageTransferProcessorState state;
    // TODO: xcd のインタフェースに修正が必要
    state.pImage = pOutImage;
    NN_RESULT_TRY(::nn::xcd::GetIrImageTransferState(
        &commonData, &state, handle))
        NN_RESULT_CATCH(::nn::xcd::ResultIrWriteRegisterCountTimeout)
        {
            NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    ConvertImageProcessorState(pOutState, commonData, state);

    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::GetXcdImageProcessorStates(
    ::nn::irsensor::PointingProcessorMarkerState* pOutStates,
    int* pOutCount,
    int countMax,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    auto countMaxMin = ::std::min(::nn::xcd::IrDpdProcessorStateCountMax, countMax);

    ::nn::xcd::IrCommonData commonData[::nn::xcd::IrDpdProcessorStateCountMax];
    ::nn::xcd::IrDpdProcessorState state[::nn::xcd::IrDpdProcessorStateCountMax];
    auto subcount = 0;
    NN_RESULT_DO(::nn::xcd::GetIrDpdStates(
        &commonData[0], &state[0], &subcount, countMaxMin, handle));
    NN_SDK_ASSERT_MINMAX(subcount, 0, countMaxMin);

    for (auto i = 0; i < subcount; i++)
    {
        ConvertImageProcessorState(&pOutStates[i], commonData[i], state[i]);
    }

     *pOutCount = subcount;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::GetXcdImageProcessorStates(
    ::nn::irsensor::TeraPluginProcessorState* pOutStates,
    int* pOutCount,
    int countMax,
    ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(countMax);

    ::nn::xcd::IrCommonData commonData;
    ::nn::xcd::IrTeraPluginProcessorState state;
    *pOutCount = 0;
    NN_RESULT_DO(::nn::xcd::GetIrTeraPluginState(
        &commonData, &state, handle));

    ConvertImageProcessorState(pOutStates, commonData, state);
    *pOutCount = 1;
    NN_RESULT_SUCCESS;
}

void IrSensorXcdDriver::ConvertImageProcessorState(
    ::nn::irsensor::MomentProcessorState* pOutState,
    const ::nn::xcd::IrCommonData& commonData,
    const ::nn::xcd::IrMomentProcessorState& state) NN_NOEXCEPT
{
    NN_UNUSED(commonData);

    pOutState->samplingNumber = m_SamplingNumber++;
    int skippedFrameCount = state.diffTimeStampCount;
    if (m_IsSubtractionEnabled)
    {
        // ON/OFF 差分有効時は内部のフレーム数が倍になるので、2で割っておく
        skippedFrameCount /= 2;
    }
    m_TimeStamp += ::nn::TimeSpanType::FromMicroSeconds(MomentFrameInterval.GetMicroSeconds() * skippedFrameCount);
    pOutState->timeStamp = m_TimeStamp;

    for (auto i = 0; i < ::nn::irsensor::MomentProcessorBlockCount; ++i)
    {
        pOutState->blocks[i].averageIntensity = state.blocks[i].averageIntensity;
        pOutState->blocks[i].centroid = state.blocks[i].centroid;
    }
    // 環境ノイズレベルを計算する
    pOutState->ambientNoiseLevel = ComputeNoiseLevel(commonData);
}

void IrSensorXcdDriver::ConvertImageProcessorState(
    ::nn::irsensor::ClusteringProcessorState* pOutState,
    const ::nn::xcd::IrCommonData& commonData,
    const ::nn::xcd::IrClusteringProcessorState& state) NN_NOEXCEPT
{
    NN_UNUSED(commonData);

    pOutState->samplingNumber = m_SamplingNumber++;
    int skippedFrameCount = state.diffTimeStampCount;
    if (m_IsSubtractionEnabled)
    {
        // ON/OFF 差分有効時は内部のフレーム数が倍になるので、2で割っておく
        skippedFrameCount /= 2;
    }
    m_TimeStamp += ::nn::TimeSpanType::FromMicroSeconds(ClusteringFrameInterval.GetMicroSeconds() * skippedFrameCount);
    pOutState->timeStamp = m_TimeStamp;

    pOutState->objectCount = state.objectCount;
    for (auto i = 0; i < pOutState->objectCount; ++i)
    {
        pOutState->objects[i].averageIntensity = state.objects[i].averageIntensity;
        pOutState->objects[i].centroid = state.objects[i].centroid;
        pOutState->objects[i].bound.x = state.objects[i].bound.x;
        pOutState->objects[i].bound.y = state.objects[i].bound.y;
        pOutState->objects[i].bound.width = state.objects[i].bound.width;
        pOutState->objects[i].bound.height = state.objects[i].bound.height;
        pOutState->objects[i].pixelCount = state.objects[i].pixelCount;
    }
    // 環境ノイズレベルを計算する
    pOutState->ambientNoiseLevel = ComputeNoiseLevel(commonData);
}

void IrSensorXcdDriver::ConvertImageProcessorState(
    ::nn::irsensor::ImageTransferProcessorState* pOutState,
    const ::nn::xcd::IrCommonData& commonData,
    const ::nn::xcd::IrImageTransferProcessorState& state) NN_NOEXCEPT
{
    NN_UNUSED(commonData);
    NN_UNUSED(state);

    pOutState->samplingNumber = m_SamplingNumber++;
    // 環境ノイズレベルを計算する
    pOutState->ambientNoiseLevel = ComputeNoiseLevel(commonData);
}

void IrSensorXcdDriver::ConvertImageProcessorState(
    ::nn::irsensor::PointingProcessorMarkerState* pOutState,
    const ::nn::xcd::IrCommonData& commonData,
    const ::nn::xcd::IrDpdProcessorState& state) NN_NOEXCEPT
{
    NN_UNUSED(commonData);

    // Pointing の時は必ず ON/OFF 差分無効
    NN_SDK_ASSERT_EQUAL(m_IsSubtractionEnabled, false);

    pOutState->samplingNumber = m_SamplingNumber++;
    int skippedFrameCount = state.diffTimeStampCount;
    m_TimeStamp += ::nn::TimeSpanType::FromMilliSeconds(DpdFrameInterval.GetMilliSeconds() * skippedFrameCount);
    pOutState->timeStamp = m_TimeStamp;

    for (auto i = 0; i < ::nn::irsensor::PointingProcessorMarkerObjectCount; i++)
    {
        pOutState->objects[i].isDataValid = state.objects[i].isDataValid;
        pOutState->objects[i].pixelCount = state.objects[i].pixelCount;
        pOutState->objects[i].averageIntensity = state.objects[i].averageIntensity;
        pOutState->objects[i].centroid.x = state.objects[i].centroid.x;
        pOutState->objects[i].centroid.y = state.objects[i].centroid.y;
        pOutState->objects[i].bound.x = state.objects[i].bound.x;
        pOutState->objects[i].bound.y = state.objects[i].bound.y;
        pOutState->objects[i].bound.width = state.objects[i].bound.width;
        pOutState->objects[i].bound.height = state.objects[i].bound.height;
    }
}

void IrSensorXcdDriver::ConvertImageProcessorState(
    ::nn::irsensor::TeraPluginProcessorState* pOutState,
    const ::nn::xcd::IrCommonData& commonData,
    const ::nn::xcd::IrTeraPluginProcessorState& state) NN_NOEXCEPT
{
    NN_UNUSED(commonData);
    NN_UNUSED(state);

    pOutState->samplingNumber = m_SamplingNumber++;
    int skippedFrameCount = state.diffTimeStampCount;
    if (m_IsSubtractionEnabled)
    {
        // ON/OFF 差分有効時は内部のフレーム数が倍になるので、2で割っておく
        skippedFrameCount /= 2;
    }

    m_TimeStamp += ::nn::TimeSpanType::FromMilliSeconds(TeraPluginFrameInterval.GetMilliSeconds() * skippedFrameCount);
    pOutState->timeStamp = m_TimeStamp;

    ::std::memcpy(pOutState->data, state.rawData, sizeof(state.rawData));
    // 環境ノイズレベルを計算する
    pOutState->ambientNoiseLevel = ComputeNoiseLevel(commonData);
}

::nn::Result IrSensorXcdDriver::SetIrCameraStatus(
    ::nn::irsensor::IrCameraStatus status) NN_NOEXCEPT
{
    m_IrCameraStatus = status;
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::LockMcu() NN_NOEXCEPT
{
    NN_RESULT_DO(SetMcuState(::nn::xcd::McuState_Ir));
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::UnlockMcu() NN_NOEXCEPT
{
    NN_RESULT_DO(SetMcuState(::nn::xcd::McuState_Standby));
    NN_RESULT_SUCCESS;
}

::nn::Result IrSensorXcdDriver::SetMcuState(
    ::nn::xcd::McuState state) NN_NOEXCEPT
{
    const auto SleepTime = ::nn::TimeSpan::FromMilliSeconds(15);
    const auto StandbySleepTime = ::nn::TimeSpan::FromMilliSeconds(40);

    auto trialCounter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        NN_RESULT_TRY(::nn::xcd::SetIrMcuState(state, m_Handle))
            NN_RESULT_CATCH(::nn::xcd::ResultMcuBusy)
            {
                // McuBusy の場合はリトライする
                // リトライ上限を超えた場合はエラーを返す
                NN_RESULT_THROW_UNLESS(
                    trialCounter < SettingTrialCountMax,
                    ::nn::irsensor::ResultIrsensorMcuNotAvailable());

                ::nn::os::SleepThread(SleepTime);
                trialCounter++;
                continue;
            }
            NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
            {
                // 未接続状態の場合は、Busy を返す。
                NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
            }
            NN_RESULT_CATCH(::nn::xcd::ResultInvalidMcuState)
            {
                // Mcu の状態が不正なため設定が失敗している。
                // Busy を返してリトライする。
                NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
            }
            NN_RESULT_CATCH_ALL
            {
                // このパスは通り得ない
                // その他のエラーの場合は、上に通知
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY
        // 成功した場合はループを抜けて、ベリファイに進む
        break;
    }

    // 起動時のみ Mcu の正当性をチェックする。
    // 故障していた場合は NeedUpdate を返す
    if (state != ::nn::xcd::McuState_Standby)
    {
        NN_RESULT_DO(CheckMcuValidity());
    }

    for (auto i = 0; i < SettingTrialCountMax; ++i)
    {
        auto current = ::nn::xcd::McuState_Idle;
        NN_RESULT_DO(::nn::xcd::GetIrMcuState(&current, m_Handle));
        if (current == state)
        {
            if (current == ::nn::xcd::McuState_Standby)
            {
                ::nn::os::SleepThread(StandbySleepTime);
            }
            NN_RESULT_SUCCESS;
        }
        ::nn::os::SleepThread(SleepTime);
    }

    NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
}

::nn::Result IrSensorXcdDriver::SetMode(
    ::nn::xcd::IrProcessorType mode, int modeOffset, ::nn::xcd::IrCommandType type,
    const ::nn::irsensor::PackedFunctionLevel& functionLevel) NN_NOEXCEPT
{
    const auto SleepTime = ::nn::TimeSpan::FromMilliSeconds(5);

    auto trialCounter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        ::nn::os::ClearSystemEvent(&m_CommandRequestSuccessEvent);

        // XCD には、FunctionLevel で指定された要求バージョンを通知する
        ::nn::xcd::IrMcuVersion requiredVersion;
        requiredVersion.major = RequiredVersionTableForFunctionLevel[functionLevel.level].major;
        requiredVersion.minor = RequiredVersionTableForFunctionLevel[functionLevel.level].minor;
        NN_RESULT_TRY(::nn::xcd::SetIrProcessorType(mode, modeOffset, m_PluginParam, requiredVersion, type, m_Handle))
            NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
            {
                // 接続が不正な状態になっているため、 Busy を返してリトライする。
                NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
            }
            NN_RESULT_CATCH(::nn::xcd::ResultInvalidMcuState)
            {
                // Mcu の状態が不正なため設定が失敗している。
                // Busy を返してリトライする。
                NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
            }
        NN_RESULT_END_TRY
        if (!::nn::os::TimedWaitSystemEvent(
            &m_CommandRequestSuccessEvent, CommandExecutionTimeout))
        {
            return ::nn::irsensor::ResultIrsensorCommandRequestTimeout();
        }
        ::nn::os::ClearSystemEvent(&m_CommandRequestSuccessEvent);

        for (auto j = 0; j < SettingTrialCountMax; ++j)
        {
            ::nn::xcd::IrProcessorType current;
            ::nn::xcd::IrMcuVersion compatibleMcuVersion;
            NN_RESULT_TRY(::nn::xcd::GetIrProcessorType(&current, &compatibleMcuVersion, m_Handle))
                NN_RESULT_CATCH(::nn::xcd::ResultIrHardwareError)
                {
                    NN_RESULT_THROW_UNLESS(trialCounter < HardwareErrorCheckTrialCountMax,
                        ::nn::irsensor::ResultIrsensorHardwareError());
                    trialCounter++;
                    continue;
                }
                NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
                {
                    // 接続が不正な状態になっているため、 Busy を返してリトライする。
                    NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
                }
                NN_RESULT_CATCH(::nn::xcd::ResultInvalidMcuState)
                {
                    // Mcu の状態が不正なため設定が失敗している。
                    // Busy を返してリトライする。
                    NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
                }
            NN_RESULT_END_TRY
            if (current == mode)
            {
                NN_RESULT_SUCCESS;
            }
            ::nn::os::SleepThread(SleepTime);
        }
        if (trialCounter == 0)
        {
            break;
        }
    }
    NN_RESULT_THROW(::nn::irsensor::ResultIrsensorBusy());
}

::nn::irsensor::IrCameraAmbientNoiseLevel IrSensorXcdDriver::ComputeNoiseLevel(
    const ::nn::xcd::IrCommonData& commonData) NN_NOEXCEPT
{
    auto noiseLevel = ::nn::irsensor::IrCameraAmbientNoiseLevel_Unknown;

    // 差分が有効の時のみ計算を行う
    if (m_IsSubtractionEnabled)
    {
        // ノイズ混入率
        auto noiseInjectedRatio = 0.0f;
        if (commonData.lightPixelOn != 0)
        {
            noiseInjectedRatio = commonData.lightPixelOff / static_cast<float>(commonData.lightPixelOn);
        }

        // 基本的には、noiseInjectedRatio は0.0以上、1.0以下に収まるが、
        // 厳密にはON のピクセル数情報とOFF のピクセル数情報では計測フレームが異なるため、
        // ノイズ混入率が1.0を超える可能性があります。
        // その場合も、Noise が大きい環境としてユーザに返します。
        if (noiseInjectedRatio > AmbientNoiseThresholdHigh)
        {
            noiseLevel = ::nn::irsensor::IrCameraAmbientNoiseLevel_High;
        }
        else if ((noiseInjectedRatio <= AmbientNoiseThresholdHigh)
            && (noiseInjectedRatio > AmbientNoiseThresholdLow))
        {
            noiseLevel = ::nn::irsensor::IrCameraAmbientNoiseLevel_Middle;
        }
        else if (noiseInjectedRatio <= AmbientNoiseThresholdLow)
        {
            noiseLevel = ::nn::irsensor::IrCameraAmbientNoiseLevel_Low;
        }
    }
    return noiseLevel;
}

::nn::irsensor::Rect IrSensorXcdDriver::CalculateImageTransferWoi(
    const ::nn::irsensor::PackedImageTransferProcessorExConfig& config) NN_NOEXCEPT
{
    ::nn::irsensor::Rect woi = {};

    int16_t destWidth = ::nn::irsensor::IrCameraImageWidth >> config.trimmingFormat;
    int16_t destHeight = ::nn::irsensor::IrCameraImageHeight >> config.trimmingFormat;

    int16_t startX = config.trimmingStartX;
    int16_t startY = config.trimmingStartY;

    // 切り取り領域のサイズを 320x240 から切り出す位置に設定
    int origFormat = config.origFormat;
    woi.x = startX << origFormat;
    woi.y = startY << origFormat;
    woi.width = destWidth << origFormat;
    woi.height = destHeight << origFormat;
    return woi;
}

}}} // namespace nn::hid::detail
