﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>

#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_SharedMemoryFormat.h"

namespace nn { namespace hid { namespace detail {

//!< InputDetector インターフェイスの解決を担うマネージャを扱うクラスです。
class InputDetectorManager final
{
    NN_DISALLOW_COPY(InputDetectorManager);
    NN_DISALLOW_MOVE(InputDetectorManager);

private:

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< サンプリング番号
    int64_t m_SamplingNumbers[::nn::hid::system::InputSourceIdCountMax];

    //!< アプレットリソースマネージャ
    AppletResourceManager* m_pAppletResourceManager;

    //!< アプレットリソースマネージャのミューテックス
    ::nn::os::SdkRecursiveMutex* m_pAppletResourceManagerMutex;

    //!< 入力状態
    InputDetectorState m_States[::nn::hid::system::InputSourceIdCountMax];

    //!< ARUID 処理用の一時的メモリ領域
    ::nn::applet::AppletResourceUserId m_TempAruid;

    //!< InputSourceId 処理用の一時的メモリ領域
    system::InputSourceIdSet m_TempId;

public:
    InputDetectorManager() NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< アプレットリソースの状態を保証します。
    ::nn::Result EnsureAppletResource(::nn::applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT;

    //!< アプレットリソースマネージャを設定します。
    void SetAppletResourceManager(AppletResourceManager* pManager,
                                  ::nn::os::SdkRecursiveMutex* pMutex) NN_NOEXCEPT;

    //!< 入力状態を更新します。
    ::nn::Result Notify(const system::InputSourceIdSet& id) NN_NOEXCEPT;

private:

    //!< 共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(InputDetectorManager* that,
                          InputDetectorSharedMemoryFormat* address,
                          ::nn::applet::AppletResourceUserId aruid,
                          bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
