﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TickApi.h>
#include <nn/os/os_TickTypes.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_AppletResourceManager.h"
#include "hid_CommonStateUtility.h"
#include "hid_InputDetectorId.h"
#include "hid_InputDetectorManager.h"

namespace nn { namespace hid { namespace detail {

InputDetectorManager::InputDetectorManager() NN_NOEXCEPT
    : m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
    , m_TempAruid()
    , m_TempId()
{
    for (int i = 0; i < system::InputSourceIdCountMax; i++)
    {
        m_SamplingNumbers[i] = 0;
        m_States[i] = InputDetectorState();
    }
}

::nn::Result InputDetectorManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ::nn::hid::system::ResultInputDetectorActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // LIFO を初期化
        this->ProcessSharedMemory([](InputDetectorManager* that,
                                     InputDetectorSharedMemoryFormat* address,
                                     ::nn::applet::AppletResourceUserId aruid,
                                     bool enablesInput) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(that);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            for (auto& entry : address->entries)
            {
                ::nn::util::Get(entry.lifo).Clear();
            }
        });

        // 入力状態を更新
        system::InputSourceIdSet id;
        id.Set();
        this->Notify(id);
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result InputDetectorManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ::nn::hid::system::ResultInputDetectorDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

::nn::Result InputDetectorManager::EnsureAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_TempAruid = aruid;

    this->ProcessSharedMemory([] (InputDetectorManager* that,
                                  InputDetectorSharedMemoryFormat* address,
                                  ::nn::applet::AppletResourceUserId haystack,
                                  bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);
        NN_UNUSED(enablesInput);

        const ::nn::applet::AppletResourceUserId& needle = that->m_TempAruid;

        for (int i = 0; i < system::InputSourceIdCountMax; i++)
        {
            const InputDetectorState& state = that->m_States[i];

            InputDetectorLifo& lifo = ::nn::util::Get(address->entries[i].lifo);

            if (lifo.IsEmpty() &&
                needle == haystack &&
                needle == ::nn::applet::AppletResourceUserId::GetInvalidId())
            {
                lifo.Append(state);
            }
        }
    });

    NN_RESULT_SUCCESS;
}

void InputDetectorManager::SetAppletResourceManager(AppletResourceManager* pManager,
                                                    ::nn::os::SdkRecursiveMutex* pMutex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

::nn::Result InputDetectorManager::Notify(const system::InputSourceIdSet& id) NN_NOEXCEPT
{
    m_TempId = id;

    this->ProcessSharedMemory([] (InputDetectorManager* that,
                                  InputDetectorSharedMemoryFormat* address,
                                  ::nn::applet::AppletResourceUserId aruid,
                                  bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);
        NN_UNUSED(aruid);
        NN_UNUSED(enablesInput);

        const system::InputSourceIdSet& tempId = that->m_TempId;

        const int64_t Timestamp = ::nn::os::GetSystemTick().GetInt64Value();
        for (int i = 0; i < system::InputSourceIdCountMax; i++)
        {
            int64_t& samplingNumber = that->m_SamplingNumbers[i];

            InputDetectorState& state = that->m_States[i];

            if (tempId.Test(i))
            {
                state.inputSourceState.timestamp = Timestamp;
                state.samplingNumber = samplingNumber++;
                ::nn::util::Get(address->entries[i].lifo).Append(state);
            }
        }
    });

    NN_RESULT_SUCCESS;
}

void InputDetectorManager::ProcessSharedMemory(
    void (*processor)(InputDetectorManager* that,
                      InputDetectorSharedMemoryFormat* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)
                      > locker(*m_pAppletResourceManagerMutex);

    for (const AppletResourceEntry& entry :
             m_pAppletResourceManager->GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);

            processor(this,
                      &entry.address->inputDetector,
                      entry.aruid,
                      entry.flags.Test<AppletResourceFlag::EnablesInput>());
        }
    }
}

}}} // namespace nn::hid::detail
