﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/detail/hid_AbstractedPadTypes.h>
#include <nn/hid/system/hid_CommonTypes.h>
#include <nn/hid/system/hid_PlayReport.h>
#include <nn/hid/detail/hid_PadTypes.h>
#include <nn/hid/detail/hid_SensorTypes.h>
#include <nn/hid/detail/hid_ColorTypes.h>
#include <nn/hid/detail/hid_RxPacketHistoryTypes.h>
#include <nn/os/os_TickTypes.h>
#include <nn/util/util_MathTypes.h>
#include <nn/xcd/xcd_Input.h>

namespace nn { namespace hid { namespace detail {

const int IAbstractedPadCountMax = 64;

class IAbstractedPad
{
private:
    //!< コントローラー番号 1 ～ 8, Handheld のように概念がないものは 0
    int m_ControllerNumber;

    //!< 電池残量低下のオーバーレイ通知を行うかどうかのフラグ
    bool m_IsBatteryLowOvlnRequired;

    //!< 電池残量が低下状態かどうか
    bool m_IsBatteryLow;

    //!< 電池残量低下通知を遅延を判断するための、LowBattery 開始時間
    ::nn::os::Tick m_BatteryLowNotifyDelayTick;

    //!< 電池残量低下の通知を開始した時間
    ::nn::os::Tick m_BatteryLowNotifyStartTick;

protected:
    //!< Id
    AbstractedPadId m_Id;

    //!< デバイスが割り当てられているかどうか
    bool m_IsAttached;

    //!< デバイスの種類
    system::DeviceTypeSet m_DeviceType;

    //!< インタフェースの種類
    system::InterfaceType m_InterfaceType;

    //!< サポートしている機能
    AbstractedPadFeatureSet m_FeatureSet;

    //!< コントローラーの電源状態
    system::PowerInfo m_PowerInfo;

    //!< 新たなコントローラー入力の受信検知フラグ
    bool m_IsNewSampleReceived;

public:
    IAbstractedPad() NN_NOEXCEPT :
        m_ControllerNumber(0),
        m_IsBatteryLowOvlnRequired(false),
        m_IsBatteryLow(false),
        m_IsAttached(false),
        m_InterfaceType(system::InterfaceType_Unknown),
        m_IsNewSampleReceived(false)
    {
        // 何もしない
    }
    virtual ~IAbstractedPad() NN_NOEXCEPT { /* 何もしない */ }

    //!< デバイスの種類を取得する
    system::DeviceTypeSet GetDeviceType() NN_NOEXCEPT
    {
        return m_DeviceType;
    }

    //!< インタフェースの種類を取得する
    system::InterfaceType GetInterfaceType() NN_NOEXCEPT
    {
        return m_InterfaceType;
    }

    //!< サポートしている機能を取得する
    AbstractedPadFeatureSet GetFeatureSet() NN_NOEXCEPT
    {
        return m_FeatureSet;
    }

    //!< デバイスの接続状態を取得する
    virtual bool IsConnected() NN_NOEXCEPT= 0;

    //!< AbstractedPadId を取得する
    AbstractedPadId GetId() NN_NOEXCEPT
    {
        return m_Id;
    }

    //!< デバイスが割り当てられているかどうかを取得する
    bool IsAttached() NN_NOEXCEPT
    {
        return m_IsAttached;
    }

    //!< コントローラー番号を設定する
    void SetControllerNumber(int controllerNumber) NN_NOEXCEPT
    {
        if (m_IsAttached == true)
        {
            m_ControllerNumber = controllerNumber;
        }
    }

    //!< コントローラー番号を取得する
    int GetControllerNumber() NN_NOEXCEPT
    {
        if (m_IsAttached == true)
        {
            return m_ControllerNumber;
        }

        return 0;
    }

    //!< 電池残量低下通知が必要かどうかを取得する
    bool IsBatteryLowOvlnRequired() NN_NOEXCEPT
    {
        return m_IsBatteryLowOvlnRequired;
    }

    //!< 電池残量低下通知時に、内部のフラグのリセット及びタイマーの更新をする
    void ResetIsBatteryLowOvlnOnNotify(nn::os::Tick tick) NN_NOEXCEPT
    {
        m_BatteryLowNotifyStartTick = tick;
        m_IsBatteryLowOvlnRequired = false;
    }

    //!< 新しいサンプルが受信されたかどうかをチェックする。チェック後はフラグがクリアされる
    bool ReadAndClearIsSampleReceived() NN_NOEXCEPT
    {
        if (m_IsNewSampleReceived == false)
        {
            return false;
        }
        m_IsNewSampleReceived = false;
        return true;
    }

    //!< AbstractedPadType を取得する
    virtual AbstractedPadType GetType() NN_NOEXCEPT = 0;

    //!< デバイスの電源状態を取得する
    virtual system::PowerInfo GetPowerInfo() NN_NOEXCEPT = 0;

    //!< デバイスのボタン/アナログスティックの入力状態を取得する
    virtual AbstractedPadState GetPadState() NN_NOEXCEPT = 0;

    //!< 最後にボタンが押下された時間を取得する
    virtual bool GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT = 0;

    //!< コントローラーの色情報を取得する
    virtual system::ControllerColor GetColor() NN_NOEXCEPT
    {

        if (m_DeviceType.Test<system::DeviceType::JoyConLeft>() == true ||
            m_DeviceType.Test<system::DeviceType::JoyConRight>() == true ||
            m_DeviceType.Test<system::DeviceType::HandheldJoyLeft>() == true ||
            m_DeviceType.Test<system::DeviceType::HandheldJoyRight>() == true)
        {
            return { nn::hid::detail::ColorJoyConDefault.color[0], nn::hid::detail::ColorJoyConDefault.color[1] };
        }
        return { nn::hid::detail::ColorSwitchProControllerDefault.fullkey.main, nn::hid::detail::ColorSwitchProControllerDefault.fullkey.sub };
    }

    //!< Switch Pro Controller グリップ色情報を取得する
    virtual bool GetGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip) NN_NOEXCEPT
    {
        if (m_DeviceType.Test<system::DeviceType::SwitchProController>() == true)
        {
            *pOutLeftGrip = nn::hid::detail::ColorSwitchProControllerDefault.fullkey.leftGrip;
            *pOutRightGrip = nn::hid::detail::ColorSwitchProControllerDefault.fullkey.rightGrip;
            return true;
        }
        return false;
    }

    //!< 6軸センサーの値を取得する。
    virtual int GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT
    {
        NN_UNUSED(pOutValue);
        NN_UNUSED(count);
        return 0;
    }

    //!< 6軸センサーの CAL 値を取得する。暫定的に xcd の構造体を利用する。有効な値がない場合は false を返す
    virtual bool GetSensorCalibrationValue(nn::xcd::SensorCalibrationValue* pOutValue) NN_NOEXCEPT
    {
        NN_UNUSED(pOutValue);
        return false;
    }

    //!< 6軸センサーのサンプル間隔を取得する
    virtual nn::TimeSpan GetSensorSamplingInterval() NN_NOEXCEPT
    {
        return nn::TimeSpan::FromMilliSeconds(0);
    }

    //!< コントローラーのインジケーターを制御する
    virtual void SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT = 0;

    //!< コントローラーのインジケーターパターンを取得する
    virtual uint8_t GetIndicator() NN_NOEXCEPT = 0;

    //!< ケーブル接続されているかどうか
    virtual bool IsWired() NN_NOEXCEPT = 0;

    //!< Usb の接続状態を取得します
    virtual bool IsUsbConnected() NN_NOEXCEPT = 0;

    //!< コントローラーを接続状態に変更します
    virtual void Connect() NN_NOEXCEPT = 0;

    //!< コントローラーを切断する
    virtual void Detach() NN_NOEXCEPT = 0;

    //!< NpadId への指定割り当てが有効かどうかをチェックします
    virtual bool IsFixedNpadAssignmentEnabled(NpadIdType* pOutValue) NN_NOEXCEPT
    {
        NN_UNUSED(pOutValue);
        // デフォルトでは無効
        return false;
    }

    //!< コントローラーを再起動する
    virtual void Reboot(bool reconnect) NN_NOEXCEPT
    {
        NN_UNUSED(reconnect);
        // 何もしない
    }

    //!< 接続時の振動を行う
    virtual void VibrationOnConnect() NN_NOEXCEPT
    {
        // 何もしない
    }

    //!< デバイスの内部状態をリセットします
    virtual void ResetInternalDeviceState() NN_NOEXCEPT
    {
        // 何もしない
    }

    //!< バッテリーを搭載しているかどうか取得します
    virtual bool HasBattery() NN_NOEXCEPT = 0;

    //!< プレイレポート用のデフォルトの情報を取得します。正しく読み出せた場合は true を返します。
    virtual bool SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT = 0;

    //!< AbstractedPad の入力状態を取得します。
    virtual void GetPadDriverState(PadDriverState* pOutState) NN_NOEXCEPT
    {
        // 非対応の場合は無入力を返す
        pOutState->buttons.Reset();
        pOutState->analogStickL = AnalogStickDriverState();
        pOutState->analogStickR = AnalogStickDriverState();
    }

    //!< 6軸センサーの状態を取得します。
    virtual int GetSixAxisSensorDriverStates(SixAxisSensorDriverState* pOutStates,
                                             int count) NN_NOEXCEPT
    {
        NN_UNUSED(pOutStates);
        NN_UNUSED(count);
        return 0;
    }

    //!< パケットの受信履歴を取得します。
    virtual RxPacketHistory GetRxPacketHistory() NN_NOEXCEPT
    {
        RxPacketHistory returnValue;
        returnValue.samplingCount = 0;
        return returnValue;
    }

protected:
    //!< 電池残量低下通知のリセット
    void ResetBatteryLowInternalState() NN_NOEXCEPT;

    //!< バッテリー残量通知が必要かどうかを更新する
    void UpdateBatteryLowNotify() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
