﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include "hid_IAbstractedPad.h"

namespace nn { namespace hid { namespace detail {

namespace
{

//!< 電池残量低下を表示開始するまでのディレイ
const ::nn::TimeSpan LowBatteryDisplayDelayTime = nn::TimeSpan::FromSeconds(1);
//!< 電池残量低下を表示する間隔 (15分)
const ::nn::TimeSpan LowBatteryDisplayInterval = nn::TimeSpan::FromSeconds(15 * 60);
//!< 電池残量低下を表示する時間
const ::nn::TimeSpan LowBatteryDisplayTime = nn::TimeSpan::FromSeconds(8);

} // namespace

void IAbstractedPad::ResetBatteryLowInternalState() NN_NOEXCEPT
{
    m_IsBatteryLow = false;
    m_IsBatteryLowOvlnRequired = false;
    m_BatteryLowNotifyStartTick = nn::os::Tick();
    m_BatteryLowNotifyDelayTick = nn::os::Tick();
}

void IAbstractedPad::UpdateBatteryLowNotify() NN_NOEXCEPT
{
    // 初回のみバッテリ通知を行うためのフラグ
    bool forceBatteryNotify = false;

    // 電池残量が CriticalLow 以外または充電中のときは、タイマーのリセットとバッテリー Low
    if (m_PowerInfo.batteryLevel != system::BatteryLevel_CriticalLow ||
        m_PowerInfo.isCharging == true)
    {
        ResetBatteryLowInternalState();
        return;
    }

    if (m_IsBatteryLowOvlnRequired == true)
    {
        // 要通知状態の場合は何もしない
        return;
    }

    auto currentTick = nn::os::GetSystemTick();

    // Low バッテリ遅延通知用 Tick が初期化状態であれば、Low バッテリーになった時間を保存
    if (m_BatteryLowNotifyDelayTick == nn::os::Tick())
    {
        m_BatteryLowNotifyDelayTick = currentTick;
        return;
    }

    // Low バッテリーが一定時間経過しているかをチェック
    if (m_IsBatteryLow == false)
    {
        if ((currentTick - m_BatteryLowNotifyDelayTick).ToTimeSpan() > LowBatteryDisplayDelayTime)
        {
            // 電池電圧が安定した
            m_IsBatteryLow = true;
            // 初回は強制通知
            forceBatteryNotify = true;
        }
        else
        {
            // まだ電池電圧が安定していない
            return;
        }
    }

    // 前回通知から一定時間経過しているか、初回の場合は通知
    if ((currentTick - m_BatteryLowNotifyStartTick).ToTimeSpan() > LowBatteryDisplayInterval ||
        forceBatteryNotify == true)
    {
        // 電池残量低下通知フラグをセット
        m_IsBatteryLowOvlnRequired = true;
    }
}

}}} // namespace nn::hid::detail
