﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_HomeButton.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceManager.h"
#include "hid_HomeButtonManager.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SystemButtonNotifier.h"

namespace nn { namespace hid { namespace detail {

const ::nn::TimeSpan HomeButtonManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(15);

HomeButtonManager::HomeButtonManager() NN_NOEXCEPT
    : m_IsHomeButtonDown(false)
    , m_NeedsToSignal(false)
    , m_HomeButtonEvent(::nn::os::EventClearMode_ManualClear, true)
    , m_SamplingNumber(0)
    , m_pTimer(nullptr)
    , m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
    , m_State()
    , m_TempState()
    , m_TempAruid()
    , m_AutoPilotHomeButtonNotifier()
{
    m_AutoPilotHomeButtonNotifier = this->MakeSystemButtonNotifier();
}

void HomeButtonManager::SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(timer);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimer = timer;
}

void HomeButtonManager::SetAppletResourceManager(
    AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

void HomeButtonManager::SetInputDetectorManager(
    InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pInputDetectorManager = pManager;
    m_pInputDetectorManagerMutex = pMutex;
}

SystemButtonNotifier HomeButtonManager::MakeSystemButtonNotifier() NN_NOEXCEPT
{
    SystemButtonNotifier notifier;
    notifier.m_pIsSystemButtonDown = &m_IsHomeButtonDown;
    notifier.m_pNeedsToSignal = &m_NeedsToSignal;
    return notifier;
}

::nn::Result HomeButtonManager::AcquireHomeButtonEventHandle(
    ::nn::os::NativeHandle* pOutHandle,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    NN_UNUSED(aruid);
    *pOutHandle = m_HomeButtonEvent.GetReadableHandle();
    NN_RESULT_SUCCESS;
}

void HomeButtonManager::SignalHomeButtonEvent() NN_NOEXCEPT
{
    m_HomeButtonEvent.Signal();
}

::nn::Result HomeButtonManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultHomeButtonActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // LIFO を初期化
        this->ProcessSharedMemory([] (HomeButtonManager* that,
                                      HomeButtonSharedMemoryFormat* address,
                                      ::nn::applet::AppletResourceUserId aruid,
                                      bool enablesInput) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(that);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            ::nn::util::Get(address->lifo).Clear();
        });

        // タイマーイベントをアクティブ化
        m_pTimer->Enable(SamplingInterval);

        // ホームボタンが押下状態にあれば通知
        m_NeedsToSignal = static_cast<bool>(m_IsHomeButtonDown);

        m_State = ::nn::hid::system::HomeButtonState();

        // 入力状態を更新
        this->Update();
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result HomeButtonManager::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultHomeButtonDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // 自動操作を無効化
        this->DisableAutoPilot();

        // タイマーイベントを非アクティブ化
        m_pTimer->Disable();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HomeButtonManager::EnsureAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_TempAruid = aruid;

    this->ProcessSharedMemory([] (HomeButtonManager* that,
                                  HomeButtonSharedMemoryFormat* address,
                                  ::nn::applet::AppletResourceUserId haystack,
                                  bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);
        NN_UNUSED(enablesInput);

        ::std::atomic<bool>& needsToSignal = that->m_NeedsToSignal;

        ::nn::os::SystemEvent& event = that->m_HomeButtonEvent;

        const ::nn::hid::system::HomeButtonState& state = that->m_State;

        const ::nn::applet::AppletResourceUserId& needle = that->m_TempAruid;

        HomeButtonLifo& lifo = ::nn::util::Get(address->lifo);

        if (lifo.IsEmpty() &&
            needle == haystack &&
            needle == ::nn::applet::AppletResourceUserId::GetInvalidId())
        {
            lifo.Append(state);

            if (needsToSignal.exchange(false))
            {
                event.Signal();
            }
        }
    });

    NN_RESULT_SUCCESS;
}

::nn::Result HomeButtonManager::SetAutoPilotState(
    const ::nn::hid::debug::HomeButtonAutoPilotState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultHomeButtonNotInitialized());
    m_AutoPilotHomeButtonNotifier.SetSystemButtonState(
        value.buttons.Test<::nn::hid::system::HomeButton::Active>());
    NN_RESULT_SUCCESS;
}

::nn::Result HomeButtonManager::UnsetAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultHomeButtonNotInitialized());
    this->DisableAutoPilot();
    NN_RESULT_SUCCESS;
}

void HomeButtonManager::Sample() NN_NOEXCEPT
{
    if (!m_ActivationCount.IsZero())
    {
        this->Update();

        this->ProcessSharedMemory([] (HomeButtonManager* that,
                                      HomeButtonSharedMemoryFormat* address,
                                      ::nn::applet::AppletResourceUserId aruid,
                                      bool enablesInput) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);

            ::std::atomic<bool>& needsToSignal = that->m_NeedsToSignal;

            ::nn::os::SystemEvent& event = that->m_HomeButtonEvent;

            const ::nn::hid::system::HomeButtonState& state = that->m_State;

            if (aruid == ::nn::applet::AppletResourceUserId::GetInvalidId())
            {
                ::nn::util::Get(address->lifo).Append(state);

                if (needsToSignal.exchange(false))
                {
                    event.Signal();
                }
            }
        });
    }
}

void HomeButtonManager::Update() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManagerMutex);

    // 入力変化検知用に状態を保存
    m_TempState = m_State;

    // 入力状態を取得
    m_State.buttons.Reset();
    m_State.buttons.Set<::nn::hid::system::HomeButton::Active>(
        static_cast<bool>(m_IsHomeButtonDown));

    // サンプリング番号を設定
    m_State.samplingNumber = m_SamplingNumber++;

    ::std::lock_guard<decltype(*m_pInputDetectorManagerMutex)
                      > locker(*m_pInputDetectorManagerMutex);

    if ((m_State.buttons ^ m_TempState.buttons).IsAnyOn())
    {
        m_pInputDetectorManager->Notify(
            ::nn::hid::system::InputSourceId::Pad::Mask);
    }
}

void HomeButtonManager::ProcessSharedMemory(
    void (*processor)(HomeButtonManager* that,
                      HomeButtonSharedMemoryFormat* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)
                      > locker(*m_pAppletResourceManagerMutex);

    for (const AppletResourceEntry& entry :
             m_pAppletResourceManager->GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);

            processor(this,
                      &entry.address->homeButton,
                      entry.aruid,
                      entry.flags.Test<AppletResourceFlag::EnablesInput>());
        }
    }
}

void HomeButtonManager::DisableAutoPilot() NN_NOEXCEPT
{
    m_AutoPilotHomeButtonNotifier.SetSystemButtonState(false);
}

}}} // namespace nn::hid::detail
