﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/xcd/xcd_Vibration.h>
#include <nn/hidbus/server/hidbus_HidbusSystemServer.h>

#include "hid_HidSystemServer.h"
#include "hid_StaticObject.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "hid_ResourceManager-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "hid_ResourceManager-os.win.h"
#endif

namespace nn { namespace hid { namespace detail {

namespace {

//!< os ハンドルを管理するか否かを表す値
const bool NeedsToBeManaged =
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    false;
#else
    false;
#endif

//!< int64_t から AnalogStickPosition に変換する
system::AnalogStickPosition GetAnalogStickPosition(int64_t value)
{
    switch (value)
    {
    case system::AnalogStickPosition_Left:
        return system::AnalogStickPosition_Left;
    case system::AnalogStickPosition_Right:
        return system::AnalogStickPosition_Right;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

} // namespace

HidSystemServer::HidSystemServer() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result HidSystemServer::SendKeyboardLockKeyEvent(
    ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetKeyboardTask()
                                     .SendKeyboardLockKeyEvent(value));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireHomeButtonEventHandle(
    ::nn::sf::Out<nn::sf::NativeHandle> outValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetHomeButtonTask()
                            .AcquireHomeButtonEventHandle(&handle, aruid));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ActivateHomeButton(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto& task = GetResourceManager().GetHomeButtonTask();
    NN_RESULT_DO(task.ActivateHomeButton());
    NN_RESULT_DO(task.EnsureHomeButtonAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireSleepButtonEventHandle(
    ::nn::sf::Out<nn::sf::NativeHandle> outValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetSleepButtonTask()
                            .AcquireSleepButtonEventHandle(&handle, aruid));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ActivateSleepButton(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto& task = GetResourceManager().GetSleepButtonTask();
    NN_RESULT_DO(task.ActivateSleepButton());
    NN_RESULT_DO(task.EnsureSleepButtonAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireCaptureButtonEventHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetCaptureButtonTask()
                            .AcquireCaptureButtonEventHandle(&handle, aruid));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ActivateCaptureButton(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto& task = GetResourceManager().GetCaptureButtonTask();
    NN_RESULT_DO(task.ActivateCaptureButton());
    NN_RESULT_DO(task.EnsureCaptureButtonAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::SetAppletResourceUserId(
    const ::nn::applet::AppletResourceUserId& value) NN_NOEXCEPT
{
    GetResourceManager().SetAppletResourceUserId(value);
    GetResourceManager().GetNpadResourceManager().SetNpadAppletResourceUserId(value);
    GetResourceManager().GetPalmaResourceManager().UpdateOnAruidSwitch();
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::RegisterAppletResourceUserId(
    const ::nn::applet::AppletResourceUserId& aruid,
    bool enablesInput) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().RegisterAppletResourceUserId(aruid, enablesInput));
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager().RegisterNpadAppletResourceUserId(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::UnregisterAppletResourceUserId(
    const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    GetResourceManager().UnregisterAppletResourceUserId(aruid);
    GetResourceManager().GetNpadResourceManager().UnregisterNpadAppletResourceUserId(aruid);
    GetResourceManager().GetPalmaResourceManager().UnregisterAppletResourceUserId(aruid);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::EnableAppletToGetInput(
    const ::nn::applet::AppletResourceUserId& aruid,
    bool enablesInput) NN_NOEXCEPT
{
    GetResourceManager().EnableAppletToGetInput(aruid, enablesInput);
    // 入力状態の有効/無効切り替えはアプレット切り替えに相当するため、Npad の内部状態を更新する
    GetResourceManager().GetNpadResourceManager().UpdateInternalStateForTargetAruid(aruid);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::SetAruidValidForVibration(
    ::nn::applet::AppletResourceUserId aruid, bool enable) NN_NOEXCEPT
{
    GetResourceManager().SetAruidValidForVibration(aruid, enable);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::EnableAppletToGetSixAxisSensor(
    const ::nn::applet::AppletResourceUserId& aruid,
    bool enablesInput) NN_NOEXCEPT
{
    GetResourceManager().EnableAppletToGetSixAxisSensor(aruid, enablesInput);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::EnableHandheldHids() NN_NOEXCEPT
{
    GetResourceManager().EnableHandheldHids();
    GetResourceManager().GetNpadResourceManager().UpdateHandheld();
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::DisableHandheldHids() NN_NOEXCEPT
{
    GetResourceManager().DisableHandheldHids();
    GetResourceManager().GetNpadResourceManager().UpdateHandheld();
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::SetVibrationMasterVolume(float masterVolume) NN_NOEXCEPT
{
    //本体設定の振動設定をNANDに書き込む
    NN_RESULT_DO(::nn::xcd::SetAndSaveVibrationMasterVolume(masterVolume));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetVibrationMasterVolume(::nn::sf::Out<float> outValue) NN_NOEXCEPT
{
    //本体設定の振動設定をNANDから読み込んでくる
    NN_RESULT_DO(::nn::xcd::LoadAndGetVibrationMasterVolume(outValue.GetPointer()));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::BeginPermitVibrationSession(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().BeginPermitVibrationSession(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::EndPermitVibrationSession() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().EndPermitVibrationSession());
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireNfcDeviceUpdateEventHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager().AcquireNfcDeviceUpdateEventHandle(&handle));
    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetNpadsWithNfc(
    ::nn::sf::Out<int64_t> outCount,
    ::nn::sf::OutArray<NpadIdType> outIds) NN_NOEXCEPT
{
    int count = GetResourceManager().GetNpadResourceManager().GetNpadsWithNfc(outIds.GetData(), static_cast<int>(outIds.GetLength()));
    outCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetXcdHandleForNpadWithNfc(
    ::nn::sf::Out<::nn::xcd::DeviceHandle> outHandle,
    NpadIdType id,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::nn::xcd::DeviceHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager().GetXcdHandleForNpadWithNfc(&handle, id, aruid));
    outHandle.Set(handle);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireNfcActivateEventHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue, NpadIdType npadId) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager().AcquireNfcActivateEventHandle(&handle, npadId));
    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ActivateNfc(
    NpadIdType npadId,
    bool activate,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager().ActivateNfc(npadId, activate, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsNfcActivated(::nn::sf::Out<bool> outIsActivated, NpadIdType id) NN_NOEXCEPT
{
    bool isActivated = GetResourceManager().GetNpadResourceManager().IsNfcActivated(id);
    outIsActivated.Set(isActivated);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireIrSensorEventHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue, NpadIdType npadId) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager().AcquireIrSensorEventHandle(&handle, npadId));
    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ActivateIrSensor(
    NpadIdType npadId,
    bool activate,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager().ActivateIrSensor(npadId, activate, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetXcdHandleForNpadWithIrSensor(NpadIdType npadId, ::nn::sf::Out<nn::xcd::DeviceHandle> outHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::nn::xcd::DeviceHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager()
                            .GetXcdHandleForNpadWithIrSensor(npadId, &handle, aruid));
    outHandle.Set(handle);

    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetIrSensorState(NpadIdType id, ::nn::sf::Out<int64_t> outValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    system::IrSensorState state;
    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager()
                            .GetIrSensorState(id, &state, aruid));
    outValue.Set(static_cast<int64_t>(state));

    NN_RESULT_SUCCESS;
}

nn::Result HidSystemServer::ActivateNpadSystem(NpadIdType id) NN_NOEXCEPT
{
    NN_UNUSED(id);
    // 何もしない

    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ApplyNpadSystemCommonPolicy(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().ApplyNpadSystemCommonPolicy(aruid);

    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ApplyNpadSystemCommonPolicyFull(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().ApplyNpadSystemCommonPolicyFull(aruid);

    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetLastActiveNpad(::nn::sf::Out<NpadIdType> outValue) NN_NOEXCEPT
{
    NpadIdType id;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().GetLastActiveNpad(&id));
    outValue.Set(id);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetNpadSystemExtStyle(::nn::sf::Out<int64_t> outMainStyle,
                                                ::nn::sf::Out<int64_t> outSubStyle,
                                                NpadIdType id) NN_NOEXCEPT
{
    system::NpadSystemExtMainStyle mainStyle;
    system::NpadSystemExtSubStyle subStyle;
    GetResourceManager().GetNpadResourceManager().GetNpadSystemExtStyle(&mainStyle, &subStyle, id);
    outMainStyle.Set(static_cast<int64_t>(mainStyle));
    outSubStyle.Set(static_cast<int64_t>(subStyle));
    NN_RESULT_SUCCESS;
}
::nn::Result HidSystemServer::EnableAssigningSingleOnSlSrPress(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().SetAssigningSingleOnSlSrPressMode(true, aruid);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::DisableAssigningSingleOnSlSrPress(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().SetAssigningSingleOnSlSrPressMode(false, aruid);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::SetNpadPlayerLedBlinkingDevice(::nn::applet::AppletResourceUserId aruid, NpadIdType npadId, system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager()
                        .SetNpadPlayerLedBlinkingDevice(aruid, npadId, deviceType);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetNpadFullKeyGripColor(::nn::sf::Out<util::Color4u8Type> outLeftGrip,
                                                      ::nn::sf::Out<util::Color4u8Type> outRightGrip,
                                                      NpadIdType id) NN_NOEXCEPT
{
    nn::util::Color4u8Type leftGrip;
    nn::util::Color4u8Type rightGrip;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().GetGripColor(&leftGrip, &rightGrip, id));
    outLeftGrip.Set(leftGrip);
    outRightGrip.Set(rightGrip);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetMaskedSupportedNpadStyleSet(::nn::sf::Out<NpadStyleSet> outValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NpadStyleSet style;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().GetMaskedSupportedNpadStyleSet(aruid, &style));
    outValue.Set(style);
    NN_RESULT_SUCCESS;
}

::nn::Result CreateHidSystemServerProxy(
    ::nn::sf::SharedPointer<IHidSystemServer>* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    *pOutValue = StaticObject<
        ::nn::sf::UnmanagedServiceObject<IHidSystemServer, HidSystemServer>
        >::Get().GetShared();
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ActivateUniquePad(::nn::applet::AppletResourceUserId aruid,
                                                system::UniquePadId id) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(id);

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // 何もしない
#else
        NN_RESULT_DO(
            GetResourceManager()
            .ActivateAbstractedGenericPad());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireUniquePadConnectionEventHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .AcquireUniquePadConnectionEventHandle(&handle));
    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetUniquePadIds(::nn::sf::Out<::std::int64_t> outGotCount,
                                              const ::nn::sf::OutArray<::nn::hid::system::UniquePadId>& outUniquePadIds) NN_NOEXCEPT
{
    int count = 0;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager().GetIds(
            &count,
            outUniquePadIds.GetData(),
            static_cast<int>(outUniquePadIds.GetLength())));
    outGotCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetUniquePadBluetoothAddress(::nn::sf::Out<::nn::bluetooth::Address> outGotAddress,
                                                           system::UniquePadId id) NN_NOEXCEPT
{
    ::nn::bluetooth::Address address;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetBluetoothAddress(&address, id));
    outGotAddress.Set(address);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::DisconnectUniquePad(system::UniquePadId id) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().GetUniquePadResourceManager()
                                        .DisconnectUniquePad(id));
}

::nn::Result HidSystemServer::GetUniquePadType(::nn::sf::Out<int64_t> outValue, system::UniquePadId id) NN_NOEXCEPT
{
    system::UniquePadType value;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetType(&value, id));
    outValue.Set(static_cast<int64_t>(value));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetUniquePadInterface(::nn::sf::Out<int64_t> outValue,
                                                    system::UniquePadId id) NN_NOEXCEPT
{
    system::UniquePadInterface value;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                        .GetInterface(&value, id));
    outValue.Set(static_cast<int64_t>(value));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetUniquePadSerialNumber(::nn::sf::Out<system::UniquePadSerialNumber> outValue,
                                                       system::UniquePadId id) NN_NOEXCEPT
{
    system::UniquePadSerialNumber value;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                        .GetSerialNumber(&value, id));
    outValue.Set(value);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetUniquePadControllerNumber(::nn::sf::Out<int64_t> outValue,
                                                         system::UniquePadId id) NN_NOEXCEPT
{
    int value;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetControllerNumber(&value, id));
    outValue.Set(static_cast<int64_t>(value));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ListSixAxisSensorHandles(::nn::sf::Out<std::int64_t> outGotCount,
                                                       const ::nn::sf::OutArray<::nn::hid::system::UniqueSixAxisSensorHandle>& outUniqueSixAxisSensorHandles,
                                                       ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    int count = 0;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager().GetUniqueSixAxisSensorHandles(
            &count,
            outUniqueSixAxisSensorHandles.GetData(),
            static_cast<int>(outUniqueSixAxisSensorHandles.GetLength()),
            id));
    outGotCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsSixAxisSensorUserCalibrationSupported(::nn::sf::Out<bool> outIsSupported,
                                                                      ::nn::hid::system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    bool isSupported;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .IsSixAxisSensorUserCalibrationSupported(&isSupported, handle));
    outIsSupported.Set(isSupported);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ResetSixAxisSensorCalibrationValues(::nn::hid::system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .ResetSixAxisSensorCalibrationValues(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::StartSixAxisSensorUserCalibration(::nn::hid::system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .StartSixAxisSensorUserCalibration(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::CancelSixAxisSensorUserCalibration(::nn::hid::system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .CancelSixAxisSensorUserCalibration(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetSixAxisSensorUserCalibrationStage(::nn::sf::Out<int64_t> outValue,
                                                                   system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT
{
    system::SixAxisSensorUserCalibrationStage value;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetSixAxisSensorUserCalibrationStage(&value, handle));
    outValue.Set(static_cast<int64_t>(value));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsSixAxisSensorAccurateUserCalibrationSupported(::nn::sf::Out<bool> outIsSupported,
                                                                              ::nn::hid::system::UniqueSixAxisSensorHandle handle,
                                                                              ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(handle);

    bool isSupported = false;

    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .IsConsoleSixAxisSensorUserCalibrationSupported(&isSupported));

    outIsSupported.Set(isSupported);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::StartSixAxisSensorAccurateUserCalibration(::nn::hid::system::UniqueSixAxisSensorHandle handle,
                                                                        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(handle);

    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .StartConsoleSixAxisSensorUserCalibration());

    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::CancelSixAxisSensorAccurateUserCalibration(::nn::hid::system::UniqueSixAxisSensorHandle handle,
                                                                         ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(handle);

    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .CancelConsoleSixAxisSensorUserCalibration());

    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetSixAxisSensorAccurateUserCalibrationState(::nn::sf::Out<::nn::hid::system::SixAxisSensorAccurateUserCalibrationState> outValue,
                                                                           ::nn::hid::system::UniqueSixAxisSensorHandle handle,
                                                                           ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(handle);

    auto state = ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState();

    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .GetSixAxisSensorAccurateUserCalibrationState(&state));

    outValue.Set(state);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::StartAnalogStickManualCalibration(system::UniquePadId id, int64_t position) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .StartAnalogStickManualCalibration(id, GetAnalogStickPosition(position)));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::RetryCurrentAnalogStickManualCalibrationStage(system::UniquePadId id, int64_t position) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .RetryCurrentAnalogStickManualCalibrationStage(id, GetAnalogStickPosition(position)));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::CancelAnalogStickManualCalibration(system::UniquePadId id, int64_t position) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .CancelAnalogStickManualCalibration(id, GetAnalogStickPosition(position)));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ResetAnalogStickManualCalibration(system::UniquePadId id, int64_t position) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .ResetAnalogStickManualCalibration(id, GetAnalogStickPosition(position)));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetAnalogStickState(::nn::sf::Out<AnalogStickState> outValue,
                                                           system::UniquePadId id,
                                                           int64_t position) NN_NOEXCEPT
{
    AnalogStickState value;
    GetResourceManager().GetUniquePadResourceManager().GetAnalogStickState(&value, id, GetAnalogStickPosition(position));
    outValue.Set(value);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetAnalogStickManualCalibrationStage(::nn::sf::Out<int64_t> outValue,
                                                                            system::UniquePadId id,
                                                                            int64_t position) NN_NOEXCEPT
{
    system::AnalogStickManualCalibrationStage value;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager().GetAnalogStickManualCalibrationStage(&value, id, GetAnalogStickPosition(position)));
    outValue.Set(value);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsAnalogStickButtonPressed(::nn::sf::Out<bool> outValue,
                                                                  system::UniquePadId id,
                                                                  int64_t position) NN_NOEXCEPT
{
    outValue.Set(GetResourceManager().GetUniquePadResourceManager().IsAnalogStickButtonPressed(id, GetAnalogStickPosition(position)));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsAnalogStickInReleasePosition(::nn::sf::Out<bool> outValue,
                                                                      system::UniquePadId id,
                                                                      int64_t position) NN_NOEXCEPT
{
    outValue.Set(GetResourceManager().GetUniquePadResourceManager().IsAnalogStickInReleasePosition(id, GetAnalogStickPosition(position)));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsAnalogStickInCircumference(::nn::sf::Out<bool> outValue,
                                                           system::UniquePadId id,
                                                           int64_t position) NN_NOEXCEPT
{
    outValue.Set(GetResourceManager().GetUniquePadResourceManager().IsAnalogStickInCircumference(id, GetAnalogStickPosition(position)));
    NN_RESULT_SUCCESS;
}
::nn::Result HidSystemServer::ActivateInputDetector(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto& task = GetResourceManager().GetInputDetectorTask();
    NN_RESULT_DO(task.ActivateInputDetector());
    NN_RESULT_DO(task.EnsureInputDetectorAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::NotifyInputDetector(nn::hid::system::InputSourceIdSet id) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetInputDetectorTask()
                                     .NotifyInputDetector(id));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireJoyDetachOnBluetoothOffEventHandle(
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().AcquireJoyDetachOnBluetoothOffEventHandle(&handle, aruid));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquirePlayReportControllerUsageUpdateEvent(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetPlayReportTask()
                            .AcquirePlayReportControllerUsageUpdateEventHandle(&handle));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetPlayReportControllerUsages(::nn::sf::Out<std::int64_t> outGotCount,
                                                            const ::nn::sf::OutArray<::nn::hid::system::PlayReportControllerUsage>& outValues) NN_NOEXCEPT
{
    int count = GetResourceManager().GetPlayReportTask().GetPlayReportControllerUsages(outValues.GetData(), static_cast<int>(outValues.GetLength()));
    outGotCount.Set(count);
    NN_RESULT_SUCCESS;
}


::nn::Result HidSystemServer::AcquirePlayReportRegisteredDeviceUpdateEvent(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetPlayReportTask()
                            .AcquirePlayReportRegisteredDevicesEventHandle(&handle));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetRegisteredDevicesOld(::nn::sf::Out<std::int64_t> outGotCount,
                                                   const ::nn::sf::OutArray<::nn::hid::system::RegisteredDeviceOld>& outValues) NN_NOEXCEPT
{
    int count = GetResourceManager().GetRegisteredDeviceTask().GetRegisteredDevices(outValues.GetData(), static_cast<int>(outValues.GetLength()));
    outGotCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetRegisteredDevices(::nn::sf::Out<std::int64_t> outGotCount,
                                                   const ::nn::sf::OutArray<::nn::hid::system::RegisteredDevice>& outValues) NN_NOEXCEPT
{
    int count = GetResourceManager().GetRegisteredDeviceTask().GetRegisteredDevices(outValues.GetData(), static_cast<int>(outValues.GetLength()));
    outGotCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetConnectableRegisteredDevices(::nn::sf::Out<std::int64_t> outGotCount,
                                                              const ::nn::sf::OutArray<::nn::hid::system::RegisteredDevice>& outValues) NN_NOEXCEPT
{
    // 現状は RegisteredDevice との差異がないので同じものを返す。
    // 登録はできるが接続ができないみたいな概念ができた場合は実装分岐が必要
    int count = GetResourceManager().GetRegisteredDeviceTask().GetRegisteredDevices(outValues.GetData(), static_cast<int>(outValues.GetLength()));
    outGotCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsUsbFullKeyControllerEnabled(::nn::sf::Out<bool> outIsEnabled) NN_NOEXCEPT
{
    bool isEnabled;
    NN_RESULT_DO(GetResourceManager().GetAbstractedPadXcdTask()
                                     .IsUsbFullKeyControllerEnabled(&isEnabled));
    outIsEnabled.Set(static_cast<bool>(isEnabled));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetAbstractedPadXcdTask()
                                     .EnableUsbFullKeyController(enabled));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsUsbConnected(::nn::sf::Out<bool> outIsConnected, ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    bool isConnected;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .IsUsbConnected(&isConnected, id));
    outIsConnected.Set(static_cast<bool>(isConnected));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::InitializeFirmwareUpdate() NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .ActivateFirmwareUpdate());
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::StartFirmwareUpdate(
    ::nn::sf::Out<::nn::hid::system::FirmwareUpdateDeviceHandle> outHandle,
    ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    ::nn::hid::system::FirmwareUpdateDeviceHandle deviceHandle;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .StartFirmwareUpdate(&deviceHandle, id));
    outHandle.Set(deviceHandle);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AbortFirmwareUpdate() NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .AbortFirmwareUpdate());
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetFirmwareVersion(
    ::nn::sf::Out<::nn::hid::system::FirmwareVersion> outValue,
    ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    ::nn::hid::system::FirmwareVersion version;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .GetFirmwareVersion(&version, id));
    outValue.Set(version);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetAvailableFirmwareVersion(
    ::nn::sf::Out<::nn::hid::system::FirmwareVersion> outValue,
    ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    ::nn::hid::system::FirmwareVersion version;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .GetDestinationFirmwareVersion(&version, id));
    outValue.Set(version);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsFirmwareUpdateAvailable(
    ::nn::sf::Out<bool> outIsAvailable,
    ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    bool isAvailable;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .IsFirmwareUpdateAvailable(&isAvailable, id));
    outIsAvailable.Set(isAvailable);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::CheckFirmwareUpdateRequired(
    ::nn::sf::Out<int64_t> outReason,
    ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    ::nn::hid::system::FirmwareUpdateRequiredReason reason;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .CheckFirmwareUpdateRequired(&reason, id));
    outReason.Set(reason);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetFirmwareUpdateState(
    ::nn::sf::Out<::nn::hid::system::FirmwareUpdateState> outState,
    ::nn::hid::system::FirmwareUpdateDeviceHandle handle) NN_NOEXCEPT
{
    ::nn::hid::system::FirmwareUpdateState state;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .GetFirmwareUpdateState(&state, handle));
    outState.Set(state);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::SetFirmwareHotfixUpdateSkipEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .SetFirmwareHotfixUpdateSkipEnabled(isEnabled));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireConnectionTriggerTimeoutEvent(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetRegisteredDeviceTask()
                            .AcquireConnectionTriggerTimeoutEvent(&handle));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::SendConnectionTrigger(nn::bluetooth::Address address) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().GetRegisteredDeviceTask()
                            .SendConnectionTrigger(address)
    );
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetAllowedBluetoothLinksCount(::nn::sf::Out<std::int64_t> outGotCount) NN_NOEXCEPT
{
    outGotCount.Set(static_cast<int64_t>(GetResourceManager().GetUniquePadResourceManager().GetAllowedBluetoothLinksCount()));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireDeviceRegisteredEventForControllerSupport(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetRegisteredDeviceTask()
                            .AcquireDeviceRegisteredEventForControllerSupport(&handle));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetUniquePadsFromNpad(::nn::sf::Out<int64_t> outGotCount, ::nn::sf::OutArray<::nn::hid::system::UniquePadId> outIds, NpadIdType id) NN_NOEXCEPT
{
    int count = GetResourceManager().GetNpadResourceManager().GetUniquePadsFromNpad(outIds.GetData(), static_cast<int>(outIds.GetLength()), id);
    outGotCount.Set(static_cast<int64_t>(count));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::ActivateAudioControl() NN_NOEXCEPT
{
    auto& task = GetResourceManager().GetAudioControlTask();
    NN_RESULT_DO(task.ActivateAudioControl());
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::AcquireAudioControlEventHandle(
    ::nn::sf::Out<nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().GetAudioControlTask()
                            .AcquireAudioControlEventHandle(&handle));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::GetAudioControlStates(
    ::nn::sf::Out<int64_t> outCount,
    const nn::sf::OutArray<system::AudioControlState>& outStates) NN_NOEXCEPT
{
    int count = 0;
    NN_RESULT_DO(
        GetResourceManager().GetAudioControlTask()
                            .GetAudioControlStates(
                                &count,
                                outStates.GetData(),
                                static_cast<int>(outStates.GetLength() )));
    outCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::DeactivateAudioControl() NN_NOEXCEPT
{
    auto& task = GetResourceManager().GetAudioControlTask();
    NN_RESULT_DO(task.DeactivateAudioControl());
    NN_RESULT_SUCCESS;
}

::nn::Result HidSystemServer::IsHandheldButtonPressedOnConsoleMode(::nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    auto isPressed = GetResourceManager().GetNpadResourceManager().IsHandheldButtonPressedOnConsoleMode();
    outValue.Set(isPressed);
    NN_RESULT_SUCCESS;
}

nn::Result HidSystemServer::GetHidbusSystemServiceObject(nn::sf::Out<nn::sf::SharedPointer<nn::hidbus::IHidbusSystemServer>> outService) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    ::nn::sf::SharedPointer<
        ::nn::hidbus::IHidbusSystemServer> pHidbusSystemServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hidbus::server::CreateHidbusSystemServerProxy(&pHidbusSystemServer));
    outService.Set(pHidbusSystemServer);
    NN_RESULT_SUCCESS;
#elif defined( NN_BUILD_CONFIG_OS_WIN )
    // Win 環境ではこの API は呼ばれない
    NN_RESULT_SUCCESS;
    NN_UNUSED(outService);
#endif
}


}}} // namespace nn::hid::detail
