﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <atomic>
#include <cstring>
#include <iterator>
#include <limits>
#include <mutex>
#include <new>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_AbstractedPad.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/hid/system/hid_CaptureButton.h>
#include <nn/hid/system/hid_HomeButton.h>
#include <nn/hid/system/hid_NpadCommon.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_FileMappingObject-os.win.h"
#include "hid_HidShellPortFile-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< HidShell のファイルマッピングオブジェクトの名前です。
const char* const FileMappingObjectName = "NINTENDO_HIDSHELL";

//!< HidShell の入出力ポートの数です。
const int IoPortCount = 32;

//!< Xpad の入出力ポートの数です。
const int XpadPortCount = 8;

//!< AbstractedPad の入出力ポートの数です。
const int AbstractedPadPortCount = 8;

//!< 入出力ポートのオフセットを表す列挙体です。
enum class IoPortOffset : int32_t
{
    In,     //!< 入力ポート
    Out,    //!< 出力ポート
};

//!< 入出力ポートのモードを表す列挙体です。
enum class IoPortMapMode : int32_t
{
    None,       //!< 無し
    AutoPilot,  //!< 自動操作中
};

//!< 撮影ボタンのアドレスマップを表す構造体です。
struct CaptureButtonMap final
{
    IoPortMapMode mode;                             //!< 入出力ポートのモード
    int32_t _padding;
    ::nn::hid::system::CaptureButtonSet buttons;    //!< デジタルボタンの状態
};

//!< DebugPad のアドレスマップを表す構造体です。
struct DebugPadMap final
{
    IoPortMapMode mode;                 //!< 入出力ポートのモード
    int32_t _padding;
    DebugPadAttributeSet attributes;    //!< 入力状態の属性
    DebugPadButtonSet buttons;          //!< デジタルボタンの状態
    AnalogStickState analogStickL;      //!< 左アナログスティックの状態
    AnalogStickState analogStickR;      //!< 右アナログスティックの状態
};

//!< ホームボタンのアドレスマップを表す構造体です。
struct HomeButtonMap final
{
    IoPortMapMode mode;                         //!< 入出力ポートのモード
    int32_t _padding;
    ::nn::hid::system::HomeButtonSet buttons;   //!< デジタルボタンの状態
};

//!< TouchScreen のアドレスマップを表す構造体です。
struct TouchScreenMap final
{
    IoPortMapMode mode;                     //!< 入出力ポートのモード
    int32_t _padding1;
    int32_t count;                          //!< タッチの数
    int32_t _padding2;
    TouchState touches[TouchStateCountMax]; //!< タッチ
};

//!< Xpad のアドレスマップを表す構造体です。
struct XpadMap final
{
    IoPortMapMode mode;             //!< 入出力ポートのモード
    int32_t powerState;             //!< 電源状態
    int32_t batteryLevel;           //!< バッテリー残量
    BasicXpadButtonSet buttons;     //!< デジタルボタンの状態
    AnalogStickState analogStickL;  //!< 左アナログスティックの状態
    AnalogStickState analogStickR;  //!< 右アナログスティックの状態
};

//!< AbstractedPad の電源状態です。
struct PowerInfo final
{
    uint8_t isPowered;      //!< 給電状態
    uint8_t isCharging;     //!< 充電状態
    uint8_t padding[1];
    uint8_t batteryLevel;   //!< 電池残量
};

//!< AbstractedPad のアドレスマップを表す構造体です。
struct AbstractedPadMap final
{
    IoPortMapMode                    mode;           //!< 入出力ポートのモード
    uint8_t                          deviceType;     //!< デバイスタイプ
    debug::AbstractedPadAttributeSet attributes;     //!< AbstractedPad の属性
    system::ControllerColor          color;          //!< デバイス色
    uint8_t                          interfaceType;  //!< デバイスインターフェース
    PowerInfo                        powerInfo;      //!< 電源状態
    AbstractedPadButtonSet           buttons;        //!< digitalボタンの状態
    AnalogStickState                 analogStickL;   //!< 左スティックの状態
    AnalogStickState                 analogStickR;   //!< 右スティックの状態
};

//!< 入出力ポートのアドレスマップを表す構造体です。
struct IoPortMap final
{
    XpadMap xpadMaps[XpadPortCount];    //!< Xpad
    TouchScreenMap touchScreenMap;      //!< TouchScreen
    HomeButtonMap homeButtonMap;        //!< ホームボタン
    CaptureButtonMap captureButtonMap;  //!< 撮影ボタン
    DebugPadMap debugPadMap;            //!< DebugPad
    AbstractedPadMap abstractedPadMap[
        AbstractedPadPortCount];        //!< AbstractedPad
};

//!< HidShell のファイルマッピングオブジェクトを返します。
FileMappingObject& GetFileMappingObject() NN_NOEXCEPT;

//!< 入出力ポートを取得します。
::nn::Result GetIoPortMap(
    FileMappingObject& object, IoPortMap** ppOutValue, IoPortOffset offset
    ) NN_NOEXCEPT;

//!< Npad のデバイスの種類を返します。
int GetNpadDeviceType(::nn::hid::system::NpadDeviceTypeSet value) NN_NOEXCEPT;

} // namespace

::nn::Result GetHidShellDebugPadState(
    bool* pOutFlag,
    ::nn::hid::debug::DebugPadAutoPilotState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFlag);
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    FileMappingObject& object = GetFileMappingObject();

    NN_RESULT_DO(object.Lock());

    NN_UTIL_SCOPE_EXIT
    {
        object.Unlock();
    };

    IoPortMap* pOutMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pOutMap, IoPortOffset::Out));

    const bool outputExists = pOutMap->debugPadMap.buttons.IsAnyOn();

    IoPortMap* pInMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pInMap, IoPortOffset::In));

    const DebugPadMap& map = pInMap->debugPadMap;

    if (map.mode != IoPortMapMode::AutoPilot || outputExists)
    {
        *pOutFlag = false;
    }
    else
    {
        *pOutFlag = true;

        pOutState->attributes = map.attributes;
        pOutState->buttons = map.buttons;
        pOutState->analogStickL = map.analogStickL;
        pOutState->analogStickR = map.analogStickR;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SetHidShellDebugPadState(
    const ::nn::hid::debug::DebugPadAutoPilotState& state) NN_NOEXCEPT
{
    FileMappingObject& object = GetFileMappingObject();

    NN_RESULT_DO(object.Lock());

    NN_UTIL_SCOPE_EXIT
    {
        object.Unlock();
    };

    IoPortMap* pOutMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pOutMap, IoPortOffset::Out));

    DebugPadMap& map = pOutMap->debugPadMap;

    map.attributes = state.attributes;
    map.buttons = state.buttons;
    map.analogStickL = state.analogStickL;
    map.analogStickR = state.analogStickR;

    NN_RESULT_SUCCESS;
}

::nn::Result GetHidShellTouchScreenState(
    bool* pOutFlag,
    ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
        >* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFlag);
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    FileMappingObject& object = GetFileMappingObject();

    NN_RESULT_DO(object.Lock());

    NN_UTIL_SCOPE_EXIT
    {
        object.Unlock();
    };

    IoPortMap* pOutMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pOutMap, IoPortOffset::Out));

    const bool outputExists = (pOutMap->touchScreenMap.count > 0);

    IoPortMap* pInMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pInMap, IoPortOffset::In));

    const TouchScreenMap& map = pInMap->touchScreenMap;

    if (map.mode != IoPortMapMode::AutoPilot || outputExists)
    {
        *pOutFlag = false;
    }
    else
    {
        *pOutFlag = true;

        pOutState->count = map.count;

        ::std::copy(
            ::std::begin(map.touches), ::std::end(map.touches),
            ::std::begin(pOutState->touches));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SetHidShellTouchScreenState(
    const ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
        >& state) NN_NOEXCEPT
{
    FileMappingObject& object = GetFileMappingObject();

    NN_RESULT_DO(object.Lock());

    NN_UTIL_SCOPE_EXIT
    {
        object.Unlock();
    };

    IoPortMap* pOutMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pOutMap, IoPortOffset::Out));

    TouchScreenMap& map = pOutMap->touchScreenMap;

    map.count = state.count;

    ::std::copy(
        ::std::begin(state.touches), ::std::end(state.touches),
        ::std::begin(map.touches));

    NN_RESULT_SUCCESS;
}

bool IsValidHidShellXpadPort(int port) NN_NOEXCEPT
{
    return 0 <= port && port < XpadPortCount;
}

::nn::Result GetHidShellXpadState(
    bool* pOutFlag,
    ::nn::hid::debug::BasicXpadAutoPilotState* pOutState,
    int port) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFlag);
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES(IsValidHidShellXpadPort(port));

    FileMappingObject& object = GetFileMappingObject();

    NN_RESULT_DO(object.Lock());

    NN_UTIL_SCOPE_EXIT
    {
        object.Unlock();
    };

    IoPortMap* pOutMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pOutMap, IoPortOffset::Out));

    const bool outputExists = pOutMap->xpadMaps[port].buttons.IsAnyOn();

    IoPortMap* pInMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pInMap, IoPortOffset::In));

    const XpadMap& map = pInMap->xpadMaps[port];

    if (map.mode != IoPortMapMode::AutoPilot || outputExists)
    {
        *pOutFlag = false;
    }
    else
    {
        *pOutFlag = true;

        pOutState->powerState = map.powerState;
        pOutState->batteryLevel = map.batteryLevel;
        pOutState->buttons = map.buttons;
        pOutState->analogStickL = map.analogStickL;
        pOutState->analogStickR = map.analogStickR;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SetHidShellXpadState(
    int port,
    const ::nn::hid::debug::BasicXpadAutoPilotState& state) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidHidShellXpadPort(port));

    FileMappingObject& object = GetFileMappingObject();

    NN_RESULT_DO(object.Lock());

    NN_UTIL_SCOPE_EXIT
    {
        object.Unlock();
    };

    IoPortMap* pOutMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pOutMap, IoPortOffset::Out));

    XpadMap& map = pOutMap->xpadMaps[port];

    map.powerState = state.powerState;
    map.batteryLevel = state.batteryLevel;
    map.buttons = state.buttons;
    map.analogStickL = state.analogStickL;
    map.analogStickR = state.analogStickR;

    NN_RESULT_SUCCESS;
}

bool IsValidHidShellAbstractedPadPort(int port) NN_NOEXCEPT
{
    return 0 <= port && port < AbstractedPadPortCount;
}

::nn::Result GetHidShellAbstractedPadState(
    bool* pOutFlag,
    ::nn::hid::debug::AbstractedPadState* pOutState,
    int port) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFlag);
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES(IsValidHidShellAbstractedPadPort(port));

    FileMappingObject& object = GetFileMappingObject();

    NN_RESULT_DO(object.Lock());

    NN_UTIL_SCOPE_EXIT
    {
        object.Unlock();
    };

    IoPortMap* pOutMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pOutMap, IoPortOffset::Out));

    const bool outputExists = pOutMap->abstractedPadMap[port].buttons.IsAnyOn();

    IoPortMap* pInMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pInMap, IoPortOffset::In));

    const AbstractedPadMap& map = pInMap->abstractedPadMap[port];

    if (map.mode != IoPortMapMode::AutoPilot || outputExists)
    {
        *pOutFlag = false;
    }
    else
    {
        *pOutFlag = true;

        pOutState->deviceType.Reset();
        pOutState->deviceType.Set(map.deviceType & 0x1f);
        pOutState->attributes = map.attributes;
        pOutState->color = map.color;
        pOutState->interfaceType = map.interfaceType;
        pOutState->powerInfo.isPowered = map.powerInfo.isPowered == 1;
        pOutState->powerInfo.isCharging = map.powerInfo.isCharging == 1;
        pOutState->powerInfo.batteryLevel = map.powerInfo.batteryLevel;
        pOutState->buttons = map.buttons;
        pOutState->analogStickL = map.analogStickL;
        pOutState->analogStickR = map.analogStickR;
        pOutState->sixAxisSensorState = SixAxisSensorState();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SetHidShellAbstractedPadState(
    int port,
    const ::nn::hid::debug::AbstractedPadState& state) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidHidShellAbstractedPadPort(port));

    FileMappingObject& object = GetFileMappingObject();

    NN_RESULT_DO(object.Lock());

    NN_UTIL_SCOPE_EXIT
    {
        object.Unlock();
    };

    IoPortMap* pOutMap = nullptr;

    NN_RESULT_DO(GetIoPortMap(object, &pOutMap, IoPortOffset::Out));

    AbstractedPadMap& map = pOutMap->abstractedPadMap[port];

    map.deviceType = static_cast<uint8_t>(GetNpadDeviceType(state.deviceType));
    map.attributes = state.attributes;
    map.color = state.color;
    map.interfaceType = state.interfaceType;
    map.powerInfo.isPowered = state.powerInfo.isPowered;
    map.powerInfo.isCharging = state.powerInfo.isCharging;
    map.powerInfo.batteryLevel =
        static_cast<uint8_t>(state.powerInfo.batteryLevel);
    map.buttons = state.buttons;
    map.analogStickL = state.analogStickL;
    map.analogStickR = state.analogStickR;

    NN_RESULT_SUCCESS;
}

namespace {

struct IoPortWholeMap final
{
    int64_t ioPortIds[IoPortCount];
    IoPortMap ioPortMaps[2 * (IoPortCount + 1)];
};

FileMappingObject& GetFileMappingObject() NN_NOEXCEPT
{
    static ::nn::util::TypedStorage<FileMappingObject,
                             sizeof(FileMappingObject),
                         NN_ALIGNOF(FileMappingObject)> s_Storage = {};

    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

    if (!s_IsInitialized)
    {
        static ::nn::os::SdkMutexType s_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialized)
        {
            new (&::nn::util::Get(s_Storage)) FileMappingObject(
                sizeof(IoPortWholeMap), FileMappingObjectName);

            s_IsInitialized = true;
        }
    }

    return ::nn::util::Get(s_Storage);
}

::nn::Result GetIoPortWholeMap(
    FileMappingObject& object, IoPortWholeMap** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    auto pointer = LPVOID();

    auto exists = true;

    NN_RESULT_DO(object.GetAddress(&pointer, &exists));

    auto& map = *static_cast<IoPortWholeMap*>(pointer);

    if (!exists)
    {
        ::std::memset(&map, 0, sizeof(map));

        for (int64_t& ioPortId : map.ioPortIds)
        {
            ioPortId = ::std::numeric_limits<int64_t>::min();
        }
    }

    *ppOutValue = &map;

    NN_RESULT_SUCCESS;
}

::nn::Result GetIoPortMap(
    FileMappingObject& object, IoPortMap** ppOutValue, IoPortOffset offset
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    IoPortWholeMap* pMap = nullptr;

    NN_RESULT_DO(GetIoPortWholeMap(object, &pMap));

    *ppOutValue = &pMap->ioPortMaps[static_cast<int32_t>(offset)];

    NN_RESULT_SUCCESS;
}

int GetNpadDeviceType(::nn::hid::system::NpadDeviceTypeSet value) NN_NOEXCEPT
{
    if (value.Test<::nn::hid::system::DeviceType::SwitchProController>())
    {
        return ::nn::hid::system::DeviceType::SwitchProController::Index;
    }

    if (value.Test<::nn::hid::system::DeviceType::Handheld>())
    {
        return ::nn::hid::system::DeviceType::Handheld::Index;
    }

    if (value.Test<::nn::hid::system::DeviceType::HandheldJoyLeft>())
    {
        return ::nn::hid::system::DeviceType::HandheldJoyLeft::Index;
    }

    if (value.Test<::nn::hid::system::DeviceType::HandheldJoyRight>())
    {
        return ::nn::hid::system::DeviceType::HandheldJoyRight::Index;
    }

    if (value.Test<::nn::hid::system::DeviceType::JoyConLeft>())
    {
        return ::nn::hid::system::DeviceType::JoyConLeft::Index;
    }

    if (value.Test<::nn::hid::system::DeviceType::JoyConRight>())
    {
        return ::nn::hid::system::DeviceType::JoyConRight::Index;
    }

    if (value.Test<::nn::hid::system::DeviceType::Palma>())
    {
        return ::nn::hid::system::DeviceType::Palma::Index;
    }

    return ::nn::hid::system::DeviceType::Unknown::Index;
}

} // namespace

}}} // namespace nn::hid::detail
