﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_HidServer.h"
#include "hid_LockableMutexType.h"
#include "hid_NpadIndicatorPattern.h"
#include "hid_Settings.h"
#include "hid_SixAxisSensorHandle.h"
#include "hid_StaticObject.h"
#include "hid_XpadId.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "hid_ResourceManager-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "hid_ResourceManager-os.win.h"
#endif

namespace nn { namespace hid { namespace detail {

namespace {

//!< 共有オブジェクト用ヒープメモリを扱うクラスです。
typedef ::nn::sf::ExpHeapStaticAllocator<16 * 1024, IAppletResource> Allocator;

//!< os ハンドルを管理するか否かを表す値
const bool NeedsToBeManaged =
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    false;
#else
    false;
#endif

} // namespace

AppletResource::AppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    : m_Aruid(aruid)
    , m_IsOwner(false)
{
    // 何もしない
}

AppletResource::AppletResource(AppletResource&& other) NN_NOEXCEPT
    : m_Aruid(other.m_Aruid)
    , m_IsOwner(other.m_IsOwner)
{
    other.m_IsOwner = false;
}

AppletResource::~AppletResource() NN_NOEXCEPT
{
    if (m_IsOwner)
    {
        if (m_Aruid == ::nn::applet::AppletResourceUserId::GetInvalidId())
        {
            GetResourceManager().DeactivateSharedAppletResource();
            GetResourceManager().GetNpadResourceManager()
                                .DeactivateNpadSharedAppletResource();
        }
        else
        {
            GetResourceManager().ResetAppletResourceEntry(m_Aruid);
            GetResourceManager().GetNpadResourceManager()
                                .DeactivateNpadAppletResourceEntry(m_Aruid);
        }
    }
}

::nn::Result AppletResource::Initialize() NN_NOEXCEPT
{
    if (m_Aruid == ::nn::applet::AppletResourceUserId::GetInvalidId())
    {
        NN_RESULT_DO(
            GetResourceManager().ActivateSharedAppletResource());
        NN_RESULT_DO(
            GetResourceManager().GetNpadResourceManager()
                                .ActivateNpadSharedAppletResource());
    }
    else
    {
        NN_RESULT_DO(
            GetResourceManager().SetAppletResourceEntry(m_Aruid));
        NN_RESULT_DO(
            GetResourceManager().GetNpadResourceManager()
                                .ActivateNpadAppletResourceEntry(m_Aruid));
    }

    m_IsOwner = true;

    NN_RESULT_SUCCESS;
}

::nn::Result AppletResource::GetSharedMemoryHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    auto handle = ::nn::os::NativeHandle();
    NN_RESULT_DO(
        GetResourceManager().GetAppletResourceSharedMemoryHandle(
            &handle, m_Aruid));
    outValue.Set(::nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

ActiveVibrationDeviceList::ActiveVibrationDeviceList() NN_NOEXCEPT
    : m_Mutex(false), m_ActiveVibrationDeviceCount(0)
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    for (int i = 0; i < ActiveVibrationDeviceCountMax; i++)
    {
        m_HandleList[i]._storage = 0;
    }
}

ActiveVibrationDeviceList::ActiveVibrationDeviceList(ActiveVibrationDeviceList&& other) NN_NOEXCEPT
    : m_Mutex(false), m_ActiveVibrationDeviceCount(other.m_ActiveVibrationDeviceCount)
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    for (int i = 0; i < ActiveVibrationDeviceCountMax; i++)
    {
        m_HandleList[i] = other.m_HandleList[i];
    }
}

ActiveVibrationDeviceList::~ActiveVibrationDeviceList() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    for (int i = 0; i < ActiveVibrationDeviceCountMax; i++)
    {
        VibrationDeviceHandle handle = m_HandleList[i];

        // 参照カウントを減らす
        if (GetResourceManager().AssertValidVibrationDeviceHandle(handle).IsSuccess())
        {
            (void)(GetResourceManager().GetVibrator(handle).DecreaseReferenceCount());
        }

        m_HandleList[i]._storage = 0;
    }
}

::nn::Result ActiveVibrationDeviceList::ActivateVibrationDevice(
    ::nn::hid::VibrationDeviceHandle handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_RESULT_DO(GetResourceManager().AssertValidVibrationDeviceHandle(handle));

    for (int i = 0; i < m_ActiveVibrationDeviceCount; i++)
    {
        if (m_HandleList[i]._storage == handle._storage)
        {
            // 既にリストに追加されていたら何もしない
            NN_RESULT_SUCCESS;
        }
    }

    if (m_ActiveVibrationDeviceCount < ActiveVibrationDeviceCountMax)
    {
        // 参照カウントを増やした上でリストに追加
        NN_RESULT_DO(GetResourceManager().GetVibrator(handle).IncreaseReferenceCount());
        m_HandleList[m_ActiveVibrationDeviceCount] = handle;
        m_ActiveVibrationDeviceCount++;
        NN_RESULT_SUCCESS;
    }
    else
    {
        // リストの容量オーバー
        NN_RESULT_THROW(nn::hid::ResultVibrationDeviceHandleDeviceIdxOutOfRange());
    }
}

HidServer::HidServer() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result HidServer::CreateAppletResource(
    ::nn::sf::Out<::nn::sf::SharedPointer<IAppletResource>> outValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    AppletResource resource(aruid);

    NN_RESULT_DO(resource.Initialize());

    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

    if (!s_IsInitialized)
    {
        static LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialized)
        {
            Allocator::Initialize(::nn::lmem::CreationOption_NoOption);

            s_IsInitialized = true;
        }
    }

    outValue.Set(
        ::nn::sf::ObjectFactory<Allocator::Policy>::CreateSharedEmplaced<
            IAppletResource, AppletResource>(::std::move(resource)));

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateDebugPad(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(GetResourceManager().GetDebugPadTask().ActivateDebugPad());
    }

    NN_RESULT_DO(GetResourceManager().GetDebugPadTask()
                                     .EnsureDebugPadAppletResource(aruid));

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateKeyboard(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(GetResourceManager().GetKeyboardTask().ActivateKeyboard());
    }

    NN_RESULT_DO(GetResourceManager().GetKeyboardTask()
                                     .EnsureKeyboardAppletResource(aruid));

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateMouse(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(GetResourceManager().GetMouseTask().ActivateMouse());
    }

    NN_RESULT_DO(GetResourceManager().GetMouseTask()
                                     .EnsureMouseAppletResource(aruid));

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateTouchScreen(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            GetResourceManager().GetTouchScreenTask().ActivateTouchScreen());
    }

    NN_RESULT_DO(GetResourceManager().GetTouchScreenTask()
                                     .EnsureTouchScreenAppletResource(aruid));

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateConsoleSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                         .ActivateConsoleSixAxisSensor());
    }

    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .EnsureConsoleSixAxisSensorAppletResource(aruid));

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateSevenSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // TORIAEZU
    ActivateConsoleSixAxisSensor(aruid);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::AcquireXpadIdEventHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
    uint64_t clientId) NN_NOEXCEPT
{
    ::nn::os::NativeHandle handle;
    NN_RESULT_DO(
        GetResourceManager().AcquireXpadIdEventHandle(&handle, clientId));
    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ReleaseXpadIdEventHandle(uint64_t clientId) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().ReleaseXpadIdEventHandle(clientId));
}

::nn::Result HidServer::ActivateXpad(
    ::nn::applet::AppletResourceUserId aruid, BasicXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidBasicXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    auto& task = GetResourceManager().GetXpadTask();
    task.EnableXpadSampling();
    NN_RESULT_DO(task.EnsureBasicXpadAppletResource(aruid, playerNumber));
#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    task.EnableBasicXpadSampling();
    NN_RESULT_DO(task.ActivateBasicXpad());
    NN_RESULT_DO(task.EnsureBasicXpadAppletResource(aruid));
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetXpadIds(
    ::nn::sf::Out<int64_t> outCount,
    const ::nn::sf::OutArray<BasicXpadId>& outXpadIds) NN_NOEXCEPT
{
    int count = 0;
    NN_RESULT_DO(
        GetResourceManager().GetXpadIds(
            &count,
            outXpadIds.GetData(),
            static_cast<int>(outXpadIds.GetLength())));
    outCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateJoyXpad(JoyXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidJoyXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    auto& task = GetResourceManager().GetXpadTask();
    task.EnableXpadSampling();
#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    task.EnableJoyXpadSampling();
    NN_RESULT_DO(task.ActivateJoyXpad());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetJoyXpadLifoHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
    JoyXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidJoyXpadPlayerNumber(playerNumber));
    ::nn::os::NativeHandle handle;
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    auto& task = GetResourceManager().GetXpadTask();
    NN_RESULT_DO(task.GetJoyXpadLifoHandle(&handle, playerNumber));
#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    NN_RESULT_DO(task.GetJoyXpadLifoHandle(&handle));
#endif
    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetJoyXpadIds(
    ::nn::sf::Out<int64_t> outGotCount,
    const ::nn::sf::OutArray<JoyXpadId>& outXpadIds) NN_NOEXCEPT
{
    int count = 0;
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    NN_RESULT_DO(
        GetResourceManager().GetJoyXpadIds(
            &count,
            outXpadIds.GetData(),
            static_cast<int>(outXpadIds.GetLength())));
#else
    NN_RESULT_DO(
        GetResourceManager().GetXpadIds(
            &count,
            outXpadIds.GetData(),
            static_cast<int>(outXpadIds.GetLength())));
#endif
    outGotCount.Set(count);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateGesture(
    ::nn::applet::AppletResourceUserId aruid, int32_t version) NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            GetResourceManager().GetGestureTask().ActivateGesture());
    }

    NN_RESULT_DO(
        GetResourceManager().GetGestureTask()
                            .EnsureGestureAppletResource(aruid, version));

    NN_RESULT_SUCCESS;
}

nn::Result HidServer::SetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet style) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().SetSupportedNpadStyleSet(aruid, style));

    NN_RESULT_SUCCESS;
}

nn::Result HidServer::GetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid,::nn::sf::Out<NpadStyleSet> outValue) NN_NOEXCEPT
{
    NpadStyleSet style;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().GetSupportedNpadStyleSet(aruid, &style));

    outValue.Set(style);
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::SetSupportedNpadIdType(::nn::applet::AppletResourceUserId aruid, const ::nn::sf::InArray<::nn::hid::NpadIdType>& ids) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().SetSupportedNpadIdType(aruid, ids.GetData(), static_cast<int>(ids.GetLength())));
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::ActivateNpad(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(ActivateNpadWithRevision(aruid, NpadApplicationRevision_0));
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::ActivateNpadWithRevision(::nn::applet::AppletResourceUserId aruid, int32_t revision) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().SetNpadApplicationRevision(aruid, static_cast<NpadApplicationRevision>(revision));

    NN_RESULT_DO(
        GetResourceManager()
        .GetNpadResourceManager()
        .EnsureNpadAppletResource(aruid));
#ifndef NN_BUILD_CONFIG_OS_HORIZON
    NN_RESULT_DO(
        GetResourceManager()
        .ActivateAbstractedGenericPad());
#endif

    NN_RESULT_SUCCESS;
}

nn::Result HidServer::DeactivateNpad(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_RESULT_SUCCESS;
}

nn::Result HidServer::AcquireNpadStyleSetUpdateEventHandle(::nn::applet::AppletResourceUserId aruid,
                                                ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
                                                NpadIdType id,
                                                uint64_t clientId) NN_NOEXCEPT
{
    NN_UNUSED(clientId);
    ::nn::os::NativeHandle handle;

    NN_RESULT_DO(
        GetResourceManager().GetNpadResourceManager().AcquireStyleUpdateEventHandle(aruid, &handle, id));

    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::DisconnectNpad(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().Disconnect(aruid, id);
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::GetPlayerLedPattern(::nn::sf::Out<uint64_t> outValue,
                                          NpadIdType id) NN_NOEXCEPT
{
    outValue.Set(static_cast<uint64_t>(GetIndicatorPattern(id, false)));
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::SetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid, std::int64_t holdType) NN_NOEXCEPT
{
    NpadJoyHoldType joyHoldType;

    switch (holdType)
    {
    case NpadJoyHoldType_Horizontal:
        joyHoldType = NpadJoyHoldType_Horizontal;
        break;
    case NpadJoyHoldType_Vertical:
        joyHoldType = NpadJoyHoldType_Vertical;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().SetNpadJoyHoldType(aruid, joyHoldType));
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::GetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid, ::nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
{
    NpadJoyHoldType holdType;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().GetNpadJoyHoldType(aruid, &holdType));
    outValue.Set(static_cast<int64_t>(holdType));
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::SetNpadJoyAssignmentModeSingleByDefault(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().SetNpadJoyAssignmentModeSingleByDefault(aruid, id);
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::SetNpadJoyAssignmentModeSingle(::nn::applet::AppletResourceUserId aruid, NpadIdType id, int64_t type) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().SetNpadJoyAssignmentModeSingle(aruid,id, static_cast<NpadJoyDeviceType>(type));
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::SetNpadJoyAssignmentModeSingleWithDestination(::nn::sf::Out<bool> outIsAssigned,
                                                                    ::nn::sf::Out<NpadIdType> outId,
                                                                    ::nn::applet::AppletResourceUserId aruid, NpadIdType id, int64_t type) NN_NOEXCEPT
{
    NpadIdType destination;
    auto isAssigned = GetResourceManager().GetNpadResourceManager().SetNpadJoyAssignmentModeSingleWithDestination(&destination, aruid,id, static_cast<NpadJoyDeviceType>(type));
    outIsAssigned.Set(isAssigned);
    outId.Set(destination);
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::SetNpadJoyAssignmentModeDual(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().SetNpadJoyAssignmentModeDual(aruid, id);
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::MergeSingleJoyAsDualJoy(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().MergeSingleJoyAsDualJoy(aruid, id1, id2));
    NN_RESULT_SUCCESS;
}

nn::Result HidServer::SwapNpadAssignment(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().SwapNpadAssignment(aruid, id1, id2));

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::StartLrAssignmentMode(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().EnableLrAssignmentMode(aruid);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::StopLrAssignmentMode(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    GetResourceManager().GetNpadResourceManager().DisableLrAssignmentMode(aruid);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid,
                                                      ::std::int64_t mode) NN_NOEXCEPT
{
    NpadHandheldActivationMode activationMode;

    switch (mode)
    {
    case NpadHandheldActivationMode_Single:
        activationMode = NpadHandheldActivationMode_Single;
        break;
    case NpadHandheldActivationMode_Dual:
        activationMode = NpadHandheldActivationMode_Dual;
        break;
    case NpadHandheldActivationMode_None:
        activationMode = NpadHandheldActivationMode_None;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().SetNpadHandheldActivationMode(aruid, activationMode));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid,
                                                      ::nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
{
    NpadHandheldActivationMode mode;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().GetNpadHandheldActivationMode(aruid, &mode));
    outValue.Set(static_cast<int64_t>(mode));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::IsUnintendedHomeButtonInputProtectionEnabled(nn::sf::Out<bool> outIsEnabled,
                                                                     nn::applet::AppletResourceUserId aruid,
                                                                     nn::hid::NpadIdType id) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyValidNpadId(id));

    bool isEnabled;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager()
                                     .IsUnintendedHomeButtonInputProtectionEnabled(&isEnabled, aruid, id));
    outIsEnabled.Set(static_cast<bool>(isEnabled));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::EnableUnintendedHomeButtonInputProtection(::nn::applet::AppletResourceUserId aruid,
                                                                  ::nn::hid::NpadIdType id,
                                                                  bool enabled) NN_NOEXCEPT
{
    NN_RESULT_DO(VerifyValidNpadId(id));
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager()
                                     .EnableUnintendedHomeButtonInputProtection(aruid, id, enabled));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateSixAxisSensor(BasicXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidBasicXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    auto& task = GetResourceManager().GetXpadTask();
    task.EnableXpadSampling();
#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    task.EnableBasicXpadSampling();
    NN_RESULT_DO(task.ActivateSixAxisSensor());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::DeactivateSixAxisSensor(BasicXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidBasicXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // 何もしない
#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    NN_RESULT_DO(task.DeactivateSixAxisSensor());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetSixAxisSensorLifoHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
    BasicXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidBasicXpadPlayerNumber(playerNumber));
    ::nn::os::NativeHandle handle;
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    auto& task = GetResourceManager().GetXpadTask();
    NN_RESULT_DO(task.GetSixAxisSensorLifoHandle(&handle, playerNumber));
#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    NN_RESULT_DO(task.GetSixAxisSensorLifoHandle(&handle));
#endif
    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ActivateJoySixAxisSensor(JoyXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidJoyXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    auto& task = GetResourceManager().GetXpadTask();
    task.EnableXpadSampling();
#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    task.EnableJoyXpadSampling();
    NN_RESULT_DO(task.ActivateJoySixAxisSensor());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::DeactivateJoySixAxisSensor(JoyXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidJoyXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // 何もしない
#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    NN_RESULT_DO(task.DeactivateJoySixAxisSensor());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetJoySixAxisSensorLifoHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
    JoyXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidJoyXpadPlayerNumber(playerNumber));
    ::nn::os::NativeHandle handle;
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    auto& task = GetResourceManager().GetXpadTask();
    NN_RESULT_DO(task.GetJoySixAxisSensorLifoHandle(&handle, playerNumber));

#else
    auto& task = GetResourceManager().GetXpadTask(playerNumber);
    NN_RESULT_DO(task.GetJoySixAxisSensorLifoHandle(&handle));
#endif
    outValue.Set(::nn::sf::NativeHandle(handle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::StartSixAxisSensor(nn::applet::AppletResourceUserId aruid,
                                           SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    pSetting->sixAxisSensorProcessorSetting.EnableSampling(true);

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::StopSixAxisSensor(nn::applet::AppletResourceUserId aruid,
                                          SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    pSetting->sixAxisSensorProcessorSetting.EnableSampling(false);

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::StartConsoleSixAxisSensor(nn::applet::AppletResourceUserId aruid,
                                                  nn::hid::ConsoleSixAxisSensorHandle handle) NN_NOEXCEPT
{
    ConsoleSixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetConsoleSixAxisSensorAppletSettingManager()
                            .GetConsoleSixAxisSensorSetting(&pSetting, aruid, handle));

    pSetting->EnableSampling(true);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::StopConsoleSixAxisSensor(nn::applet::AppletResourceUserId aruid,
                                                 nn::hid::ConsoleSixAxisSensorHandle handle) NN_NOEXCEPT
{
    ConsoleSixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetConsoleSixAxisSensorAppletSettingManager()
                            .GetConsoleSixAxisSensorSetting(&pSetting, aruid, handle));

    pSetting->EnableSampling(false);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::StartSevenSixAxisSensor(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(
        StartConsoleSixAxisSensor(aruid, ::nn::hid::ConsoleSixAxisSensorHandle()));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::StopSevenSixAxisSensor(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(
        StopConsoleSixAxisSensor(aruid, ::nn::hid::ConsoleSixAxisSensorHandle()));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::InitializeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid,
                                                     ::nn::sf::NativeHandle&& stateBufferHandle,
                                                     ::std::uint64_t stateBufferSize,
                                                     ::nn::sf::NativeHandle&& workBufferHandle,
                                                     ::std::uint64_t workBufferSize) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .InitializeSevenSixAxisSensor(aruid));
    // stateBuffer の割り当て
    {
        auto osHandle = stateBufferHandle.GetOsHandle();
        auto managed = stateBufferHandle.IsManaged();

        NN_SDK_REQUIRES_EQUAL(true, managed);

        stateBufferHandle.Detach();

        NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                         .SetStateBufferMemoryHandle(aruid,
                                                                     ::std::move(osHandle),
                                                                     static_cast<size_t>(stateBufferSize),
                                                                     managed));
    }

    // workBuffer の割り当て
    {
        auto osHandle = workBufferHandle.GetOsHandle();
        auto managed = workBufferHandle.IsManaged();

        NN_SDK_REQUIRES_EQUAL(true, managed);

        workBufferHandle.Detach();

        NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                         .SetWorkBufferMemoryHandle(aruid,
                                                                    ::std::move(osHandle),
                                                                    static_cast<size_t>(workBufferSize),
                                                                    managed));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::FinalizeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    StopSevenSixAxisSensor(aruid);

    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .FinalizeSevenSixAxisSensor(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetSevenSixAxisSensorFusionStrength(::nn::applet::AppletResourceUserId aruid, float strength) NN_NOEXCEPT
{
    ConsoleSixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetConsoleSixAxisSensorAppletSettingManager()
                            .GetConsoleSixAxisSensorSetting(&pSetting,
                                                            aruid,
                                                            ::nn::hid::ConsoleSixAxisSensorHandle()));
    pSetting->SetSevenSixAxisSensorFusionStrength(strength);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetSevenSixAxisSensorFusionStrength(::nn::sf::Out<float> strength, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ConsoleSixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetConsoleSixAxisSensorAppletSettingManager()
                            .GetConsoleSixAxisSensorSetting(&pSetting,
                                                            aruid,
                                                            ::nn::hid::ConsoleSixAxisSensorHandle()));
    float value = pSetting->GetSevenSixAxisSensorFusionStrength();
    strength.Set(value);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::IsSixAxisSensorFusionEnabled(nn::sf::Out<bool> outIsEnabled,
                                                     nn::applet::AppletResourceUserId aruid,
                                                     nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    bool isEnabled = pSetting->accelerometerSetting.IsSixAxisSensorFusionEnabled();
    outIsEnabled.Set(isEnabled);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::EnableSixAxisSensorFusion(nn::applet::AppletResourceUserId aruid,
                                                  nn::hid::SixAxisSensorHandle handle,
                                                  bool enable) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    pSetting->accelerometerSetting.EnableSixAxisSensorFusion(enable);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetSixAxisSensorFusionParameters(nn::applet::AppletResourceUserId aruid,
                                                         nn::hid::SixAxisSensorHandle handle,
                                                         float revisePower,
                                                         float reviseRange) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    pSetting->accelerometerSetting.SetSixAxisSensorFusionParameters(revisePower,
                                                                  reviseRange);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetSixAxisSensorFusionParameters(nn::sf::Out<float> outRevisePower,
                                                         nn::sf::Out<float> outReviseRange,
                                                         nn::applet::AppletResourceUserId aruid,
                                                         nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    float revisePower;
    float reviseRange;

    pSetting->accelerometerSetting.GetSixAxisSensorFusionParameters(&revisePower,
                                                                  &reviseRange);
    outRevisePower.Set(revisePower);
    outReviseRange.Set(reviseRange);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ResetSixAxisSensorFusionParameters(nn::applet::AppletResourceUserId aruid,
                                                           nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));
    pSetting->accelerometerSetting.ResetAccelerationReviseParameters();
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetAccelerometerParameters(nn::applet::AppletResourceUserId aruid,
                                                   nn::hid::SixAxisSensorHandle handle,
                                                   float playRadius,
                                                   float sensitivity) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    pSetting->accelerometerSetting.SetAccelerometerParameters(playRadius,
                                                            sensitivity);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetAccelerometerParameters(nn::sf::Out<float> outPlayRadius,
                                                   nn::sf::Out<float> outSensitivity,
                                                   nn::applet::AppletResourceUserId aruid,
                                                   nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    float playRadius;
    float sensitivity;

    pSetting->accelerometerSetting.GetAccelerometerParameters(&playRadius,
                                                            &sensitivity);
    outPlayRadius.Set(playRadius);
    outSensitivity.Set(sensitivity);

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ResetAccelerometerParameters(nn::applet::AppletResourceUserId aruid,
                                                     nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));
    pSetting->accelerometerSetting.ResetPlayModeParameters();
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetAccelerometerPlayMode(nn::applet::AppletResourceUserId aruid,
                                                 nn::hid::SixAxisSensorHandle handle, std::uint32_t mode) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));
    pSetting->accelerometerSetting.SetAccelerometerPlayMode(static_cast<nn::hid::AccelerometerPlayMode>(mode));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetAccelerometerPlayMode(nn::sf::Out<std::uint32_t> outMode,
                                                 nn::applet::AppletResourceUserId aruid,
                                                 nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    nn::hid::AccelerometerPlayMode mode;
    pSetting->accelerometerSetting.GetAccelerometerPlayMode(&mode);
    outMode.Set(mode);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ResetAccelerometerPlayMode(nn::applet::AppletResourceUserId aruid,
                                                   nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));
    pSetting->accelerometerSetting.ResetPlayMode();
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetGyroscopeZeroDriftMode(nn::applet::AppletResourceUserId aruid,
                                                  nn::hid::SixAxisSensorHandle handle,
                                                  std::uint32_t mode) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));
    pSetting->gyroscopeSetting.SetGyroscopeZeroDriftMode(static_cast<nn::hid::GyroscopeZeroDriftMode>(mode));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetGyroscopeZeroDriftMode(nn::sf::Out<std::uint32_t> outMode,
                                                  nn::applet::AppletResourceUserId aruid,
                                                  nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));

    nn::hid::GyroscopeZeroDriftMode mode;
    pSetting->gyroscopeSetting.GetGyroscopeZeroDriftMode(&mode);
    outMode.Set(mode);
    NN_RESULT_SUCCESS;
}
::nn::Result HidServer::ResetGyroscopeZeroDriftMode(nn::applet::AppletResourceUserId aruid,
                                                    nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    NN_RESULT_DO(
        GetResourceManager().GetSixAxisSensorAppletSettingManager()
                            .GetSixAxisSensorSetting(&pSetting, aruid, handle));
    pSetting->gyroscopeSetting.ResetZeroDriftMode();
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::IsSixAxisSensorAtRest(nn::sf::Out<bool> outIsAtRest,
                                              nn::applet::AppletResourceUserId aruid,
                                              nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    bool isAtRest = GetResourceManager().GetSixAxisSensorProcessor(handle)
                                        .IsSixAxisSensorAtRest();
    outIsAtRest.Set(isAtRest);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetVibrationDeviceInfo(
    ::nn::sf::Out<VibrationDeviceInfoForIpc> outValue,
    VibrationDeviceHandle handle) NN_NOEXCEPT
{
    VibrationDeviceInfo info;
    NN_RESULT_DO(GetResourceManager().GetVibrationDeviceInfo(&info, handle));
    outValue.Set(VibrationDeviceInfoForIpc::FromVibrationDeviceInfo(info));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SendVibrationValue(
    ::nn::applet::AppletResourceUserId aruid,
    VibrationDeviceHandle handle,
    const VibrationValue& value) NN_NOEXCEPT
{
    bool isEnabled = false;
    NN_RESULT_DO(GetResourceManager().IsAruidValidForVibration(aruid, &isEnabled));

    if (isEnabled == false)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(GetResourceManager().AssertValidVibrationDeviceHandle(handle));

    auto pVibrator = GetResourceManager().GetVibratorXcd(handle);

    if (pVibrator != nullptr)
    {
        NN_RESULT_TRY(pVibrator->SendVibrationValue(value))
        NN_RESULT_CATCH_ALL
        {
            // xcd で未接続系のエラーが発生しても上位には伝えない
        }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SendVibrationValues(::nn::applet::AppletResourceUserId aruid,
    const ::nn::sf::InArray<::nn::hid::VibrationDeviceHandle>& handles,
    const ::nn::sf::InArray<::nn::hid::VibrationValue>& values) NN_NOEXCEPT
{
    size_t handleCount = handles.GetLength();
    size_t valueCount = values.GetLength();
    NN_RESULT_THROW_UNLESS(handleCount == valueCount, nn::hid::ResultVibrationValueInvalidCount());

    auto handleArray = handles.GetData();
    auto valueArray = values.GetData();
    if (valueCount > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(handleArray);
        NN_SDK_REQUIRES_NOT_NULL(valueArray);
    }

    for (size_t i = 0; i < valueCount; i++)
    {
        NN_RESULT_DO(SendVibrationValue(aruid, handleArray[i], valueArray[i]));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetActualVibrationValue(
    ::nn::sf::Out<VibrationValue> outValue,
    ::nn::applet::AppletResourceUserId aruid,
    VibrationDeviceHandle handle) NN_NOEXCEPT
{
    bool isEnabled = false;
    NN_RESULT_DO(GetResourceManager().IsAruidValidForVibration(aruid, &isEnabled));

    if (isEnabled == false)
    {
        outValue.Set(VibrationValue::Make());
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(GetResourceManager().AssertValidVibrationDeviceHandle(handle));

    nn::hid::VibrationValue vibrationValue;
    auto pVibrator = GetResourceManager().GetVibratorXcd(handle);

    if (pVibrator != nullptr)
    {
        NN_RESULT_TRY(pVibrator->GetActualVibrationValue(&vibrationValue))
        NN_RESULT_CATCH_ALL
        {
            // xcd で未接続系のエラーが発生しても上位には伝えない
            vibrationValue = VibrationValue::Make();
        }
        NN_RESULT_END_TRY;
    }
    else
    {
        // 有効な Vibrator が取得できない場合も上位には伝えない
        vibrationValue = VibrationValue::Make();
    }
    outValue.Set(vibrationValue);

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::CreateActiveVibrationDeviceList(
    nn::sf::Out<nn::sf::SharedPointer<nn::hid::IActiveVibrationDeviceList>> outValue) NN_NOEXCEPT
{
    ActiveVibrationDeviceList list;

    outValue.Set(
        ::nn::sf::ObjectFactory<Allocator::Policy>::CreateSharedEmplaced<
        IActiveVibrationDeviceList, ActiveVibrationDeviceList>(::std::move(list)));

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SendVibrationGcErmCommand(
    ::nn::applet::AppletResourceUserId aruid,
    VibrationDeviceHandle handle,
    const VibrationGcErmCommand& command) NN_NOEXCEPT
{
    bool isEnabled = false;
    NN_RESULT_DO(GetResourceManager().IsAruidValidForVibration(aruid, &isEnabled));

    if (isEnabled == false)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(GetResourceManager().AssertValidVibrationDeviceHandle(handle));

    auto pVibrator = GetResourceManager().GetVibratorGc(handle);

    if (pVibrator != nullptr)
    {
        NN_RESULT_TRY(pVibrator->SendCommand(command))
        NN_RESULT_CATCH_ALL
        {
            // 未接続系のエラーが発生しても上位には伝えない
        }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetActualVibrationGcErmCommand(
    ::nn::sf::Out<VibrationGcErmCommand> outValue,
    ::nn::applet::AppletResourceUserId aruid,
    VibrationDeviceHandle handle) NN_NOEXCEPT
{
    bool isEnabled = false;
    NN_RESULT_DO(GetResourceManager().IsAruidValidForVibration(aruid, &isEnabled));

    if (isEnabled == false)
    {
        outValue.Set(VibrationGcErmCommand_Stop);
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(GetResourceManager().AssertValidVibrationDeviceHandle(handle));

    nn::hid::VibrationGcErmCommand vibrationGcErmCommand;
    auto pVibrator = GetResourceManager().GetVibratorGc(handle);

    if (pVibrator != nullptr)
    {
        NN_RESULT_TRY(pVibrator->GetActualCommand(&vibrationGcErmCommand))
        NN_RESULT_CATCH_ALL
        {
            // ハンドルが VibrationGcStyle 向けでない、未接続のような状態だとしても振動停止状態として上位には渡す
            vibrationGcErmCommand = VibrationGcErmCommand_Stop;
        }
        NN_RESULT_END_TRY;

    }
    else
    {
        // ハンドルが VibrationGcStyle 向けでない、未接続のような状態だとしても振動停止状態として上位には渡す
        vibrationGcErmCommand = VibrationGcErmCommand_Stop;
    }
    outValue.Set(vibrationGcErmCommand);

    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetNpadCommunicationMode(::nn::applet::AppletResourceUserId aruid,
                                                 ::std::int64_t mode) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(mode);
    // 廃止
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetNpadCommunicationMode(nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
{
    // 廃止
    ::nn::hid::NpadCommunicationMode mode = nn::hid::NpadCommunicationMode_Default;
    outValue.Set(mode);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::PermitVibration(bool isPermitted) NN_NOEXCEPT
{
    float masterVolume = isPermitted ? 1.0f : 0.0f;
    NN_RESULT_DO(::nn::xcd::SetAndSaveVibrationMasterVolume(masterVolume));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::IsVibrationPermitted(::nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    float masterVolume;
    NN_RESULT_DO(::nn::xcd::LoadAndGetVibrationMasterVolume(&masterVolume));
    outValue.Set(masterVolume > 0.0f);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::BeginPermitVibrationSession(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().BeginPermitVibrationSession(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::EndPermitVibrationSession() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().EndPermitVibrationSession());
    NN_RESULT_SUCCESS;
}


::nn::Result HidServer::IsUsbFullKeyControllerEnabled(::nn::sf::Out<bool> outIsEnabled) NN_NOEXCEPT
{
    bool isEnabled;
    NN_RESULT_DO(GetResourceManager().GetAbstractedPadXcdTask()
                                     .IsUsbFullKeyControllerEnabled(&isEnabled));
    outIsEnabled.Set(static_cast<bool>(isEnabled));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetAbstractedPadXcdTask()
                                     .EnableUsbFullKeyController(enabled));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::IsUsbFullKeyControllerConnected(::nn::sf::Out<bool> outIsConnected, ::nn::hid::NpadIdType id) NN_NOEXCEPT
{
    bool isConnected;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager()
                                     .IsUsbFullKeyControllerConnected(&isConnected, id));
    outIsConnected.Set(static_cast<bool>(isConnected));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::HasBattery(::nn::sf::Out<bool> outHasBattery, ::nn::hid::NpadIdType id) NN_NOEXCEPT
{
    bool hasBattery;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager()
                                     .HasBattery(&hasBattery, id));
    outHasBattery.Set(static_cast<bool>(hasBattery));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::HasLeftRightBattery(::nn::sf::Out<bool> outLeftJoyHasBattery, ::nn::sf::Out<bool> outRightJoyHasBattery, ::nn::hid::NpadIdType id) NN_NOEXCEPT
{
    bool hasLeftJoyBattery;
    bool hasRightJoyBattery;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager()
                                     .HasBattery(&hasLeftJoyBattery, &hasRightJoyBattery, id));
    outLeftJoyHasBattery.Set(static_cast<bool>(hasLeftJoyBattery));
    outRightJoyHasBattery.Set(static_cast<bool>(hasRightJoyBattery));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetNpadInterfaceType(::nn::sf::Out<nn::hid::system::InterfaceType> outInterfaceType, ::nn::hid::NpadIdType id) NN_NOEXCEPT
{
    nn::hid::system::InterfaceType interfaceType;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager()
                                     .GetInterfaceType(&interfaceType, id));
    outInterfaceType.Set(static_cast<nn::hid::system::InterfaceType>(interfaceType));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetNpadLeftRightInterfaceType(  ::nn::sf::Out<nn::hid::system::InterfaceType> outLeftJoyInterfaceType,
                                                        ::nn::sf::Out<nn::hid::system::InterfaceType> outRightJoyInterfaceType,
                                                        ::nn::hid::NpadIdType id) NN_NOEXCEPT
{
    nn::hid::system::InterfaceType leftJoyInterfaceType;
    nn::hid::system::InterfaceType rightJoyInterfaceType;
    NN_RESULT_DO(GetResourceManager().GetNpadResourceManager()
                                     .GetInterfaceType(&leftJoyInterfaceType, &rightJoyInterfaceType, id));
    outLeftJoyInterfaceType.Set(static_cast<nn::hid::system::InterfaceType>(leftJoyInterfaceType));
    outRightJoyInterfaceType.Set(static_cast<nn::hid::system::InterfaceType>(rightJoyInterfaceType));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetPalmaConnectionHandle(::nn::sf::Out<PalmaConnectionHandle> outHandle, NpadIdType id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    nn::hid::PalmaConnectionHandle handle;
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager()
                                     .GetPalmaConnectionHandle(&handle, id, aruid));
    outHandle.Set(handle);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::InitializePalma(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().Initialize(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::AcquirePalmaOperationCompleteEvent(::nn::sf::Out<::nn::sf::NativeHandle> outValue,
                                                           PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::nn::os::NativeHandle nativeHandle;
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager()
                                     .AcquireOperationCompleteEventHandle(&nativeHandle, handle));
    outValue.Set(::nn::sf::NativeHandle(nativeHandle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetPalmaOperationInfo(nn::sf::Out<uint64_t> outType,
                             nn::sf::OutBuffer outBuffer,
                             PalmaConnectionHandle handle) NN_NOEXCEPT
{
    nn::hid::PalmaOperationType operationType;
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().GetPalmaOperationInfo(&operationType, outBuffer, handle));
    outType.Set(operationType);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::PlayPalmaActivity(PalmaConnectionHandle handle, uint64_t index) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager()
                                     .PlayPalmaActivity(handle, static_cast<uint16_t>(index)));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetPalmaFrModeType(PalmaConnectionHandle handle, uint64_t type) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().SetFrModeType(handle, type));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ReadPalmaStep(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().ReadStep(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::EnablePalmaStep(PalmaConnectionHandle handle, bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().EnableStepCounter(handle, isEnabled));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ResetPalmaStep(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().ResetStep(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ReadPalmaApplicationSection(PalmaConnectionHandle handle, uint64_t address, uint64_t size) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().ReadApplicationSection(handle, address, size));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::WritePalmaApplicationSection(PalmaConnectionHandle handle, uint64_t address, uint64_t size, PalmaApplicationSectionAccessBuffer buffer) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().WriteApplicationSection(handle, address, size, buffer));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ReadPalmaUniqueCode(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().ReadPalmaUniqueCode(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetPalmaUniqueCodeInvalid(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().SetPalmaUniqueCodeInvalid(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::WritePalmaActivityEntry(PalmaConnectionHandle handle, uint64_t index, uint64_t ledIndex, uint64_t waveSet, uint64_t waveIndex) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().WritePalmaActivityEntry(handle, index, ledIndex, waveSet, waveIndex));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::WritePalmaRgbLedPatternEntry(PalmaConnectionHandle handle, uint64_t index, const ::nn::sf::InBuffer& inBuffer) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().WritePalmaRgbLedPatternEntry(handle, index, inBuffer.GetPointerUnsafe(), inBuffer.GetSize()));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::WritePalmaWaveEntry(PalmaConnectionHandle handle, uint64_t waveSet, uint64_t index, ::nn::sf::NativeHandle&& transferMemoryHandle, uint64_t transferMemorySize, uint64_t waveDataSize) NN_NOEXCEPT
{
    auto osHandle = transferMemoryHandle.GetOsHandle();
    auto managed = transferMemoryHandle.IsManaged();
    transferMemoryHandle.Detach();

    ::nn::os::TransferMemoryType* pTransferMemory = nullptr;
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().GetTransferMemoryType(&pTransferMemory, handle));
    ::nn::os::AttachTransferMemory(pTransferMemory, static_cast<size_t>(transferMemorySize), osHandle, managed);

    void* address = nullptr;
    NN_RESULT_DO(::nn::os::MapTransferMemory(&address, pTransferMemory, ::nn::os::MemoryPermission_ReadOnly));

    GetResourceManager().GetPalmaResourceManager().SetTransferMemory(handle, address, transferMemorySize);

    auto result = GetResourceManager().GetPalmaResourceManager().WritePalmaWaveEntry(handle, waveSet, index, static_cast<size_t>(waveDataSize));
    if (result.IsFailure())
    {
        nn::os::UnmapTransferMemory(pTransferMemory);
        nn::os::DestroyTransferMemory(pTransferMemory);
        return result;
    }
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetPalmaDataBaseIdentificationVersion(PalmaConnectionHandle handle, int32_t version) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().SetPalmaDataBaseIdentificationVersion(handle, version));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetPalmaDataBaseIdentificationVersion(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().GetPalmaDataBaseIdentificationVersion(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SuspendPalmaFeature(PalmaConnectionHandle handle, PalmaFeatureSet suspendFeatureSet) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().SuspendPalmaFeature(handle, suspendFeatureSet));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::GetPalmaOperationResult(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().GetPalmaOperationResult(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ReadPalmaPlayLog(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().ReadPalmaPlayLog(handle, index));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::ResetPalmaPlayLog(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().ResetPalmaPlayLog(handle, index));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetIsPalmaAllConnectable(::nn::applet::AppletResourceUserId aruid, bool connectable) NN_NOEXCEPT
{
    GetResourceManager().GetPalmaResourceManager().SetIsPalmaAllConnectable(aruid, connectable);
    NN_RESULT_SUCCESS;

}

::nn::Result HidServer::SetIsPalmaPairedConnectable(::nn::applet::AppletResourceUserId aruid, bool connectable) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(connectable);
//    GetResourceManager().GetPalmaResourceManager().SetIsPalmaPairedConnectable(connectable);
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::PairPalma(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetPalmaResourceManager().PairPalma(handle));
    NN_RESULT_SUCCESS;
}

::nn::Result HidServer::SetPalmaBoostMode(bool enabled) NN_NOEXCEPT
{
    NN_UNUSED(enabled);
    // 何もしない
    NN_RESULT_SUCCESS;
}

::nn::Result CreateHidServerProxy(::nn::sf::SharedPointer<IHidServer>* pOutValue
                                  ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    *pOutValue = StaticObject<
        ::nn::sf::UnmanagedServiceObject<IHidServer, HidServer>
        >::Get().GetShared();
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
