﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_SharedMemoryHolder.h"
#include "hid_VibratorXcdDriver.h"

namespace {

const int ActiveVibrationDeviceCountMax = 256;

}

namespace nn { namespace hid { namespace detail {

//!< アプレットリソースを扱うクラスです。
class AppletResource
{
    NN_DISALLOW_COPY(AppletResource);

private:
    AppletResource& operator=(AppletResource&&) NN_NOEXCEPT;

private:
    //!< ARUID
    ::nn::applet::AppletResourceUserId m_Aruid;

    //!< アプレットリソースの所有者か否かを
    bool m_IsOwner;

public:
    explicit AppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    AppletResource(AppletResource&& other) NN_NOEXCEPT;

    virtual ~AppletResource() NN_NOEXCEPT;

    //!< 初期化します。
    ::nn::Result Initialize() NN_NOEXCEPT;

    //!< 共有メモリのハンドルを取得します。
    ::nn::Result GetSharedMemoryHandle(
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT;
};

//!< アクティベートされた VibrationDevice のリストを扱うクラスです。
class ActiveVibrationDeviceList
{
    NN_DISALLOW_COPY(ActiveVibrationDeviceList);

private:
    ::nn::os::Mutex m_Mutex;
    int m_ActiveVibrationDeviceCount;
    ::nn::hid::VibrationDeviceHandle m_HandleList[ActiveVibrationDeviceCountMax];

public:
    ActiveVibrationDeviceList() NN_NOEXCEPT;

    //!< リストが空の状態でインスタンスを生成します。
    ActiveVibrationDeviceList(ActiveVibrationDeviceList&& other) NN_NOEXCEPT;

    //!< リストに含まれる VibrationDevice すべてに対してディアクティベート処理を行います。
    virtual ~ActiveVibrationDeviceList() NN_NOEXCEPT;

    //!< VibrationDevice をアクティベートしてリストに追加します。
    ::nn::Result ActivateVibrationDevice(
        ::nn::hid::VibrationDeviceHandle handle) NN_NOEXCEPT;
};

//!< hid サービスを扱うクラスです。
class HidServer final
{
    NN_DISALLOW_COPY(HidServer);
    NN_DISALLOW_MOVE(HidServer);

public:
    HidServer() NN_NOEXCEPT;

    //!< アプレットリソースを作成します。
    ::nn::Result CreateAppletResource(
        ::nn::sf::Out<::nn::sf::SharedPointer<IAppletResource>> outValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< DebugPad をアクティブ化します。
    ::nn::Result ActivateDebugPad(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Keyboard をアクティブ化します。
    ::nn::Result ActivateKeyboard(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Mouse をアクティブ化します。
    ::nn::Result ActivateMouse(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< TouchScreen をアクティブ化します。
    ::nn::Result ActivateTouchScreen(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor をアクティブ化します。
    ::nn::Result ActivateConsoleSixAxisSensor(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< SevenSixAxisSensor をアクティブ化します。
    ::nn::Result ActivateSevenSixAxisSensor(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Xpad ID の再解決イベントのハンドルを確保します。
    ::nn::Result AcquireXpadIdEventHandle(
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
        uint64_t clientId) NN_NOEXCEPT;

    //!< Xpad ID の再解決イベントのハンドルを開放します。
    ::nn::Result ReleaseXpadIdEventHandle(uint64_t clientId) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad をアクティブ化します。
    ::nn::Result ActivateXpad(::nn::applet::AppletResourceUserId aruid,
                              ::nn::hid::BasicXpadId xpadId) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の解決された ID を取得します。
    ::nn::Result GetXpadIds(::nn::sf::Out<int64_t> outCount,
                            const ::nn::sf::OutArray<BasicXpadId>& outXpadIds
                            ) NN_NOEXCEPT;

    //!< JoyXpad をアクティブ化します。
    nn::Result ActivateJoyXpad(nn::hid::JoyXpadId xpadId) NN_NOEXCEPT;

    //!< JoyXpad の LIFO のハンドルを取得します。
    nn::Result GetJoyXpadLifoHandle(
        nn::sf::Out<nn::sf::NativeHandle> outValue,
        nn::hid::JoyXpadId xpadId) NN_NOEXCEPT;

    //!< JoyXpad の解決された ID を取得します。
    nn::Result GetJoyXpadIds(nn::sf::Out<std::int64_t> outGotCount,
                             const nn::sf::OutArray<nn::hid::JoyXpadId>& outXpadIds
                             ) NN_NOEXCEPT;

    //!< Gesture をアクティブ化します。
    ::nn::Result ActivateGesture(
        ::nn::applet::AppletResourceUserId aruid, int32_t version) NN_NOEXCEPT;

    //!< アプリが許可する NpadStyle を設定します。
    nn::Result SetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, NpadStyleSet style) NN_NOEXCEPT;

    //!< 現在許可されている NpadStyle を設定します。
    nn::Result GetSupportedNpadStyleSet(::nn::applet::AppletResourceUserId aruid, ::nn::sf::Out<NpadStyleSet> outValue) NN_NOEXCEPT;

    //!< アプリが許可する NpadIdType を設定します。
    nn::Result SetSupportedNpadIdType(::nn::applet::AppletResourceUserId aruid, const ::nn::sf::InArray<::nn::hid::NpadIdType>& ids) NN_NOEXCEPT;

    //!< Npad をアクティベートします。
    nn::Result ActivateNpad(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Npad をアクティベートします。(リビジョンつき)
    nn::Result ActivateNpadWithRevision(::nn::applet::AppletResourceUserId aruid, int32_t revision) NN_NOEXCEPT;

    //!< Npad をディアクティベートします。
    nn::Result DeactivateNpad(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadStyleSet が更新された際に通知を受けるイベントをバインドします。
    nn::Result AcquireNpadStyleSetUpdateEventHandle(::nn::applet::AppletResourceUserId aruid,
                                                    ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
                                                    NpadIdType id,
                                                    uint64_t clientId) NN_NOEXCEPT;

    //!< Npad を切断します。
    nn::Result DisconnectNpad(::nn::applet::AppletResourceUserId aruid,
                              NpadIdType id) NN_NOEXCEPT;

    //!< Npad のプレイヤーランプの点灯パターンを取得します。
    nn::Result GetPlayerLedPattern(::nn::sf::Out<uint64_t> outValue,
                                   NpadIdType id) NN_NOEXCEPT;

    //!< Npad に対するジョイコンの持ち方を設定します
    nn::Result SetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid,
                                  std::int64_t holdType) NN_NOEXCEPT;

    //!< Npad に対するジョイコンの持ち方を取得します
    nn::Result GetNpadJoyHoldType(::nn::applet::AppletResourceUserId aruid,
                                  ::nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT;

    //!< Npad に対するジョイコンの割り当てモードを取得します
    nn::Result GetNpadJoyAssignment(::nn::applet::AppletResourceUserId aruid,
                                    ::nn::sf::Out<std::int64_t> outValue, NpadIdType id) NN_NOEXCEPT;

    //!< Npad に対するジョイコンの割り当てモードを1本ずつ割り当てにデフォルト設定で設定します。
    nn::Result SetNpadJoyAssignmentModeSingleByDefault(::nn::applet::AppletResourceUserId aruid,
                                                       NpadIdType id) NN_NOEXCEPT;

    //!< Npad に対するジョイコンの割り当てモードを1本ずつ割り当てに設定します。
    nn::Result SetNpadJoyAssignmentModeSingle(::nn::applet::AppletResourceUserId aruid,
                                              NpadIdType id,
                                              int64_t type) NN_NOEXCEPT;

    //!< Npad に対するジョイコンの割り当てモードを1本ずつ割り当てに設定し、再割り当ての状態を取得します。
    nn::Result SetNpadJoyAssignmentModeSingleWithDestination(::nn::sf::Out<bool> outIsAssigned,
                                                                 ::nn::sf::Out<NpadIdType> outId,
                                                                 ::nn::applet::AppletResourceUserId aruid,
                                                                 NpadIdType id,
                                                                 int64_t type) NN_NOEXCEPT;

    //!< Npad に対するジョイコンの割り当てモードを2本セット割り当てに設定します。
    nn::Result SetNpadJoyAssignmentModeDual(::nn::applet::AppletResourceUserId aruid,
                                            NpadIdType id) NN_NOEXCEPT;

    //!< 1本ずつ割り当てになっている2つの Npad をマージして 2本セット割り当てに設定します。
    nn::Result MergeSingleJoyAsDualJoy(::nn::applet::AppletResourceUserId aruid,
                                       NpadIdType id1,
                                       NpadIdType id2) NN_NOEXCEPT;

    //!< Npad の割り当て状態をいれかえます。
    nn::Result SwapNpadAssignment(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラーの割り当てモードを開始します。
    ::nn::Result StartLrAssignmentMode(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< LR ボタン押しによるコントローラーの割り当てモードを終了します。
    ::nn::Result StopLrAssignmentMode(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Handheld 操作形態を有効化するモードを設定します。
    ::nn::Result SetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid,
                                               ::std::int64_t mode) NN_NOEXCEPT;

    //!< Handheld 操作形態を有効化するモードを取得します。
    ::nn::Result GetNpadHandheldActivationMode(::nn::applet::AppletResourceUserId aruid,
                                               ::nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT;

    //!< Npad のホームボタンが有効かどうかを取得します。
    ::nn::Result IsUnintendedHomeButtonInputProtectionEnabled(::nn::sf::Out<bool> outIsEnabled,
                                                              ::nn::applet::AppletResourceUserId aruid,
                                                              ::nn::hid::NpadIdType id) NN_NOEXCEPT;

    //!< Npad のホームボタンの有効/無効を設定します。
    ::nn::Result EnableUnintendedHomeButtonInputProtection(::nn::applet::AppletResourceUserId aruid,
                                                           ::nn::hid::NpadIdType id,
                                                           bool enabled) NN_NOEXCEPT;

    //!< 6軸センサーをアクティブ化します。
    ::nn::Result ActivateSixAxisSensor(::nn::hid::BasicXpadId xpadId) NN_NOEXCEPT;

    //!< 6軸センサーを非アクティブ化します。
    ::nn::Result DeactivateSixAxisSensor(::nn::hid::BasicXpadId xpadId) NN_NOEXCEPT;

    //!< 6軸センサーの LIFO のハンドルを取得します。
    ::nn::Result GetSixAxisSensorLifoHandle(
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
        ::nn::hid::BasicXpadId xpadId) NN_NOEXCEPT;

    ::nn::Result ActivateJoySixAxisSensor(::nn::hid::JoyXpadId xpadId) NN_NOEXCEPT;

    ::nn::Result DeactivateJoySixAxisSensor(::nn::hid::JoyXpadId xpadId) NN_NOEXCEPT;

    ::nn::Result GetJoySixAxisSensorLifoHandle(::nn::sf::Out<::nn::sf::NativeHandle> outValue, ::nn::hid::JoyXpadId xpadId) NN_NOEXCEPT;

    //!< 振動子の情報を取得します。
    ::nn::Result GetVibrationDeviceInfo(nn::sf::Out<nn::hid::VibrationDeviceInfoForIpc> outValue, nn::hid::VibrationDeviceHandle handle) NN_NOEXCEPT;

    //!< 振動子に振動値を送信します。
    ::nn::Result SendVibrationValue(::nn::applet::AppletResourceUserId aruid, nn::hid::VibrationDeviceHandle handle, const nn::hid::VibrationValue& value) NN_NOEXCEPT;

    //!< 複数の振動値をまとめて送信します。
    ::nn::Result SendVibrationValues(::nn::applet::AppletResourceUserId aruid,
        const ::nn::sf::InArray<::nn::hid::VibrationDeviceHandle>& handles,
        const ::nn::sf::InArray<::nn::hid::VibrationValue>& values) NN_NOEXCEPT;

    //!< 振動子で実際に発生している振動を取得します。
    ::nn::Result GetActualVibrationValue(nn::sf::Out<nn::hid::VibrationValue> outValue, ::nn::applet::AppletResourceUserId aruid, nn::hid::VibrationDeviceHandle handle) NN_NOEXCEPT;

    ::nn::Result CreateActiveVibrationDeviceList(nn::sf::Out<nn::sf::SharedPointer<nn::hid::IActiveVibrationDeviceList>> outValue) NN_NOEXCEPT;

    //!< Gc コントローラーの振動子に振動コマンドを送信します。
    ::nn::Result SendVibrationGcErmCommand(::nn::applet::AppletResourceUserId aruid, nn::hid::VibrationDeviceHandle handle, const nn::hid::VibrationGcErmCommand& command) NN_NOEXCEPT;

    //!< Gc コントローラーの振動子で実際に発生している振動コマンドを取得します。
    ::nn::Result GetActualVibrationGcErmCommand(nn::sf::Out<nn::hid::VibrationGcErmCommand> outValue, ::nn::applet::AppletResourceUserId aruid, nn::hid::VibrationDeviceHandle handle) NN_NOEXCEPT;

    //!< 振動機能を許可または禁止して、設定値を本体 NAND に保存します。
    ::nn::Result PermitVibration(bool isPermitted) NN_NOEXCEPT;

    //!< 振動機能が許可されているか取得します。
    ::nn::Result IsVibrationPermitted(nn::sf::Out<bool> outValue) NN_NOEXCEPT;

    //!< 一時的な振動強制ON期間を開始します。
    ::nn::Result BeginPermitVibrationSession(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 一時的な振動強制ON期間を停止します。
    ::nn::Result EndPermitVibrationSession() NN_NOEXCEPT;


    ::nn::Result StartSixAxisSensor(nn::applet::AppletResourceUserId aruid,
                                    nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result StopSixAxisSensor(nn::applet::AppletResourceUserId aruid,
                                   nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result StartConsoleSixAxisSensor(nn::applet::AppletResourceUserId aruid,
                                           nn::hid::ConsoleSixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result StopConsoleSixAxisSensor(nn::applet::AppletResourceUserId aruid,
                                          nn::hid::ConsoleSixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result StartSevenSixAxisSensor(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    ::nn::Result StopSevenSixAxisSensor(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    ::nn::Result InitializeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid,
                                              ::nn::sf::NativeHandle&& stateBufferHandle,
                                              ::std::uint64_t stateBufferSize,
                                              ::nn::sf::NativeHandle&& workBufferHandle,
                                              ::std::uint64_t workBufferSize) NN_NOEXCEPT;

    ::nn::Result FinalizeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    ::nn::Result SetSevenSixAxisSensorFusionStrength(::nn::applet::AppletResourceUserId aruid, float strength) NN_NOEXCEPT;

    ::nn::Result GetSevenSixAxisSensorFusionStrength(::nn::sf::Out<float> strength, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    ::nn::Result IsSixAxisSensorFusionEnabled(nn::sf::Out<bool> outIsEnabled,
                                              nn::applet::AppletResourceUserId aruid,
                                              nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result EnableSixAxisSensorFusion(nn::applet::AppletResourceUserId aruid,
                                           nn::hid::SixAxisSensorHandle handle, bool enable) NN_NOEXCEPT;

    ::nn::Result SetSixAxisSensorFusionParameters(nn::applet::AppletResourceUserId aruid,
                                                  nn::hid::SixAxisSensorHandle handle,
                                                  float revisePower,
                                                  float reviseRange) NN_NOEXCEPT;

    ::nn::Result GetSixAxisSensorFusionParameters(nn::sf::Out<float> outRevisePower,
                                                  nn::sf::Out<float> outReviseRange,
                                                  nn::applet::AppletResourceUserId aruid,
                                                  nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result ResetSixAxisSensorFusionParameters(nn::applet::AppletResourceUserId aruid,
                                                    nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result SetAccelerometerParameters(nn::applet::AppletResourceUserId aruid,
                                            nn::hid::SixAxisSensorHandle handle,
                                            float playRadius,
                                            float sensitivity) NN_NOEXCEPT;

    ::nn::Result GetAccelerometerParameters(nn::sf::Out<float> outPlayRadius,
                                            nn::sf::Out<float> outSensitivity,
                                            nn::applet::AppletResourceUserId aruid,
                                            nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result ResetAccelerometerParameters(nn::applet::AppletResourceUserId aruid,
                                              nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result SetAccelerometerPlayMode(nn::applet::AppletResourceUserId aruid,
                                          nn::hid::SixAxisSensorHandle handle,
                                          std::uint32_t mode) NN_NOEXCEPT;

    ::nn::Result GetAccelerometerPlayMode(nn::sf::Out<std::uint32_t> outMode,
                                          nn::applet::AppletResourceUserId aruid,
                                          nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result ResetAccelerometerPlayMode(nn::applet::AppletResourceUserId aruid,
                                            nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result SetGyroscopeZeroDriftMode(nn::applet::AppletResourceUserId aruid,
                                           nn::hid::SixAxisSensorHandle handle, std::uint32_t mode) NN_NOEXCEPT;

    ::nn::Result GetGyroscopeZeroDriftMode(nn::sf::Out<std::uint32_t> outMode,
                                           nn::applet::AppletResourceUserId aruid,
                                           nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result ResetGyroscopeZeroDriftMode(nn::applet::AppletResourceUserId aruid,
                                             nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    ::nn::Result IsSixAxisSensorAtRest(nn::sf::Out<bool> outIsAtRest,
                                       nn::applet::AppletResourceUserId aruid,
                                       nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信を有効または無効にして、設定値を本体 NAND に保存します。
    ::nn::Result EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信が有効になっているか取得します。
    ::nn::Result IsUsbFullKeyControllerEnabled(::nn::sf::Out<bool> outIsEnabled) NN_NOEXCEPT;

    //!< 有線 USB 通信中の Pro Controller かどうかを取得します。
    ::nn::Result IsUsbFullKeyControllerConnected(::nn::sf::Out<bool> outIsConnected, ::nn::hid::NpadIdType id) NN_NOEXCEPT;

    //!< バッテリーを搭載しているかどうかを取得します。
    ::nn::Result HasBattery(::nn::sf::Out<bool> outHasBattery, ::nn::hid::NpadIdType id) NN_NOEXCEPT;

    //!< バッテリーを搭載しているかどうかを取得します。
    ::nn::Result HasLeftRightBattery(::nn::sf::Out<bool> outLeftJoyHasBattery, ::nn::sf::Out<bool> outRightJoyHasBattery, ::nn::hid::NpadIdType id) NN_NOEXCEPT;

    //!< 接続形式を取得します。
    ::nn::Result GetNpadInterfaceType(::nn::sf::Out<nn::hid::system::InterfaceType> outInterfaceType, ::nn::hid::NpadIdType id) NN_NOEXCEPT;

    //!< 接続形式を取得します。
    ::nn::Result GetNpadLeftRightInterfaceType( ::nn::sf::Out<nn::hid::system::InterfaceType> outLeftJoyInterfaceType,
                                                ::nn::sf::Out<nn::hid::system::InterfaceType> outRightJoyInterfaceType,
                                                ::nn::hid::NpadIdType id) NN_NOEXCEPT;

    //!< Palma のコネクションハンドルを取得します。
    ::nn::Result GetPalmaConnectionHandle(::nn::sf::Out<PalmaConnectionHandle> outHandle, NpadIdType id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Palma を初期化します。
    ::nn::Result InitializePalma(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma に対する操作が完了したときに受け取るイベントオブジェクトのハンドルを取得します。
    ::nn::Result AcquirePalmaOperationCompleteEvent(::nn::sf::Out<::nn::sf::NativeHandle> nativeHandle,
                                                    PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma に対する操作の処理結果を取得します。
    ::nn::Result GetPalmaOperationInfo(nn::sf::Out<uint64_t> outType,
                                 nn::sf::OutBuffer outBuffer,
                                 PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma の Activity を再生します。
    ::nn::Result PlayPalmaActivity(PalmaConnectionHandle handle, uint64_t index) NN_NOEXCEPT;

    //!< Palma の FrMode の種別を設定します。
    ::nn::Result SetPalmaFrModeType(PalmaConnectionHandle handle, uint64_t type) NN_NOEXCEPT;

    //!< Palma の Step の値を読み出します。
    ::nn::Result ReadPalmaStep(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma の Step 機能を有効/無効を制御します。
    ::nn::Result EnablePalmaStep(PalmaConnectionHandle handle, bool isEnabled) NN_NOEXCEPT;

    //!< Palma の Step の値をリセットします。
    ::nn::Result ResetPalmaStep(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma のアプリケーションセクションを読み出します。
    ::nn::Result ReadPalmaApplicationSection(PalmaConnectionHandle handle, uint64_t address, uint64_t size) NN_NOEXCEPT;

    //!< Palma のアプリケーションセクションに書き込みます。
    ::nn::Result WritePalmaApplicationSection(PalmaConnectionHandle handle, uint64_t address, uint64_t size, PalmaApplicationSectionAccessBuffer buffer) NN_NOEXCEPT;

    //!< Palma の UniqueCode を読み出します。
    ::nn::Result ReadPalmaUniqueCode(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma の UniqueCode を無効化します。
    ::nn::Result SetPalmaUniqueCodeInvalid(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Activity Entry を Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaActivityEntry(PalmaConnectionHandle handle, uint64_t index, uint64_t ledIndex, uint64_t waveSet, uint64_t waveIndex) NN_NOEXCEPT;

    //!< RGB LED パターンを Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaRgbLedPatternEntry(PalmaConnectionHandle handle, uint64_t index, const ::nn::sf::InBuffer& inBuffer) NN_NOEXCEPT;

    //!< 音源データを Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaWaveEntry(PalmaConnectionHandle handle, uint64_t waveSet, uint64_t index, ::nn::sf::NativeHandle&& transferMemoryHandle, uint64_t transferMemorySize, uint64_t waveDataSize) NN_NOEXCEPT;

    //!< Palma の ActivityDataBase のデータベース識別バージョンを設定します。
    ::nn::Result SetPalmaDataBaseIdentificationVersion(PalmaConnectionHandle handle, int32_t version) NN_NOEXCEPT;

    //!< Palma の ActivityDataBase のデータベース識別バージョンを取得します。
    ::nn::Result GetPalmaDataBaseIdentificationVersion(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma の Feature Set の機能のサスペンド状態を制御します。
    ::nn::Result SuspendPalmaFeature(PalmaConnectionHandle handle, PalmaFeatureSet suspendFeatureSet) NN_NOEXCEPT;

    //!< Palma に対する操作の処理結果の Result を取得します。
    ::nn::Result GetPalmaOperationResult(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma の Play Log を読み出します。
    ::nn::Result ReadPalmaPlayLog(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT;

    //!< Palma の Play Log をゼロクリアします。
    ::nn::Result ResetPalmaPlayLog(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT;

    //!< 全ての Palma のスキャンの有効無効を切り替える
    ::nn::Result SetIsPalmaAllConnectable(::nn::applet::AppletResourceUserId aruid, bool connectable) NN_NOEXCEPT;

    //!< 登録済みの Palma のスキャンの有効無効を切り替える
    ::nn::Result SetIsPalmaPairedConnectable(::nn::applet::AppletResourceUserId aruid, bool connectable) NN_NOEXCEPT;

    //!< 指定されたデバイスとペアリングする
    ::nn::Result PairPalma(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma の高速通信モードを設定します
    ::nn::Result SetPalmaBoostMode(bool enabled) NN_NOEXCEPT;

    //!< 廃止予定
    ::nn::Result SetNpadCommunicationMode(::nn::applet::AppletResourceUserId aruid,
                                          ::std::int64_t mode) NN_NOEXCEPT;

    //!< 廃止予定
    ::nn::Result GetNpadCommunicationMode(::nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT;
};

//!< hid サービスのプロクシを作成します。
::nn::Result CreateHidServerProxy(::nn::sf::SharedPointer<IHidServer>* pOutValue
                                  ) NN_NOEXCEPT;

}}} // namespace nn::hid::detail
