﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_JoyXpad.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/TargetConfigs/build_Base.h>

#include "hid_HidDebugServer.h"
#include "hid_Settings.h"
#include "hid_StaticObject.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "hid_ResourceManager-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "hid_ResourceManager-os.win.h"
#endif

namespace nn { namespace hid { namespace detail {

namespace
{

// NpadId 指定タイプの API 向けに、UniquePad 用の Result を対応する Npad 用 Result に置換
Result ReplaceUniquePadResultToNpadResult(Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(system::ResultUniquePadDisconnected)
        {
            NN_RESULT_THROW(ResultNpadControllerNotConnected());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

//!< os ハンドルを管理するか否かを表す値
const bool NeedsToBeManaged =
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    false;
#else
    false;
#endif

}  // anonymous

HidDebugServer::HidDebugServer() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result HidDebugServer::DeactivateDebugPad() NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            GetResourceManager().GetDebugPadTask().DeactivateDebugPad());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetDebugPadAutoPilotState(
    const ::nn::hid::debug::DebugPadAutoPilotState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().GetDebugPadTask()
                                        .SetDebugPadAutoPilotState(value));
}

::nn::Result HidDebugServer::UnsetDebugPadAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().GetDebugPadTask()
                                        .UnsetDebugPadAutoPilotState());
}

::nn::Result HidDebugServer::DeactivateKeyboard() NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            GetResourceManager().GetKeyboardTask().DeactivateKeyboard());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetKeyboardAutoPilotState(
    const ::nn::hid::debug::KeyboardAutoPilotState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().GetKeyboardTask()
                                        .SetKeyboardAutoPilotState(value));
}

::nn::Result HidDebugServer::UnsetKeyboardAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().GetKeyboardTask()
                                        .UnsetKeyboardAutoPilotState());
}

::nn::Result HidDebugServer::DeactivateMouse() NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(GetResourceManager().GetMouseTask().DeactivateMouse());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetMouseAutoPilotState(
    const ::nn::hid::debug::MouseAutoPilotState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().GetMouseTask()
                                        .SetMouseAutoPilotState(value));
}

::nn::Result HidDebugServer::UnsetMouseAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().GetMouseTask()
                                        .UnsetMouseAutoPilotState());
}

::nn::Result HidDebugServer::DeactivateTouchScreen() NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            GetResourceManager().GetTouchScreenTask().DeactivateTouchScreen());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetTouchScreenAutoPilotState(
    const ::nn::sf::InArray<TouchState>& touches) NN_NOEXCEPT
{
    ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
        > autoPilotState = {};
    autoPilotState.count = static_cast<int32_t>(
        ::std::max(0, ::std::min(static_cast<int>(touches.GetLength()),
                                 TouchStateCountMax)));
    for (int i = 0; i < autoPilotState.count; ++i)
    {
        autoPilotState.touches[i] = touches[i];
    }
    NN_RESULT_DO(
        GetResourceManager().GetTouchScreenTask()
                            .SetTouchScreenAutoPilotState(autoPilotState));
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::UnsetTouchScreenAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW(GetResourceManager().GetTouchScreenTask()
                                        .UnsetTouchScreenAutoPilotState());
}

::nn::Result HidDebugServer::DeactivateXpad(BasicXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidBasicXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // 何もしない
#else
    NN_RESULT_DO(
        GetResourceManager().GetXpadTask(playerNumber).DeactivateBasicXpad());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetXpadAutoPilotState(
    BasicXpadId xpadId,
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidBasicXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    NN_RESULT_DO(
        GetResourceManager().GetXpadTask()
                            .SetBasicXpadAutoPilotState(playerNumber, value));
    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                            .SetAbstractedPadAutoPilotStateWithBasicXpad(playerNumber, value));
#else
    NN_RESULT_DO(
        GetResourceManager().GetXpadTask(playerNumber)
                            .SetBasicXpadAutoPilotState(value));
    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                            .SetAbstractedPadAutoPilotStateWithBasicXpad(playerNumber, value));
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::UnsetXpadAutoPilotState(BasicXpadId xpadId
                                                     ) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidBasicXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    NN_RESULT_DO(
        GetResourceManager().GetXpadTask()
                            .UnsetBasicXpadAutoPilotState(playerNumber));

    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                            .UnsetAbstractedPadAutoPilotState(playerNumber));
#else
    NN_RESULT_DO(
        GetResourceManager().GetXpadTask(playerNumber)
                            .UnsetBasicXpadAutoPilotState());


    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                            .UnsetAbstractedPadAutoPilotState(playerNumber));
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetAutoPilotVirtualPadState(const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId, const ::nn::hid::debug::AbstractedPadState& value) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                .SetAbstractedPadAutoPilotVirtualPadState(virtualPadId, value));
#else
    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                .SetAbstractedPadAutoPilotVirtualPadState(virtualPadId, value));
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::UnsetAutoPilotVirtualPadState(const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                .UnsetAbstractedPadAutoPilotVirtualPadState(virtualPadId));
#else
    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                .UnsetAbstractedPadAutoPilotVirtualPadState(virtualPadId));
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::UnsetAllAutoPilotVirtualPadState() NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                .UnsetAllAbstractedPadAutoPilotVirtualPadState());
#else
    NN_RESULT_DO(
        GetResourceManager().GetAbstractedAutoPilotPadTask()
                .UnsetAllAbstractedPadAutoPilotVirtualPadState());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateJoyXpad(JoyXpadId xpadId) NN_NOEXCEPT
{
    const int playerNumber = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_DO(
        GetResourceManager().AssertValidJoyXpadPlayerNumber(playerNumber));
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // 何もしない
#else
    NN_RESULT_DO(
        GetResourceManager().GetXpadTask(playerNumber).DeactivateJoyXpad());
#endif
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateGesture() NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            GetResourceManager().GetGestureTask().DeactivateGesture());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateHomeButton() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetHomeButtonTask()
                                     .DeactivateHomeButton());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetHomeButtonAutoPilotState(
    ::nn::hid::debug::HomeButtonAutoPilotState value) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetHomeButtonTask()
                                     .SetHomeButtonAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::UnsetHomeButtonAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetHomeButtonTask()
                                     .UnsetHomeButtonAutoPilotState());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateSleepButton() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetSleepButtonTask()
                                     .DeactivateSleepButton());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetSleepButtonAutoPilotState(
    ::nn::hid::debug::SleepButtonAutoPilotState value) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetSleepButtonTask()
                                     .SetSleepButtonAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::UnsetSleepButtonAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetSleepButtonTask()
                                     .UnsetSleepButtonAutoPilotState());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateCaptureButton() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetCaptureButtonTask()
                                     .DeactivateCaptureButton());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetCaptureButtonAutoPilotState(
    ::nn::hid::debug::CaptureButtonAutoPilotState value) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetCaptureButtonTask()
                                     .SetCaptureButtonAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::UnsetCaptureButtonAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetCaptureButtonTask()
                                     .UnsetCaptureButtonAutoPilotState());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateConsoleSixAxisSensor() NN_NOEXCEPT
{
    if (!::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            GetResourceManager().GetConsoleSixAxisSensorTask()
                                .DeactivateConsoleSixAxisSensor());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetConsoleSixAxisSensorSamplingFrequency(::nn::sf::Out<::std::int64_t> outValue,
                                                                      ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT

{
    int64_t samplingFrequency;

    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .GetConsoleSixAxisSensorSamplingFrequency(&samplingFrequency, aruid));

    outValue.Set(samplingFrequency);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateSevenSixAxisSensor() NN_NOEXCEPT
{
    DeactivateConsoleSixAxisSensor();
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetConsoleSixAxisSensorCountStates(
    nn::sf::Out<std::int32_t> outCount,
    const nn::sf::OutArray<nn::hid::debug::ConsoleSixAxisSensorCountState>& outStates,
    nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outCount.GetPointer()
        && outStates.GetData()
        && outStates.GetLength() <= ::nn::hid::debug::ConsoleSixAxisSensorCountStateCountMax,
        ResultConsoleSixAxisSensorInvalidStateCount());

    NN_UNUSED(aruid);

    int output = 0;
    NN_RESULT_DO(GetResourceManager().GetConsoleSixAxisSensorTask()
                                     .GetConsoleSixAxisSensorCountStates(&output,
                                                                         outStates.GetData(),
                                                                         static_cast<int>(outStates.GetLength())));
    outCount.Set(output);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateInputDetector() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetInputDetectorTask()
        .DeactivateInputDetector());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetShiftAccelerometerCalibrationValue(const SixAxisSensorHandle& handle,
                                                                   const ::nn::applet::AppletResourceUserId& aruid,
                                                                   float shiftOrigin,
                                                                   float shiftSensitivity) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    // aruid を取得
    ::nn::applet::AppletResourceUserId aruids[AppletResourceEntryCountMax] = {};
    GetResourceManager().GetSixAxisSensorAppletSettingManager()
                        .GetAppletResourceUserId(aruids, AppletResourceEntryCountMax);

    SixAxisSensorSetting* pSetting = nullptr;
    auto& mgr = GetResourceManager().GetSixAxisSensorAppletSettingManager();
    for (const auto& id : aruids)
    {
        NN_RESULT_DO(mgr.GetSixAxisSensorSetting(&pSetting,
                                                 id,
                                                 handle));
        pSetting->accelerometerSetting.SetShiftCalibrationValue(shiftOrigin, shiftSensitivity);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetShiftAccelerometerCalibrationValue(nn::sf::Out<float> outShiftOrigin,
                                                                   nn::sf::Out<float> outShiftSensitivity,
                                                                   const SixAxisSensorHandle& handle,
                                                                   const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    float origin;
    float sensitivity;

    SixAxisSensorSetting* pSetting = nullptr;

    NN_RESULT_DO(GetResourceManager().GetSixAxisSensorAppletSettingManager()
                                     .GetSixAxisSensorSetting(&pSetting,
                                                              aruid,
                                                              handle));

    pSetting->accelerometerSetting.GetShiftCalibrationValue(&origin,
                                                            &sensitivity);

    outShiftOrigin.Set(origin);
    outShiftSensitivity.Set(sensitivity);

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::SetShiftGyroscopeCalibrationValue(const SixAxisSensorHandle& handle,
                                                               const ::nn::applet::AppletResourceUserId& aruid,
                                                               float shiftOrigin,
                                                               float shiftSensitivity) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    // aruid を取得
    ::nn::applet::AppletResourceUserId aruids[AppletResourceEntryCountMax] = {};
    GetResourceManager().GetSixAxisSensorAppletSettingManager()
                        .GetAppletResourceUserId(aruids, AppletResourceEntryCountMax);

    SixAxisSensorSetting* pSetting = nullptr;
    auto& mgr = GetResourceManager().GetSixAxisSensorAppletSettingManager();
    for (const auto& id : aruids)
    {
        NN_RESULT_DO(mgr.GetSixAxisSensorSetting(&pSetting,
                                                 id,
                                                 handle));
        pSetting->gyroscopeSetting.SetShiftCalibrationValue(shiftOrigin, shiftSensitivity);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetShiftGyroscopeCalibrationValue(nn::sf::Out<float> outShiftOrigin,
                                                               nn::sf::Out<float> outShiftSensitivity,
                                                               const SixAxisSensorHandle& handle,
                                                               const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    float origin;
    float sensitivity;

    SixAxisSensorSetting* pSetting = nullptr;

    NN_RESULT_DO(GetResourceManager().GetSixAxisSensorAppletSettingManager()
                                     .GetSixAxisSensorSetting(&pSetting,
                                                              aruid,
                                                              handle));

    pSetting->gyroscopeSetting.GetShiftCalibrationValue(&origin,
                                                        &sensitivity);

    outShiftOrigin.Set(origin);
    outShiftSensitivity.Set(sensitivity);

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::ActivateFirmwareUpdate() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .ActivateFirmwareUpdate());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DeactivateFirmwareUpdate() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .DeactivateFirmwareUpdate());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::StartFirmwareUpdate(::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager().StartFirmwareUpdateForDebug(id));

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::StartFirmwareUpdateForRevert(::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager().StartFirmwareUpdateForRevert(id));

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetFirmwareVersion(::nn::sf::Out<debug::FirmwareVersion> outValue,
                                                ::nn::hid::NpadIdType id,
                                                ::nn::hid::system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    debug::FirmwareVersion version;
    bool isIapCorrupted;
    auto result =
        GetResourceManager().GetNpadResourceManager().GetFirmwareVersion(&version, &isIapCorrupted, id, deviceType);
    NN_RESULT_DO(ReplaceUniquePadResultToNpadResult(result));
    outValue.Set(version);
    NN_UNUSED(isIapCorrupted);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetAvailableFirmwareVersionForRevert(::nn::sf::Out<debug::FirmwareVersion> outValue,
                                                                  ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    ::nn::hid::system::FirmwareVersion version;
    NN_RESULT_DO(
        GetResourceManager().GetUniquePadResourceManager()
                            .GetDestinationFirmwareVersionForRevert(&version, id));
    outValue.Set(version);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::UpdateControllerColor(::nn::util::Color4u8Type mainColor, ::nn::util::Color4u8Type subColor, ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager().UpdateControllerColor(mainColor, subColor, id));
    NN_RESULT_SUCCESS;

}

::nn::Result HidDebugServer::UpdateDesignInfo(::nn::util::Color4u8Type mainColor,
                                              ::nn::util::Color4u8Type subColor,
                                              ::nn::util::Color4u8Type thirdColor,
                                              ::nn::util::Color4u8Type forthColor,
                                              uint32_t variation,
                                              ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager().UpdateDesignInfo(mainColor, subColor, thirdColor, forthColor, static_cast<uint8_t>(variation), id));
    NN_RESULT_SUCCESS;

}

::nn::Result HidDebugServer::GetDestinationFirmwareVersion(::nn::sf::Out<debug::FirmwareVersion> outValue,
                                                           ::nn::hid::NpadIdType id,
                                                           ::nn::hid::system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    debug::FirmwareVersion version;
    auto result =
        GetResourceManager().GetNpadResourceManager().GetDestinationFirmwareVersion(&version, id, deviceType);
    NN_RESULT_DO(ReplaceUniquePadResultToNpadResult(result));
    outValue.Set(version);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetFirmwareUpdateStage(::nn::sf::Out<::std::int64_t> outStage, ::nn::sf::Out<::std::int64_t> outProgress) NN_NOEXCEPT
{
    debug::FirmwareUpdateStage stage;
    uint8_t progress;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetFirmwareUpdateStage(&stage, &progress));

    outStage.Set(static_cast<int64_t>(stage));
    outProgress.Set(static_cast<int64_t>(progress));
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::IsFirmwareUpdatingDevice(
    ::nn::sf::Out<bool> outIsUpdating,
    ::nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    bool isUpdating;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .IsFirmwareUpdatingDevice(&isUpdating, id));

    outIsUpdating.Set(isUpdating);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::StartFirmwareUpdateIndividual(
    ::nn::sf::Out<::nn::hid::system::FirmwareUpdateDeviceHandle> outHandle,
    ::nn::hid::system::UniquePadId id,
    std::int64_t target,
    ::nn::sf::NativeHandle&& imageHandle,
    std::uint64_t imageSize) NN_NOEXCEPT
{
    XcdFirmwareUpdater::FirmwareUpdateTarget updateTarget;
    switch (target)
    {
    case ::nn::hid::debug::FirmwareUpdateTargetChip_Bluetooth:
        updateTarget = XcdFirmwareUpdater::FirmwareUpdateTarget::Bluetooth;
        break;

    case ::nn::hid::debug::FirmwareUpdateTargetChip_Mcu:
        updateTarget = XcdFirmwareUpdater::FirmwareUpdateTarget::Mcu;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    XcdFirmwareImageInfo imageInfo = {};
    imageInfo.handle          = imageHandle.GetOsHandle();
    imageInfo.isHandleManaged = imageHandle.IsManaged();
    imageInfo.size            = static_cast<size_t>(imageSize);

    NN_UTIL_SCOPE_EXIT
    {
        imageHandle.Detach();
    };

    ::nn::hid::system::FirmwareUpdateDeviceHandle deviceHandle;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .StartFirmwareUpdateByTransferMemory(&deviceHandle, id, updateTarget, imageInfo));
    outHandle.Set(deviceHandle);

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DiscardFirmwareInfoCacheForRevert() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .DiscardFirmwareInfoCacheForRevert());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetUniquePadDriverState(::nn::sf::Out<PadDriverState> outState,
                                                     const system::UniquePadId& id) NN_NOEXCEPT
{
    PadDriverState output;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetPadDriverState(&output, id));
    outState.Set(output);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetSixAxisSensorDriverStates(::nn::sf::Out<::std::int64_t> outCount,
                                                          ::nn::sf::OutArray<SixAxisSensorDriverState> outStates,
                                                          const system::UniquePadId& id) NN_NOEXCEPT
{
    int output = 0;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetSixAxisSensorDriverStates(&output, outStates.GetData(), static_cast<int>(outStates.GetLength()), id));
    outCount.Set(output);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetRxPacketHistory(::nn::sf::Out<RxPacketHistory> outValue, const system::UniquePadId& id) NN_NOEXCEPT
{
    RxPacketHistory output;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetRxPacketHistory(&output, id));
    outValue.Set(output);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::AcquireSerialFlashEventHandle(
    ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    ::nn::os::NativeHandle nativeHandle;
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .AcquireSerialFlashEventHandle(&nativeHandle, id));
    outValue.Set(::nn::sf::NativeHandle(nativeHandle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;

}

::nn::Result HidDebugServer::ReadSerialFlash(
    uint32_t address,
    ::nn::sf::NativeHandle handle,
    uint64_t size,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    auto osHandle = handle.GetOsHandle();
    auto managed = handle.IsManaged();
    handle.Detach();

    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .ReadSerialFlash(
                                         address,
                                         osHandle,
                                         managed,
                                         static_cast<int>(size),
                                         id));
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::WriteSerialFlash(
    uint32_t address,
    ::nn::sf::NativeHandle handle,
    uint64_t bufferSize,
    uint64_t writeSize,
    const system::UniquePadId& id) NN_NOEXCEPT
{
    auto osHandle = handle.GetOsHandle();
    auto managed = handle.IsManaged();
    handle.Detach();

    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .WriteSerialFlash(
                                         address,
                                         osHandle,
                                         managed,
                                         static_cast<int>(bufferSize),
                                         static_cast<int>(writeSize),
                                         id));
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetSerialFlashResult(const system::UniquePadId& id) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .GetSerialFlashResult(id));
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::ConnectUsbPadsAsync() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .ConnectUsbPadsAsync());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::DisconnectUsbPadsAsync() NN_NOEXCEPT
{
    NN_RESULT_DO(GetResourceManager().GetUniquePadResourceManager()
                                     .DisconnectUsbPadsAsync());
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::AddRegisteredDevice(nn::hid::system::RegisteredDevice device) NN_NOEXCEPT
{
    GetResourceManager().GetRegisteredDeviceTask()
                                     .AddRegisteredDevices(device);
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetAbstractedPadHandles(
    ::nn::sf::Out<int>pOutCount,
    ::nn::sf::OutArray<debug::AbstractedPadHandle> pOutHandles) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutCount.GetPointer()
        && pOutHandles.GetData()
        && pOutHandles.GetLength() <= ::nn::hid::debug::AbstractedPadHandleCountMax,
        ResultAbstractedPadInvalidArgument());

    nn::hid::detail::AbstractedPadId ids[nn::hid::detail::IAbstractedPadCountMax];
    NN_RESULT_DO(GetResourceManager().GetAbstractedPadManager()
                                     .GetActiveAbstractedPadIds(pOutCount.GetPointer(), ids, static_cast<int>(pOutHandles.GetLength())));
    for (int i = 0; i < *pOutCount; ++i)
    {
        pOutHandles[i].value = ids[i].value;
    }
    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetAbstractedPadState(
    ::nn::sf::Out<debug::AbstractedPadState> pOutState,
    ::nn::hid::debug::AbstractedPadHandle handle) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutState.GetPointer(), ResultAbstractedPadInvalidArgument());

    nn::hid::detail::IAbstractedPad* pPad;
    nn::hid::detail::AbstractedPadId id;
    id.value = handle.value;
    NN_RESULT_DO(GetResourceManager().GetAbstractedPadManager()
                                     .GetAbstractedPad(&pPad, id));
    pOutState->deviceType = pPad->GetDeviceType();

    pOutState->attributes.Reset();
    pOutState->attributes.Set<
        nn::hid::debug::AbstractedPadAttribute::IsConnected>(pPad->IsConnected());
    pOutState->attributes.Set<
        nn::hid::debug::AbstractedPadAttribute::IsSixAxisSensorEnabled>();

    pOutState->color = pPad->GetColor();
    pOutState->interfaceType = pPad->GetInterfaceType();

    pOutState->powerInfo = pPad->GetPowerInfo();

    auto padState = pPad->GetPadState();
    pOutState->buttons = padState.buttons;
    pOutState->analogStickL = padState.analogStickL;
    pOutState->analogStickR = padState.analogStickR;

    NN_RESULT_SUCCESS;
}

::nn::Result HidDebugServer::GetAbstractedPadsState(
    ::nn::sf::Out<std::int32_t> pOutCount,
    const ::nn::sf::OutArray<::nn::hid::debug::AbstractedPadHandle>& pOutHandles,
    const ::nn::sf::OutArray<::nn::hid::debug::AbstractedPadState>& pOutStates) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutCount.GetPointer()
        && pOutHandles.GetData()
        && pOutHandles.GetLength() <= ::nn::hid::debug::AbstractedPadHandleCountMax
        && pOutStates.GetData()
        && pOutStates.GetLength() <= ::nn::hid::debug::AbstractedPadHandleCountMax,
        ResultAbstractedPadInvalidArgument());

    int abstractedPadCount = 0;
    nn::hid::detail::AbstractedPadId ids[nn::hid::debug::AbstractedPadHandleCountMax] = {};
    nn::hid::detail::IAbstractedPad* pPads[nn::hid::debug::AbstractedPadHandleCountMax] = {};
    NN_RESULT_DO(GetResourceManager().GetAbstractedPadManager()
        .GetActiveAbstractedPadIds(&abstractedPadCount, ids, debug::AbstractedPadHandleCountMax));

    for (int i = 0; i < abstractedPadCount; ++i)
    {
        NN_RESULT_DO(GetResourceManager().GetAbstractedPadManager()
            .GetAbstractedPad(&pPads[i], ids[i]));
    }

    for (int i = 0; i < abstractedPadCount; ++i)
    {
        pOutHandles[i].value = ids[i].value;

        pOutStates[i].deviceType = pPads[i]->GetDeviceType();

        pOutStates[i].attributes.Reset();
        pOutStates[i].attributes.Set<
            nn::hid::debug::AbstractedPadAttribute::IsConnected>(pPads[i]->IsConnected());
        pOutStates[i].attributes.Set<
            nn::hid::debug::AbstractedPadAttribute::IsSixAxisSensorEnabled>();

        pOutStates[i].color = pPads[i]->GetColor();
        pOutStates[i].interfaceType = pPads[i]->GetInterfaceType();

        pOutStates[i].powerInfo = pPads[i]->GetPowerInfo();

        auto padState = pPads[i]->GetPadState();
        pOutStates[i].buttons = padState.buttons;
        pOutStates[i].analogStickL = padState.analogStickL;
        pOutStates[i].analogStickR = padState.analogStickR;
    }

    *pOutCount = abstractedPadCount;

    NN_RESULT_SUCCESS;
}

::nn::Result CreateHidDebugServerProxy(
    ::nn::sf::SharedPointer<IHidDebugServer>* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    *pOutValue = StaticObject<
        ::nn::sf::UnmanagedServiceObject<IHidDebugServer, HidDebugServer>
        >::Get().GetShared();
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
