﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Gesture.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>

#include "hid_AppletResourceUserId.h"
#include "hid_GestureImpl.h"
#include "hid_GestureLifo.h"
#include "hid_GestureRecognizer.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Gesture の共有メモリへのアクセスを扱うクラスです。
class GestureSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    //!< Gesture の LIFO
    ::std::atomic<GestureLifo*> m_pLifo;

public:
    GestureSharedMemoryAccessor() NN_NOEXCEPT
        : m_pLifo(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultGestureActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultGestureDeactivationLowerLimitOver());
    }

    virtual ~GestureSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< Gesture の LIFO を返します。
    const GestureLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(
            pHidServer->ActivateGesture(
                GetAppletResourceUserId(), GestureRecognizerVersion));
        m_pLifo = &::nn::util::Get(ptr->gesture.lifo);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateGesture());
        m_pLifo = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< Gesture の共有メモリアクセッサを返します。
GestureSharedMemoryAccessor& GetGestureSharedMemoryAccessor() NN_NOEXCEPT
{
    return StaticObject<GestureSharedMemoryAccessor>::Get();
}

} // namespace

::nn::Result InitializeGesture() NN_NOEXCEPT
{
    GestureSharedMemoryAccessor& accessor = GetGestureSharedMemoryAccessor();
    NN_RESULT_DO(accessor.SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(accessor.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeGesture() NN_NOEXCEPT
{
    NN_RESULT_DO(GetGestureSharedMemoryAccessor().Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetGestureStates(
    int* pOutCount, GestureState outStates[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultGestureNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultGestureInvalidStateCount());
    const GestureLifo* pLifo = GetGestureSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultGestureNotInitialized());
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    *pOutCount = pLifo->Read(outStates, count);
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
