﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_FileMappingObject-os.win.h"

namespace nn { namespace hid { namespace detail {

const char* const FileMappingObject::MutexSuffix = "_MUTEX";

FileMappingObject::FileMappingObject(DWORD count, const char* name) NN_NOEXCEPT
    : m_Mutex()
    , m_Count(count)
    , m_Handle(nullptr)
    , m_MutexHandle(nullptr)
    , m_Address(nullptr)
{
    NN_SDK_REQUIRES_EQUAL(
        static_cast<size_t>(MutexSuffixLength),
        ::strnlen(MutexSuffix, MutexSuffixLength + 1));
    const size_t length = ::strnlen(name, NameLengthMax);
    NN_SDK_ASSERT_RANGE(length, 1u, static_cast<size_t>(NameLengthMax + 1));
    ::std::strncpy(m_Name, name, length + 1);
    ::std::strncpy(m_MutexName, name, length + 1);
    ::std::strncpy(&m_MutexName[length], MutexSuffix, MutexSuffixLength + 1);
}

FileMappingObject::~FileMappingObject() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_Address != nullptr)
    {
        ::UnmapViewOfFile(m_Address);
    }

    if (m_Handle != nullptr)
    {
        ::CloseHandle(m_Handle);
    }

    if (m_MutexHandle != nullptr)
    {
        ::CloseHandle(m_MutexHandle);
    }
}

size_t FileMappingObject::GetCount() const NN_NOEXCEPT
{
    return m_Count;
}

::nn::Result FileMappingObject::Lock() NN_NOEXCEPT
{
    HANDLE mutexHandle = m_MutexHandle;

    if (mutexHandle == nullptr)
    {
        // ハンドルが未取得ならば初期化処理を実施

        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        mutexHandle = m_MutexHandle;

        if (mutexHandle == nullptr)
        {
            // 排他した状態での再チェックにも失敗した場合のみ初期化処理を実施

            mutexHandle = ::CreateMutexA(NULL, FALSE, m_MutexName);

            NN_RESULT_THROW_UNLESS(
                mutexHandle != NULL, ResultWin32ApiFailedToCreateMutex());

            m_MutexHandle = mutexHandle;
        }
    }

    ::WaitForSingleObject(mutexHandle, INFINITE);

    NN_RESULT_SUCCESS;
}

void FileMappingObject::Unlock() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_MutexHandle);

    ::ReleaseMutex(m_MutexHandle);
}

::nn::Result FileMappingObject::GetAddress(
    LPVOID* pOutAddress, bool* pOutExists) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    NN_SDK_REQUIRES_NOT_NULL(pOutExists);

    LPVOID address = m_Address;

    bool exists = true;

    if (address == nullptr)
    {
        // メモリアドレスが未取得ならば初期化処理を実施

        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        address = m_Address;

        if (address == nullptr)
        {
            // 排他した状態での再チェックにも失敗した場合のみ初期化処理を実施

            m_Handle = ::CreateFileMappingA(
                INVALID_HANDLE_VALUE, NULL, PAGE_READWRITE,
                0, static_cast<DWORD>(m_Count), m_Name);

            if (m_Handle == NULL)
            {
                m_Handle = nullptr;

                NN_RESULT_THROW(ResultWin32ApiFailedToCreateFileMapping());
            }

            exists = (::GetLastError() == ERROR_ALREADY_EXISTS);

            address = ::MapViewOfFile(m_Handle, FILE_MAP_WRITE, 0, 0, m_Count);

            if (address == 0)
            {
                ::CloseHandle(m_Handle);

                m_Handle = nullptr;

                NN_RESULT_THROW(ResultWin32ApiFailedToMapViewOfFile());
            }

            m_Address = address;
        }
    }

    *pOutAddress = address;

    *pOutExists = exists;

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
