﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/hid/hid_ExternalBus.h>

#include "hid_ExternalBusHandle.h"
#include "hid_ExternalBusImpl.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "hid_ResourceManager-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "hid_ResourceManager-os.win.h"
#endif

namespace nn { namespace hid { namespace detail {

nn::Result InitializeExternalBus(const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));

    // 今は何もしない

    NN_RESULT_SUCCESS;
}

nn::Result FinalizeExternalBus(const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));

    // 今は何もなし

    NN_RESULT_SUCCESS;
}

nn::Result IsExternalBusDeviceConnected(bool* pOutIsConnected, const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    *pOutIsConnected = pAccsessor->IsExternalConnecterDeviceConnected();
    NN_RESULT_SUCCESS;
}

nn::Result EnableExternalBusDevice(const ExternalBusHandle& handle, bool isEnabled, uint64_t version) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));

    nn::hid::NpadIdType npadType = GetExternalBusHandlePlayerNumber(handle);
    nn::hid::detail::NpadJoyConRailIndex npadJoyConRailIndex;
    NN_RESULT_DO(GetNpadJoyConRailIndexFromExternalBusHandle(&npadJoyConRailIndex, handle));
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    auto pEnableMutex = pAccsessor->GetInternalEnableMutexPointer();

    ::std::lock_guard<decltype(*pEnableMutex)> locker(*pEnableMutex);

    if (isEnabled)
    {
        // Activate
        NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().ActivateRailAttachment(npadType, npadJoyConRailIndex, true));

        NN_RESULT_DO(pAccsessor->EnableExternalConnecterDevice(isEnabled, version));
    }
    else
    {
        NN_RESULT_DO(pAccsessor->EnableExternalConnecterDevice(isEnabled, version));

        NN_RESULT_DO(GetResourceManager().GetNpadResourceManager().ActivateRailAttachment(npadType, npadJoyConRailIndex, false));
    }
    NN_RESULT_SUCCESS;
}

nn::Result GetExternalBusDeviceInfo(ExternalBusDeviceInfo* pOutInfo, const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    return pAccsessor->GetExternalConnecterDeviceInfo(pOutInfo);
}

nn::Result SendCommandAsyncToExternalBusDevice(const uint8_t* pInCommand, size_t inCommandSize, const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    return pAccsessor->SendCommandAsyncToExternalBusDevice(pInCommand, inCommandSize);
}

nn::Result GetSendCommandAsynceResultFromExternalBusDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize, const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    return pAccsessor->GetSendCommandAsynceResultFromExternalBusDevice(pOutSize, pOutCommand, outCommandSize);
}

nn::Result BindEventForExternalBusDeviceSendCommandAsycResult(nn::sf::Out<nn::sf::NativeHandle> receiveEventHandle, const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));
    // SF でもらってきたハンドルと内部に抱えるイベントを紐づける
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    auto pInternalReceiveEvent = pAccsessor->GetInternalReceiveEventPointer();
    pAccsessor->SetEventForExternalBusDeviceSendCommandAsycResult(pInternalReceiveEvent->GetBase());

    nn::os::NativeHandle internalHandle = nn::os::GetReadableHandleOfSystemEvent(pInternalReceiveEvent->GetBase());

    // 内部の IsManaged を false にしているので、寿命管理はされない。
    *receiveEventHandle = nn::sf::NativeHandle(internalHandle, false);
    NN_RESULT_SUCCESS;
}

nn::Result EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, const ExternalBusHandle& handle, ExternalBusJoyPollingMode mode) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    return pAccsessor->EnablePollingReceiveModeForAttachmentDevice(pInCommand, inCommandSize, mode);
}

nn::Result DisablePollingReceiveModeForAttachmentDevice(const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    return pAccsessor->DisablePollingReceiveModeForAttachmentDevice();
}

nn::Result GetPollingDataForAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize, const ExternalBusHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckValidExternalBusHandle(handle));
    auto pAccsessor = GetResourceManager().GetNpadResourceManager().GetExternalBusDriver(handle);
    if (pAccsessor == nullptr)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    return pAccsessor->GetPollingDataForAttachmentDevice(pOutSize, pOutCommand, outCommandSize);
}

}}} // namespace nn::hid::detail

