﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <limits>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_DebugPad.h>

#include "hid_CommonStateUtility.h"

namespace nn { namespace hid { namespace detail {

//!< Cpad のアナログスティックのカウント理論値
const int CpadAnalogStickTheoreticalCount = 106;

//!< Cpad のアナログスティックのデッドゾーン
const int CpadAnalogStickDeadZoneCount = 15;

//!< Cpad のアナログスティックの高さ（垂直方向）
const int CpadAnalogStickHorizontalCount = 72;

//!< Cpad のアナログスティックの高さ（斜め方向）
const int CpadAnalogStickDiagonalCount = 40;

//!< アナログスティックの入力変化閾値
const int32_t AnalogStickThreshold = 1;

//!< DebugPad のデジタルボタンの入力状態をマージします。
inline DebugPadButtonSet MergeDebugPadButtons(const DebugPadButtonSet& lhs,
                                              const DebugPadButtonSet& rhs
                                              ) NN_NOEXCEPT
{
    return MergeButtonsIncludingDpad(lhs, rhs);
}

//!< DebugPad のデジタルボタンに禁則処理を適応します。
inline DebugPadButtonSet RestrictDebugPadButtons(
    const DebugPadButtonSet& buttons) NN_NOEXCEPT
{
    return RestrictButtonsIncludingDpad(buttons, ControlPadRestrictionType_KeepUpAndLeft, true);
}

//!< DebugPad のアナログスティックの座標値を変域内に収めます。
inline AnalogStickState LimitDebugPadAnalogStick(
    const AnalogStickState& state,
    const int deadZoneCount,
    const int horizontalCount,
    const int diagonalCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(deadZoneCount, 0);
    NN_SDK_ASSERT_GREATER_EQUAL(horizontalCount, 0);
    NN_SDK_ASSERT_GREATER_EQUAL(diagonalCount, 0);

    // 座標の極性を記録
    const bool isXPositive = state.x >= 0;
    const bool isYPositive = state.y >= 0;

    AnalogStickState value;

    // 座標値を正の値に変換
    value.x = isXPositive ? state.x : -state.x;
    value.y = isYPositive ? state.y : -state.y;

    // 下限側を処理
    value.x = (value.x <= deadZoneCount) ? 0 : value.x - deadZoneCount;
    value.y = (value.y <= deadZoneCount) ? 0 : value.y - deadZoneCount;

    // 座標がデッドゾーン内だった場合は処理を完了
    if (value.x == 0 && value.y == 0)
    {
        return value;
    }

    // 精度保証用のスケール値
    const int scale = 7;

    // 変域の内接円の半径の 2 乗
    const auto h = static_cast<int64_t>(horizontalCount);
    const auto d = static_cast<int64_t>(diagonalCount);
    const auto g = h - d;
    const auto x = ((h * d * g) << scale) / (d * d + g * g);
    const auto y = ((h * d * d) << scale) / (d * d + g * g);
    const auto r2 = x * x + y * y;

    // 変域の内接円の半径
    const int64_t r = GetSquareRoot(r2);

    // 原点からの距離の 2 乗
    const auto s2 = static_cast<int64_t>(value.x * value.x +
                                         value.y * value.y) << (scale * 2);

    if (r2 < s2)
    {
        // 座標値が上限を超えていた場合は処理
        const int64_t s = GetSquareRoot(s2);
        value.x = static_cast<int32_t>(r * value.x / s);
        value.y = static_cast<int32_t>(r * value.y / s);
    }

    // 座標の極性を復元
    value.x = isXPositive ? value.x : -value.x;
    value.y = isYPositive ? value.y : -value.y;

    // 上下限値を復元
    value.x = static_cast<int32_t>(AnalogStickMax * value.x / (r >> scale));
    value.y = static_cast<int32_t>(AnalogStickMax * value.y / (r >> scale));

    return value;
}

template <typename T>
inline bool IsChanged(const T& state1, const T& state2, const T& threshold)
{
    T deltaState = state1 - state2;

    if (deltaState < 0)
    {
        deltaState = -deltaState;
    }

    return (deltaState > threshold);
}

//!< DebugPadState に変化があるか否かを返します
template <typename stateT>
inline bool IsDebugPadStateChanged(const stateT& state1,
                                   const stateT& state2) NN_NOEXCEPT
{
    // ディジタルボタンに入力変化があるか
    const bool IsDebugPadButtonSetChanged = (
        (state1.buttons ^ state2.buttons).CountPopulation() != 0
    );

    const bool IsLeftAnalogStickStateChanged = (
        IsChanged(state1.analogStickL.x, state2.analogStickL.x, AnalogStickThreshold) ||
        IsChanged(state1.analogStickL.y, state2.analogStickL.y, AnalogStickThreshold)
        );

    const bool IsRightAnalogStickStateChanged = (
        IsChanged(state1.analogStickR.x, state2.analogStickR.x, AnalogStickThreshold) ||
        IsChanged(state1.analogStickR.y, state2.analogStickR.y, AnalogStickThreshold)
    );

    return (
        IsDebugPadButtonSetChanged ||
        IsLeftAnalogStickStateChanged ||
        IsRightAnalogStickStateChanged
    );
}

}}} // namespace nn::hid::detail
