﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/settings/settings_GenericPadAxis.h>
#include <nn/settings/settings_GenericPadButton.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_DebugPadMappingManager.h"
#include "hid_WindowsGenericPadAccessor-os.win.h"

namespace nn { namespace hid { namespace detail {

DebugPadMappingManager::DebugPadMappingManager() NN_NOEXCEPT
    : m_Mutex(false)
{
    m_DebugPadMap.isGenericPadMapInitialized = false;
    m_DebugPadMap.genericPadMap = ::nn::settings::DebugPadGenericPadMap();
    m_DebugPadMap.keyboardMap = ::nn::settings::DebugPadKeyboardMap();
}

bool DebugPadMappingManager::IsValidDebugPadGenericPadMap(
    const ::nn::settings::DebugPadGenericPadMap& value) NN_NOEXCEPT
{
    // 汎用コントローラ ID が有効かつ有効な値が一つでも設定されていれば有効
    return IsValidWindowsGenericPadId(value.genericPadId) &&
           (value.buttonA > 0 || value.buttonB > 0 || value.buttonX > 0 ||
            value.buttonY > 0 || value.buttonL > 0 || value.buttonR > 0 ||
            value.buttonZL > 0 || value.buttonZR > 0 ||
            value.buttonStart > 0 || value.buttonSelect > 0 ||
            value.buttonLeft > 0 || value.buttonUp > 0 ||
            value.buttonRight > 0 || value.buttonDown > 0);
}

bool DebugPadMappingManager::IsValidDebugPadKeyboardMap(
    const ::nn::settings::DebugPadKeyboardMap& value) NN_NOEXCEPT
{
    // 有効な値が一つでも設定されていれば有効
    return (value.buttonA > 0 || value.buttonB > 0 || value.buttonX > 0 ||
            value.buttonY > 0 || value.buttonL > 0 || value.buttonR > 0 ||
            value.buttonZL > 0 || value.buttonZR > 0 ||
            value.buttonStart > 0 || value.buttonSelect > 0 ||
            value.buttonLeft > 0 || value.buttonUp > 0 ||
            value.buttonRight > 0 || value.buttonDown > 0);
}

::nn::Result DebugPadMappingManager::GetDebugPadGenericPadMap(
    ::nn::settings::DebugPadGenericPadMap* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultDebugPadNullGenericPadMapBuffer());

    if (!m_DebugPadMap.isGenericPadMapInitialized)
    {
        // 未初期化の場合のみ割り当ての初期化を実施

        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        if (!m_DebugPadMap.isGenericPadMapInitialized)
        {
            // 排他した状態での再チェックにも失敗した場合のみ初期化処理を実施

            ::nn::settings::DebugPadGenericPadMap map;

            GetDefaultDebugPadGenericPadMap(&map);

            NN_RESULT_DO(GetUnbusyWindowsGenericPad(&map.genericPadId));

            m_DebugPadMap.genericPadMap = map;

            m_DebugPadMap.isGenericPadMapInitialized = true;
        }
    }

    *pOutValue = m_DebugPadMap.genericPadMap;

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadMappingManager::SetDebugPadGenericPadMap(
    const ::nn::settings::DebugPadGenericPadMap& value) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    const int32_t newId = value.genericPadId;

    if (!m_DebugPadMap.isGenericPadMapInitialized)
    {
        // 未初期化の場合は汎用コントローラの割り当てを実施

        NN_RESULT_DO(AddWindowsGenericPadUser(newId));

        m_DebugPadMap.isGenericPadMapInitialized = true;
    }
    else
    {
        const int32_t oldId = m_DebugPadMap.genericPadMap.load().genericPadId;

        if (newId != oldId)
        {
            // 汎用コントローラの割り当てに変更がある場合は状態を更新

            auto needsRollback = true;

            NN_RESULT_DO(AddWindowsGenericPadUser(newId));

            NN_UTIL_SCOPE_EXIT
            {
                if (needsRollback)
                {
                    RemoveWindowsGenericPadUser(newId);
                }
            };

            NN_RESULT_DO(RemoveWindowsGenericPadUser(oldId));

            needsRollback = false;
        }
    }

    m_DebugPadMap.genericPadMap = value;

    NN_RESULT_SUCCESS;
}

void DebugPadMappingManager::ResetDebugPadGenericPadMap() NN_NOEXCEPT
{
    if (m_DebugPadMap.isGenericPadMapInitialized)
    {
        // 初期化済みだった場合のみ規定値へリセット

        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        ::nn::settings::DebugPadGenericPadMap map = m_DebugPadMap.genericPadMap;

        int genericPadId = map.genericPadId;

        GetDefaultDebugPadGenericPadMap(&map);

        map.genericPadId = genericPadId;

        m_DebugPadMap.genericPadMap = map;
    }
}

::nn::Result DebugPadMappingManager::GetDebugPadKeyboardMap(
    ::nn::settings::DebugPadKeyboardMap* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultDebugPadNullKeyboardMapBuffer());
    *pOutValue = m_DebugPadMap.keyboardMap;
    NN_RESULT_SUCCESS;
}

void DebugPadMappingManager::SetDebugPadKeyboardMap(
    const ::nn::settings::DebugPadKeyboardMap& value) NN_NOEXCEPT
{
    m_DebugPadMap.keyboardMap = value;
}

void DebugPadMappingManager::ResetDebugPadKeyboardMap() NN_NOEXCEPT
{
    m_DebugPadMap.keyboardMap = ::nn::settings::DebugPadKeyboardMap();
}

void DebugPadMappingManager::GetDefaultDebugPadGenericPadMap(
    ::nn::settings::DebugPadGenericPadMap* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    pOutValue->genericPadId = 0;
    pOutValue->buttonA = ::nn::settings::GenericPadButton_4;
    pOutValue->buttonB = ::nn::settings::GenericPadButton_3;
    pOutValue->buttonX = ::nn::settings::GenericPadButton_2;
    pOutValue->buttonY = ::nn::settings::GenericPadButton_1;
    pOutValue->buttonL = ::nn::settings::GenericPadButton_5;
    pOutValue->buttonR = ::nn::settings::GenericPadButton_6;
    pOutValue->buttonZL = ::nn::settings::GenericPadButton_7;
    pOutValue->buttonZR = ::nn::settings::GenericPadButton_8;
    pOutValue->buttonStart = ::nn::settings::GenericPadButton_10;
    pOutValue->buttonSelect = ::nn::settings::GenericPadButton_9;
    pOutValue->buttonLeft = ::nn::settings::GenericPadButton_Left;
    pOutValue->buttonUp = ::nn::settings::GenericPadButton_Up;
    pOutValue->buttonRight = ::nn::settings::GenericPadButton_Right;
    pOutValue->buttonDown = ::nn::settings::GenericPadButton_Down;
    pOutValue->axisAnalogStickRX = ::nn::settings::GenericPadAxis_Z;
    pOutValue->axisAnalogStickRY = ::nn::settings::GenericPadAxis_R;
    pOutValue->axisAnalogStickLX = ::nn::settings::GenericPadAxis_X;
    pOutValue->axisAnalogStickLY = ::nn::settings::GenericPadAxis_Y;
    pOutValue->attributes.Reset();
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>(false);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>(true);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>(false);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>(true);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>(false);
    pOutValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>(false);
}

}}} // namespace nn::hid::detail
