﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceManager.h"
#include "hid_CommonStateUtility.h"
#include "hid_DebugPadLifo.h"
#include "hid_DebugPadManager.h"
#include "hid_DebugPadStateUtility.h"
#include "hid_IDebugPadDriver.h"
#include "hid_SharedMemoryFormat.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< DebugPad の入力状態を計算します。
void CalculateDebugPadState(DebugPadState* pOutValue,
                            const DebugPadState& state,
                            bool enablesInput,
                            DebugPadButtonSet& buttonMask) NN_NOEXCEPT;

} // namespace

const ::nn::TimeSpan DebugPadManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(10);

DebugPadManager::DebugPadManager() NN_NOEXCEPT
    : m_SamplingNumber(0)
    , m_pTimer(nullptr)
    , m_pDriver(nullptr)
    , m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
    , m_State()
    , m_TempState()
    , m_TempAruid()
    , m_IsAutoPilotEnabled(false)
    , m_AutoPilotState()
{
    for (DebugPadAppletResourceEntry& extraEntry : m_AppletResourceEntries)
    {
        extraEntry = DebugPadAppletResourceEntry();
    }
}

void DebugPadManager::SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(timer);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimer = timer;
}

void DebugPadManager::SetDriver(IDebugPadDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pDriver = pDriver;
}

void DebugPadManager::SetAppletResourceManager(
    AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

void DebugPadManager::SetInputDetectorManager(
    InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pInputDetectorManager = pManager;
    m_pInputDetectorManagerMutex = pMutex;
}

::nn::Result DebugPadManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultDebugPadActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // ドライバをアクティブ化
        NN_RESULT_DO(m_pDriver->Activate());

        // LIFO を初期化
        this->ProcessSharedMemory([] (DebugPadManager* that,
                                      DebugPadSharedMemoryFormat* address,
                                      ::nn::applet::AppletResourceUserId aruid,
                                      bool enablesInput,
                                      DebugPadButtonSet& buttonMask) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(that);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            NN_UNUSED(buttonMask);
            ::nn::util::Get(address->lifo).Clear();
        });

        // タイマーイベントをアクティブ化
        m_pTimer->Enable(SamplingInterval);

        m_State = DebugPadState();

        // 入力状態を更新
        this->Update();
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadManager::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultDebugPadDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // 自動操作を無効化
        this->DisableAutoPilot();

        // タイマーイベントを非アクティブ化
        m_pTimer->Disable();

        // ドライバを非アクティブ化
        NN_RESULT_DO(m_pDriver->Deactivate());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadManager::EnsureAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_TempAruid = aruid;

    this->ProcessSharedMemory([] (DebugPadManager* that,
                                  DebugPadSharedMemoryFormat* address,
                                  ::nn::applet::AppletResourceUserId haystack,
                                  bool enablesInput,
                                  DebugPadButtonSet& buttonMask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        DebugPadState* pOutValue = &that->m_TempState;

        const DebugPadState& state = that->m_State;

        const ::nn::applet::AppletResourceUserId& needle = that->m_TempAruid;

        DebugPadLifo& lifo = ::nn::util::Get(address->lifo);

        if (lifo.IsEmpty() && needle == haystack)
        {
            CalculateDebugPadState(pOutValue, state, enablesInput, buttonMask);

            lifo.Append(*pOutValue);
        }
    });

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadManager::SetAutoPilotState(
    const ::nn::hid::debug::DebugPadAutoPilotState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultDebugPadNotInitialized());

    m_IsAutoPilotEnabled = true;

    m_AutoPilotState.buttons = RestrictDebugPadButtons(value.buttons);

    m_AutoPilotState.analogStickL = LimitAnalogStick(
        value.analogStickL, 0, AnalogStickMax, AnalogStickMax);

    m_AutoPilotState.analogStickR = LimitAnalogStick(
        value.analogStickR, 0, AnalogStickMax, AnalogStickMax);

    m_AutoPilotState.attributes.Set<DebugPadAttribute::IsConnected>(
        value.attributes.Test<DebugPadAttribute::IsConnected>() ||
        m_AutoPilotState.buttons.IsAnyOn() ||
        m_AutoPilotState.analogStickL.x != 0 ||
        m_AutoPilotState.analogStickL.y != 0 ||
        m_AutoPilotState.analogStickR.x != 0 ||
        m_AutoPilotState.analogStickR.y != 0);

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadManager::UnsetAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultDebugPadNotInitialized());
    this->DisableAutoPilot();
    NN_RESULT_SUCCESS;
}

void DebugPadManager::Sample() NN_NOEXCEPT
{
    if (!m_ActivationCount.IsZero())
    {
        this->Update();

        this->ProcessSharedMemory([] (DebugPadManager* that,
                                      DebugPadSharedMemoryFormat* address,
                                      ::nn::applet::AppletResourceUserId aruid,
                                      bool enablesInput,
                                      DebugPadButtonSet& buttonMask) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(aruid);

            DebugPadState* pOutValue = &that->m_TempState;

            const DebugPadState& state = that->m_State;

            CalculateDebugPadState(pOutValue, state, enablesInput, buttonMask);

            ::nn::util::Get(address->lifo).Append(*pOutValue);
        });
    }
}

void DebugPadManager::Update() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManagerMutex);

    // 入力変化検知用に状態を保存
    m_TempState = m_State;

    // 入力状態を取得
    m_pDriver->GetState(&m_State);

    // サンプリング番号を設定
    m_State.samplingNumber = m_SamplingNumber++;

    if (m_IsAutoPilotEnabled)
    {
        // 自動操作が有効な場合は自動操作状態をマージ

        m_State.attributes |= m_AutoPilotState.attributes;

        m_State.buttons = MergeDebugPadButtons(
            m_State.buttons, m_AutoPilotState.buttons);

        m_State.analogStickL = MergeAnalogStick(
            m_State.analogStickL, m_AutoPilotState.analogStickL);

        m_State.analogStickR = MergeAnalogStick(
            m_State.analogStickR, m_AutoPilotState.analogStickR);
    }

    ::std::lock_guard<decltype(*m_pInputDetectorManagerMutex)
                      > locker(*m_pInputDetectorManagerMutex);

    if(IsDebugPadStateChanged(m_State, m_TempState))
    {
        m_pInputDetectorManager->Notify(
            ::nn::hid::system::InputSourceId::DebugPad::Mask);
    }
}

void DebugPadManager::ProcessSharedMemory(
    void (*processor)(DebugPadManager* that,
                      DebugPadSharedMemoryFormat* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput,
                      DebugPadButtonSet& buttonMask) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)
                      > locker(*m_pAppletResourceManagerMutex);

    const AppletResourceEntry (&entries)[AppletResourceEntryCountMax] =
        m_pAppletResourceManager->GetAppletResourceEntries();

    for (size_t i = 0; i < AppletResourceEntryCountMax; ++i)
    {
        const AppletResourceEntry& entry = entries[i];

        DebugPadAppletResourceEntry& extraEntry = m_AppletResourceEntries[i];

        if (!entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            extraEntry = DebugPadAppletResourceEntry();
        }
        else
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);

            if (entry.aruid != extraEntry.aruid)
            {
                extraEntry = DebugPadAppletResourceEntry();

                extraEntry.aruid = entry.aruid;
            }

            processor(this,
                      &entry.address->debugPad,
                      entry.aruid,
                      entry.flags.Test<AppletResourceFlag::EnablesInput>(),
                      extraEntry.buttonMask);
        }
    }
}

void DebugPadManager::DisableAutoPilot() NN_NOEXCEPT
{
    m_IsAutoPilotEnabled = false;
    m_AutoPilotState = ::nn::hid::debug::DebugPadAutoPilotState();
}

namespace {

void CalculateDebugPadState(DebugPadState* pOutValue,
                            const DebugPadState& state,
                            bool enablesInput,
                            DebugPadButtonSet& buttonMask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = state;

    if (enablesInput)
    {
        buttonMask |= ~state.buttons;

        pOutValue->buttons &= buttonMask;
    }
    else
    {
        buttonMask.Reset();

        pOutValue->buttons.Reset();
        pOutValue->analogStickR = AnalogStickState();
        pOutValue->analogStickL = AnalogStickState();
    }
}

} // namespace

}}} // namespace nn::hid::detail
