﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/cpad/cpad.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>

#include "hid_DebugPadDriver-os.horizon.h"
#include "hid_DebugPadStateUtility.h"
#include "hid_Settings.h"
#include "hid_SystemButtonNotifier.h"

// ペアリングトリガーのため (削除予定)
namespace nn { namespace hid { namespace detail {

namespace {

//!< Cpad のデジタルボタンの押下状態を DebugPad のものに変換します。
DebugPadButtonSet ConvertCpadButtonSet(const ::nn::util::BitPack32& value
                                       ) NN_NOEXCEPT
{
    typedef ::nn::cpad::CpadButton CpadButton;
    DebugPadButtonSet buttons;
    buttons.Reset();
    buttons.Set<DebugPadButton::A>(value.Get<CpadButton::A>());
    buttons.Set<DebugPadButton::B>(value.Get<CpadButton::B>());
    buttons.Set<DebugPadButton::X>(value.Get<CpadButton::X>());
    buttons.Set<DebugPadButton::Y>(value.Get<CpadButton::Y>());
    buttons.Set<DebugPadButton::L>(value.Get<CpadButton::L>());
    buttons.Set<DebugPadButton::R>(value.Get<CpadButton::R>());
    buttons.Set<DebugPadButton::ZL>(value.Get<CpadButton::ZL>());
    buttons.Set<DebugPadButton::ZR>(value.Get<CpadButton::ZR>());
    buttons.Set<DebugPadButton::Start>(value.Get<CpadButton::Start>());
    buttons.Set<DebugPadButton::Select>(value.Get<CpadButton::Select>());
    buttons.Set<DebugPadButton::Left>(value.Get<CpadButton::Left>());
    buttons.Set<DebugPadButton::Up>(value.Get<CpadButton::Up>());
    buttons.Set<DebugPadButton::Right>(value.Get<CpadButton::Right>());
    buttons.Set<DebugPadButton::Down>(value.Get<CpadButton::Down>());
    return buttons;
}

//!< Cpad のアナログスティックの入力座標を DebugPad のものに変換します。
AnalogStickState ConvertCpadStickState(const ::nn::cpad::CpadStickState& value
                                       ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(::nn::cpad::CpadStickOrigin == 0);
    NN_STATIC_ASSERT(::nn::cpad::CpadStickMax == -::nn::cpad::CpadStickMin);

    AnalogStickState analogStick = { value.x, value.y };

    // DebugPad のアナログスティックの高さ（斜め方向）を計算
    const int diagonalCount =
        CpadAnalogStickDiagonalCount * ::nn::cpad::CpadStickMax /
        CpadAnalogStickHorizontalCount;

    return LimitDebugPadAnalogStick(
        analogStick, 0, ::nn::cpad::CpadStickMax, diagonalCount);
}

} // namespace

DebugPadDriver::DebugPadDriver() NN_NOEXCEPT
    : m_SamplingNumber(0)
    , m_HomePressed(false)
    , m_PairingTriggered(false)
{
    // 何もしない
}

DebugPadDriver::~DebugPadDriver() NN_NOEXCEPT
{
    // 何もしない
}

void DebugPadDriver::SetHomeButtonNotifier(SystemButtonNotifier value
                                           ) NN_NOEXCEPT
{
    m_HomeButtonNotifier = value;
}

::nn::Result DebugPadDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultDebugPadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        if (::nn::hid::detail::FirmwareDebugSettings::EnablesDebugPad())
        {
            ::nn::cpad::InitializeCpad();
        }
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultDebugPadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された場合のみ非アクティブ化を実施

        m_HomeButtonNotifier.SetSystemButtonState(false);
    }

    NN_RESULT_SUCCESS;
}

void DebugPadDriver::GetState(DebugPadState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // 無入力状態に初期化
    pOutValue->samplingNumber = m_SamplingNumber++;
    pOutValue->attributes.Reset();
    pOutValue->buttons.Reset();
    pOutValue->analogStickR = AnalogStickState();
    pOutValue->analogStickL = AnalogStickState();

    ::nn::cpad::CpadState cpadState = {};

    if (::nn::hid::detail::FirmwareDebugSettings::EnablesDebugPad())
    {
        if (::nn::cpad::GetCpadState(&cpadState).IsSuccess())
        {
            // Cpad の入力状態の取得に成功した場合は DebugPad の状態として反映

            pOutValue->attributes.Set<DebugPadAttribute::IsConnected>();
            pOutValue->buttons = ConvertCpadButtonSet(cpadState.buttons);
            pOutValue->analogStickR = ConvertCpadStickState(cpadState.stickR);
            pOutValue->analogStickL = ConvertCpadStickState(cpadState.stickL);
        }
    }

    // ペアリングトリガーのため (削除予定)
    if (cpadState.buttons.Get<nn::cpad::CpadButton::Home>())
    {
        m_HomeButtonNotifier.SetSystemButtonState(true);

        if (m_HomePressed == false)
        {
            m_HomePressed = true;
            m_TimeHomeTriggered = nn::os::GetSystemTick();
        }
        else if (m_PairingTriggered == false)
        {
            auto timeElapsed = (nn::os::GetSystemTick() - m_TimeHomeTriggered).ToTimeSpan();
            if (timeElapsed.GetMilliSeconds() > 1000)
            {
                m_pXcdDriver->StartButtonPairing();
                m_PairingTriggered = true;
            }
        }
    }
    else
    {
        m_HomeButtonNotifier.SetSystemButtonState(false);

        if (m_HomePressed == true)
        {
            m_HomePressed = false;
            m_PairingTriggered = false;
        }
    }
}

// ペアリングトリガーのため (削除予定)
void DebugPadDriver::SetXcdDriver(XcdDriver* pDriver) NN_NOEXCEPT
{
    m_pXcdDriver = pDriver;
}

}}} // namespace nn::hid::detail
