﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include <nn/xcd/xcd_Input.h>
#include <nn/hid/detail/hid_Log.h>

#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_ConsoleSixAxisSensorProcessor.h"

namespace {

::nn::xcd::SixAxisSensorState Convert(const ::nn::hid::tmp::SixAxisSensorCountState& state) NN_NOEXCEPT
{
    const float CoefAccFrom8gTo2g = 4.f;           // FSR 8G の単位系のデータを 2G の単位系に変換する係数
    const float CoefGyroFrom1000dps2000dps = 0.5f; // FSR 1000dps の単位系のデータを 2000dps の単位系に変換する係数

    return ::nn::xcd::SixAxisSensorState
    {
        state.samplingNumber,
        {{{
            static_cast<float>(state.acceleration.y  * CoefAccFrom8gTo2g),
            static_cast<float>(-state.acceleration.x * CoefAccFrom8gTo2g),
            static_cast<float>(-state.acceleration.z * CoefAccFrom8gTo2g),
        }}},
        {{{
            static_cast<float>(-state.angularVelocity.y * CoefGyroFrom1000dps2000dps),
            static_cast<float>(state.angularVelocity.x  * CoefGyroFrom1000dps2000dps),
            static_cast<float>(state.angularVelocity.z  * CoefGyroFrom1000dps2000dps),
        }}},
    };
}

} // namespace
namespace nn { namespace hid { namespace detail {

ConsoleSixAxisSensorProcessor::ConsoleSixAxisSensorProcessor() NN_NOEXCEPT
    : m_SixAxisSensorProcessor()
    , m_CalibrationValues()
    , m_LastSixAxisSensorCountState()
{
    // 何もしない
}

void ConsoleSixAxisSensorProcessor::Reset() NN_NOEXCEPT
{
    m_SixAxisSensorProcessor.Reset();
}

void ConsoleSixAxisSensorProcessor::SetSixAxisSensorSetting(const SixAxisSensorSetting* const pSetting) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSetting);

    m_SixAxisSensorProcessor.SetSixAxisSensorSetting(pSetting);
}

void ConsoleSixAxisSensorProcessor::Sampling(nn::hid::tmp::SixAxisSensorCountState& state) NN_NOEXCEPT
{
    const auto XcdState = Convert(state);

    const auto DeltaTime = ::nn::os::Tick(state.tick - m_LastSixAxisSensorCountState.tick).ToTimeSpan();

    m_SixAxisSensorProcessor.Sampling(XcdState, DeltaTime);
    m_LastSixAxisSensorCountState = state;
}

int ConsoleSixAxisSensorProcessor::GetSixAxisSensorStates(SixAxisSensorState* pOutStates, int count ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    return m_SixAxisSensorProcessor.GetSixAxisSensorStates(pOutStates, count);
}

void ConsoleSixAxisSensorProcessor::SetCalibrationValues(const tmp::ConsoleSixAxisSensorCalibrationValues& values) NN_NOEXCEPT
{
    m_CalibrationValues = values;

    ::nn::xcd::SensorCalibrationValue cal = {
        //!< 加速度センサーの原点値
        {
            values.accelerationZeroCount.x,
            values.accelerationZeroCount.y,
            values.accelerationZeroCount.z,
        },
        //!< 加速度センサーの感度
        {
            values.accelerationSensitivityCount.x,
            values.accelerationSensitivityCount.y,
            values.accelerationSensitivityCount.z,
        },
        //!< ジャイロセンサーの原点値
        {
            values.angularVelocityZeroCount.x,
            values.angularVelocityZeroCount.y,
            values.angularVelocityZeroCount.z,
        },
        //!< ジャイロセンサーの感度
        {
            values.angularVelocitySensitivityCount.x,
            values.angularVelocitySensitivityCount.y,
            values.angularVelocitySensitivityCount.z,
        }
    };

    m_SixAxisSensorProcessor.SetSensorCalibrationValue(cal);
}

const SixAxisSensorProcessor& ConsoleSixAxisSensorProcessor::GetSixAxisSensorProcessor() const NN_NOEXCEPT
{
    return m_SixAxisSensorProcessor;
}

//!< 削除予定: 開発用途でキャリブレーション値を取得します
void ConsoleSixAxisSensorProcessor::GetCalibrationValues(tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) const NN_NOEXCEPT
{
    *pOutValues = m_CalibrationValues;
}

}}} // namespace nn::hid::detail
