﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>
#include <nn/os/os_MemoryHeapCommon.h>

#include <nnd/lsm6ds3/lsm6ds3.h>

#include "hid_ActivationCount.h"
#include "hid_IConsoleSixAxisSensorDriver.h"

namespace nn { namespace hid { namespace detail {

//!< ドライバの設定値
struct DriverConfigurations
{
    ::nn::TimeSpan samplingInterval;
    nnd::lsm6ds3::AccelerometerFsr accelerometerFsr;
    nnd::lsm6ds3::AccelerometerOdr accelerometerOdr;
    nnd::lsm6ds3::GyroscopeFsr gyroscopeFsr;
    nnd::lsm6ds3::GyroscopeOdr gyroscopeOdr;
    bool isFilterEnabled;
};

struct DriverCalibrationValues
{
    ::nn::util::Float3 accelerometerCalibrationOffset;
    ::nn::util::Float3 accelerometerCalibrationScale;
    ::nn::util::Float3 angularVelocityCalibrationOffset;
    ::nn::util::Float3 angularVelocityCalibrationScale;
};

class ConsoleSixAxisSensorDriver final : public IConsoleSixAxisSensorDriver
{
    NN_DISALLOW_COPY(ConsoleSixAxisSensorDriver);
    NN_DISALLOW_MOVE(ConsoleSixAxisSensorDriver);

private:
    //!< このインスタンスからドライバがアクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< デバイスが利用可能か否か
    bool m_IsDeviceAvailable;

    //!< 起床状態にあるか否か
    bool m_IsAwake;

    //!< サンプリング開始状態にあるか否か
    bool m_IsSamplingStarted;

    //!< サンプリング番号
    int64_t m_SamplingNumber;

    //!< タイムスタンプ
    ::nn::TimeSpan m_LastTime;

    //!< ドライバの設定値
    DriverConfigurations m_DriverConfigurations;

    //!< キャリブレーション値
    DriverCalibrationValues m_DriverCalibrationValues;

    /**
    * @brief       定期取得用のバッファです。
    *
    * @details     定期取得用のバッファです。nn::os::MemoryPageSize の整数倍である必要があります。
    */
    NN_ALIGNAS(nn::os::MemoryPageSize) char m_IntervalReceiveBuffer[nn::os::MemoryPageSize];

public:
    ConsoleSixAxisSensorDriver() NN_NOEXCEPT;

    virtual ~ConsoleSixAxisSensorDriver() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Activate() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Wake() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Sleep() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Start(::nn::TimeSpan samplingInterval) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Stop() NN_NOEXCEPT NN_OVERRIDE;

    virtual int GetStates(tmp::SixAxisSensorCountState* pOutValue, int count) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result GetConsoleSixAxisSensorCalibrationValues(tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) NN_NOEXCEPT NN_OVERRIDE;

    virtual void ConvertAcceleration(nn::util::Float3* outValue,
                                     const tmp::SensorCountState& acceleration) const NN_NOEXCEPT NN_OVERRIDE;

    virtual void ConvertAngularVelocity(nn::util::Float3* outValue,
                                        const tmp::SensorCountState& angularVelocity) const NN_NOEXCEPT NN_OVERRIDE;

private:

    void UpdateCalibrationValues(const tmp::ConsoleSixAxisSensorCalibrationValues& values) NN_NOEXCEPT;

    ::nn::Result ActivateDevice() NN_NOEXCEPT;

    void ConvertAxisAcceleration(nn::util::Float3* outValue,
                                 const nn::util::Float3& acceleration) const NN_NOEXCEPT;

    void ConvertAxisAngularVelocity(nn::util::Float3* outValue,
                                    const nn::util::Float3& angularVelocity) const NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
