﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fs.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/detail/hid_Log.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_MotionSensor.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nnd/lsm6ds3/lsm6ds3.h>

#include "hid_ConsoleSixAxisSensorDriver.horizon.lsm6ds3.h"
#include "hid_RingLifo.h"

//!< システム設定の情報ログを出力します。
#define NN_HID_ConsoleSixAxisSensor_INFO(...) \
    NN_DETAIL_HID_INFO("[hid::ConsoleSixAxisSensor] Information: " __VA_ARGS__)

//!< システム設定の警告ログを出力します。
#define NN_HID_ConsoleSixAxisSensor_WARN(...) \
    NN_DETAIL_HID_WARN("[hid::ConsoleSixAxisSensor] Warning: " __VA_ARGS__)

namespace nn { namespace hid { namespace detail {

namespace {

const DriverConfigurations DefaultDriverConfigurations
{
    ::nn::TimeSpan::FromMilliSeconds(1),
    nnd::lsm6ds3::AccelerometerFsr_8G,
    nnd::lsm6ds3::AccelerometerOdr_833Hz,
    nnd::lsm6ds3::GyroscopeFsr_1000dps,
    nnd::lsm6ds3::GyroscopeOdr_833Hz,
    true,
};

const tmp::ConsoleSixAxisSensorCalibrationValues DefaultCalibrationValues = {
    { 0, 0, 0 },
    { 16384, 16384, 16384 },
    { 0, 0, 0 },
    { 13371, 13371, 13371 },
};

// 最大入力値のテーブル(G)
static const float AccelerometerFsrTable[] =
{
    2.0f,
    4.0f,
    8.0f,
    16.0f
};

// 最大入力値のテーブル(deg/s)
static const float GyroscopeFsrTable[] =
{
    125.0f,
    245.0f,
    500.0f,
    1000.0f,
    2000.0f
};

// 加速度センサキャリブレーション基準値(G)
static const float AccelerometerCalibrationBase = 1.f;

// 角速度センサキャリブレーション基準値(deg/s)
static const float GyroscopeCalibrationBase = 936.f;

} // namespace

ConsoleSixAxisSensorDriver::ConsoleSixAxisSensorDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_IsDeviceAvailable(false)
    , m_IsAwake(false)
    , m_IsSamplingStarted(false)
    , m_SamplingNumber(0)
    , m_LastTime()
    , m_DriverConfigurations(DefaultDriverConfigurations)
{
    UpdateCalibrationValues(DefaultCalibrationValues);
}

ConsoleSixAxisSensorDriver::~ConsoleSixAxisSensorDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result ConsoleSixAxisSensorDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultConsoleSixAxisSensorDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        // ライブラリを初期化
        auto result = ::nnd::lsm6ds3::Initialize();

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                ::nnd::lsm6ds3::Finalize();
            }
        };

        if (result.IsSuccess() && ActivateDevice().IsSuccess())
        {
            m_IsDeviceAvailable = true;
            m_IsAwake = true;

            needsRollback = false;
        }
    }

    // このインスタンスからアクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultConsoleSixAxisSensorDriverDeactivationLowerLimitOver());

    // このインスタンスからアクティブ化した回数をデクリメント
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
        ::nnd::lsm6ds3::StopPeriodicReceiveMode();
        ::nnd::lsm6ds3::Finalize();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorDriver::Wake() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    m_IsAwake = true;

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorDriver::Sleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    m_IsAwake = false;

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorDriver::Start(::nn::TimeSpan samplingInterval) NN_NOEXCEPT
{
    if (m_IsDeviceAvailable == false)
    {
        NN_RESULT_SUCCESS;
    }

    if (m_IsSamplingStarted == true)
    {
        // 既にサンプリングを開始済みなので何もしない
        NN_RESULT_SUCCESS;
    }

    // サンプリング間隔を設定
    m_DriverConfigurations.samplingInterval = samplingInterval;

    // 定期サンプリングを開始
    ::nnd::lsm6ds3::StartPeriodicReceiveMode(m_DriverConfigurations.samplingInterval,
                                             m_IntervalReceiveBuffer,
                                             sizeof(m_IntervalReceiveBuffer));

    NN_DETAIL_HID_TRACE("%s: PeriodicReceiveMode started. Interval = %d[ms]\n"
        , NN_CURRENT_FUNCTION_NAME
        , m_DriverConfigurations.samplingInterval.GetMilliSeconds());

    m_IsSamplingStarted = true;
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorDriver::Stop() NN_NOEXCEPT
{
    if (m_IsDeviceAvailable == false)
    {
        NN_RESULT_SUCCESS;
    }

    if (m_IsSamplingStarted == false)
    {
        // 既にサンプリングを停止済みなので何もしない
        NN_RESULT_SUCCESS;
    }

    // 定期サンプリングを停止
    ::nnd::lsm6ds3::StopPeriodicReceiveMode();
    NN_DETAIL_HID_TRACE("%s: PeriodicReceiveMode stopped.\n"
        , NN_CURRENT_FUNCTION_NAME);

    m_IsSamplingStarted = false;
    NN_RESULT_SUCCESS;
}

int ConsoleSixAxisSensorDriver::GetStates(tmp::SixAxisSensorCountState* pOutValue, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_UNUSED(count); // TORIAEZU
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    if (m_IsDeviceAvailable == false)
    {
        return 0;
    }

    // サンプリング開始前は何もデータを返さない
    if (m_IsSamplingStarted == false)
    {
        return 0;
    }

    ::nnd::lsm6ds3::SixAxisSensorCountState states[10] = {};

    int validCount = ::nnd::lsm6ds3::GetStatesForPeriodicReceiveMode(states, 10);

    for (int i = 0; i < validCount; i++)
    {
        pOutValue[i].acceleration.x = states[i].acceleration.x;
        pOutValue[i].acceleration.y = states[i].acceleration.y;
        pOutValue[i].acceleration.z = states[i].acceleration.z;
        pOutValue[i].angularVelocity.x = states[i].angularVelocity.x;
        pOutValue[i].angularVelocity.y = states[i].angularVelocity.y;
        pOutValue[i].angularVelocity.z = states[i].angularVelocity.z;
        pOutValue[i].samplingNumber = states[i].samplingNumber;
        pOutValue[i].tick = states[i].tick;
    }

    return validCount;
}

::nn::Result ConsoleSixAxisSensorDriver::GetConsoleSixAxisSensorCalibrationValues(tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) NN_NOEXCEPT
{
    nn::settings::factory::AccelerometerOffset accOffset;
    nn::settings::factory::GyroscopeOffset gyroOffset;

    NN_RESULT_DO(::nn::settings::factory::GetAccelerometerOffset(&accOffset));
    NN_RESULT_DO(::nn::settings::factory::GetGyroscopeOffset(&gyroOffset));

    pOutValues->accelerationZeroCount.x = accOffset.x;
    pOutValues->accelerationZeroCount.y = accOffset.y;
    pOutValues->accelerationZeroCount.z = accOffset.z;

    pOutValues->angularVelocityZeroCount.x = gyroOffset.x;
    pOutValues->angularVelocityZeroCount.y = gyroOffset.y;
    pOutValues->angularVelocityZeroCount.z = gyroOffset.z;

    // (SPRD-4043) NAND に書き込まれた加速度感度, 角速度感度が仕様と異なる問題があるため、
    //             固定値を設定するワークアラウンドを適用しています。

    pOutValues->accelerationSensitivityCount = DefaultCalibrationValues.accelerationSensitivityCount;
    pOutValues->angularVelocitySensitivityCount = DefaultCalibrationValues.angularVelocitySensitivityCount;

    NN_RESULT_SUCCESS;
}

void ConsoleSixAxisSensorDriver::UpdateCalibrationValues(const tmp::ConsoleSixAxisSensorCalibrationValues& values) NN_NOEXCEPT
{
    // 加速度
    {
        const float CalFsrScale =
            AccelerometerFsrTable[::nnd::lsm6ds3::AccelerometerFsr_2G]
            / AccelerometerFsrTable[m_DriverConfigurations.accelerometerFsr];

        const ::nn::util::Float3 AccelerometerOriginCount =
        {{{
            static_cast<float>(values.accelerationZeroCount.x) * CalFsrScale,
            static_cast<float>(values.accelerationZeroCount.y) * CalFsrScale,
            static_cast<float>(values.accelerationZeroCount.z) * CalFsrScale,
        }}};

        const ::nn::util::Float3 AccelerometerSensitivityCount =
        {{{
            static_cast<float>(values.accelerationSensitivityCount.x) * CalFsrScale,
            static_cast<float>(values.accelerationSensitivityCount.y) * CalFsrScale,
            static_cast<float>(values.accelerationSensitivityCount.z) * CalFsrScale,
        }}};

        for (int i = 0; i < 3; i++)
        {
            // 補正スケールを求める
            m_DriverCalibrationValues.accelerometerCalibrationScale.v[i] =
                AccelerometerCalibrationBase /
                (AccelerometerSensitivityCount.v[i] - AccelerometerOriginCount.v[i]);

            // 補正オフセットを求める
            m_DriverCalibrationValues.accelerometerCalibrationOffset.v[i] = AccelerometerOriginCount.v[i];
        }
    }

    // 角速度
    {
        const float CalFsrScale =
            GyroscopeFsrTable[::nnd::lsm6ds3::GyroscopeFsr_2000dps]
            / GyroscopeFsrTable[m_DriverConfigurations.gyroscopeFsr];

        const ::nn::util::Float3 AngularVelcityOriginCount =
        {{{
            static_cast<float>(values.angularVelocityZeroCount.x) * CalFsrScale,
            static_cast<float>(values.angularVelocityZeroCount.y) * CalFsrScale,
            static_cast<float>(values.angularVelocityZeroCount.z) * CalFsrScale,
        }}};

        const ::nn::util::Float3 AngularVelcitySensitivityCount =
        {{{
            static_cast<float>(values.angularVelocitySensitivityCount.x) * CalFsrScale,
            static_cast<float>(values.angularVelocitySensitivityCount.y) * CalFsrScale,
            static_cast<float>(values.angularVelocitySensitivityCount.z) * CalFsrScale,
        }}};

        const float ConvertScale = 1.f / 360.f; // 360 deg/s を 1.0 とする変換

        for (int i = 0; i < 3; i++)
        {
            // 補正スケールを求める
            m_DriverCalibrationValues.angularVelocityCalibrationScale.v[i] =
                GyroscopeCalibrationBase
                / (AngularVelcitySensitivityCount.v[i] - AngularVelcityOriginCount.v[i]) * ConvertScale;

            // 補正オフセットを求める
            m_DriverCalibrationValues.angularVelocityCalibrationOffset.v[i] = AngularVelcityOriginCount.v[i];
        }
    }
}

::nn::Result ConsoleSixAxisSensorDriver::ActivateDevice() NN_NOEXCEPT
{
    {
        tmp::ConsoleSixAxisSensorCalibrationValues values;

        // キャリブレーション設定の読み出し
        NN_RESULT_DO(GetConsoleSixAxisSensorCalibrationValues(&values));

        UpdateCalibrationValues(values);
    }

    // 各種設定
    m_DriverConfigurations = DefaultDriverConfigurations;

    // センサーの電源を ON
    nnd::lsm6ds3::StartAccelerometer();
    nnd::lsm6ds3::StartGyroscope();

    // 設定値をデバイスに反映
    nnd::lsm6ds3::SetAccelerometerFsr(m_DriverConfigurations.accelerometerFsr);
    nnd::lsm6ds3::SetAccelerometerOdr(m_DriverConfigurations.accelerometerOdr);
    nnd::lsm6ds3::SetGyroscopeFsr(m_DriverConfigurations.gyroscopeFsr);
    nnd::lsm6ds3::SetGyroscopeOdr(m_DriverConfigurations.gyroscopeOdr);
    nnd::lsm6ds3::SetNoiseReductionFilter(m_DriverConfigurations.isFilterEnabled);

    NN_RESULT_SUCCESS;
}

void ConsoleSixAxisSensorDriver::ConvertAcceleration(nn::util::Float3* outValue,
                                                     const tmp::SensorCountState& acceleration) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    ::nn::util::Float3 state3f = {{{
        static_cast<float>(acceleration.x),
        static_cast<float>(acceleration.y),
        static_cast<float>(acceleration.z)
    }}};

    ::nn::util::Float3 accelerationFloat3;
    for (int i = 0; i < 3; i++)
    {
        accelerationFloat3.v[i] =
            (state3f.v[i] - m_DriverCalibrationValues.accelerometerCalibrationOffset.v[i])
                * m_DriverCalibrationValues.accelerometerCalibrationScale.v[i];
    }

    // スケール変化後に軸変換
    ConvertAxisAcceleration(outValue, accelerationFloat3);
}

void ConsoleSixAxisSensorDriver::ConvertAngularVelocity(nn::util::Float3* outValue,
                                                        const tmp::SensorCountState& angularVelocity) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    ::nn::util::Float3 state3f = {{{
        static_cast<float>(angularVelocity.x),
        static_cast<float>(angularVelocity.y),
        static_cast<float>(angularVelocity.z)
    }}};

    ::nn::util::Float3 angularVelocityFloat3;
    for (int i = 0; i < 3; i++)
    {
        angularVelocityFloat3.v[i] =
            (state3f.v[i] - m_DriverCalibrationValues.angularVelocityCalibrationOffset.v[i])
                * m_DriverCalibrationValues.angularVelocityCalibrationScale.v[i];
    }

    // スケール変化後に軸変換
    ConvertAxisAngularVelocity(outValue, angularVelocityFloat3);
}

void ConsoleSixAxisSensorDriver::ConvertAxisAcceleration(nn::util::Float3* outValue,
                                                         const nn::util::Float3& acceleration) const NN_NOEXCEPT
{
    outValue->x = acceleration.y;
    outValue->y = -acceleration.x;
    outValue->z = -acceleration.z;
}

void ConsoleSixAxisSensorDriver::ConvertAxisAngularVelocity(nn::util::Float3* outValue,
                                                            const nn::util::Float3& angularVelocity) const NN_NOEXCEPT
{
    outValue->x = -angularVelocity.y;
    outValue->y = angularVelocity.x;
    outValue->z = angularVelocity.z;
}

}}} // namespace nn::hid::detail
