﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TimerEvent.h>

#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_AudioControlLifo.h"
#include "hid_IAudioControlDriver.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SynchronizedTimer.h"

namespace nn { namespace hid { namespace detail {

//!< AudioControl インターフェイスの解決を担うマネージャを扱うクラスです。
class AudioControlManager final
{
    NN_DISALLOW_COPY(AudioControlManager);
    NN_DISALLOW_MOVE(AudioControlManager);

public:
    //!< サンプリング間隔
    static const ::nn::TimeSpan SamplingInterval;

private:
    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< タイマーイベント
    SynchronizedTimer* m_pTimer;

    //!< ドライバ
    IAudioControlDriver* m_pDriver;

    //!< 入力通知が必要か否か
    ::std::atomic<bool> m_NeedsToSignal;

    //!< 入力状態
    system::AudioControlState m_State;

    //!< 前回の入力状態
    system::AudioControlState m_RecentState;

    //!< 入力状態が変化した時トリガーされるイベント
    nn::os::SystemEventType m_AudioControlEvent;

    AudioControlLifo m_Lifo;

public:
    AudioControlManager() NN_NOEXCEPT;

    //!< タイマーイベントを設定します。
    void SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT;

    //!< ドライバを設定します。
    void SetDriver(IAudioControlDriver* pDriver) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< 入力状態を解決します。
    void Sample() NN_NOEXCEPT;

    //!< オーディオコントロールの入力通知をシグナル状態にします。
    void SignalAudioControlEvent() NN_NOEXCEPT;

    //!< 入力状態変化をバインドします。
    ::nn::Result AcquireAudioControlEventHandle(
        ::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT;

    ::nn::Result GetStates(int* pOutCount,
                           system::AudioControlState outStates[],
                           int count) NN_NOEXCEPT;

private:
    //!< 入力状態を更新します。
    void Update() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail

